package com.agilex.healthcare.mobilehealthplatfrom.domain.filter;

import static org.junit.Assert.assertEquals;

import java.util.Date;

import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.domain.Appointment;
import com.agilex.healthcare.mobilehealthplatform.domain.Appointments;
import com.agilex.healthcare.mobilehealthplatform.domain.Dateable;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilter;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilterFactory;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilterer;
import com.agilex.healthcare.utility.DateHelper;

public class DateFiltererTest {

	@Test
	public void meetsBeginFilter() {
		runBeginFilterTest("1/1/2012", "1/1/2000", true);
	}

	@Test
	public void meetsBeginFilterWithTime() {
		runBeginFilterTestWithTime("1/1/2012 11:00:01", "1/1/2012 11:00:00", true);
	}

	@Test
	public void equalsBeginFilter() {
		runBeginFilterTest("1/1/2012", "1/1/2012", true);
	}

	@Test
	public void doesNotMeetBeginFilter() {
		runBeginFilterTest("1/1/2012", "1/1/2013", false);
	}

	@Test
	public void nullFilterDateOnBeginFilter() {
		runBeginFilterTest("1/1/2012", null, true);
	}

	@Test
	public void nullCompareDateOnBeginFilter() {
		runBeginFilterTest(null, "1/1/2012", false);
	}

	@Test
	public void nullDatesOnBeginFilter() {
		runBeginFilterTest(null, null, true);
	}

	@Test
	public void doesNotMeetBeginFilterWithTime() {
		runBeginFilterTestWithTime("1/1/2012 11:00:00", "1/1/2012 11:01:01", false);
	}

	private void runBeginFilterTest(String dateToCheck, String beginFilterDate, boolean expected) {
		boolean actual = DateFilterer.meetsBeginDateFilter(DateHelper.parseDate(dateToCheck), DateHelper.parseDate(beginFilterDate));
		assertEquals(expected, actual);
	}

	private void runBeginFilterTestWithTime(String dateToCheck, String beginFilterDate, boolean expected) {
		boolean actual = DateFilterer.meetsBeginDateFilter(DateHelper.parseDateTime(dateToCheck), DateHelper.parseDateTime(beginFilterDate));
		assertEquals(expected, actual);
	}

	@Test
	public void meetsEndFilter() {
		runEndFilterTest("1/1/2012", "1/1/2013", true);
	}

	@Test
	public void meetsEndFilterWithTime() {
		runEndFilterTestWithTime("1/1/2012 11:00:01", "1/1/2012 11:05:00", true);
	}

	@Test
	public void equalsEndFilter() {
		runEndFilterTest("1/1/2012", "1/1/2012", true);
	}

	@Test
	public void doesNotMeetEndFilter() {
		runEndFilterTest("1/1/2012", "1/1/2011", false);
	}

	@Test
	public void doesNotMeetEndFilterWithTime() {
		runEndFilterTestWithTime("1/1/2012 11:02:00", "1/1/2012 11:01:01", false);
	}

	@Test
	public void nullFilterDateOnEndFilter() {
		runEndFilterTest("1/1/2012", null, true);
	}

	@Test
	public void nullCompareDateOnEndFilter() {
		runEndFilterTest(null, "1/1/2012", false);
	}

	@Test
	public void nullDatesOnEndFilter() {
		runEndFilterTest(null, null, true);
	}

	private void runEndFilterTest(String dateToCheck, String EndFilterDate, boolean expected) {
		boolean actual = DateFilterer.meetsEndDateFilter(DateHelper.parseDate(dateToCheck), DateHelper.parseDate(EndFilterDate));
		assertEquals(expected, actual);
	}

	private void runEndFilterTestWithTime(String dateToCheck, String EndFilterDate, boolean expected) {
		boolean actual = DateFilterer.meetsEndDateFilter(DateHelper.parseDateTime(dateToCheck), DateHelper.parseDateTime(EndFilterDate));
		assertEquals(expected, actual);
	}

	@Test
	public void meetsSingleDateFilter() {
		runFilterTest("6/1/2012 11:00:00", "6/1/2012", true);
	}

	@Test
	public void beforeSingleDateFilter() {
		runFilterTest("6/1/2012 11:00:00", "6/2/2012", false);
	}

	@Test
	public void afterSingleDateFilter() {
		runFilterTest("6/1/2012 11:00:00", "5/2/2012", false);
	}

	@Test
	public void meetsSingleDateFilterMidnight() {
		runFilterTest("6/1/2012 00:00:00", "6/1/2012", true);
	}

	@Test
	public void nullFilterDateOnSingleDateFilter() {
		runFilterTest("6/1/2012 00:00:00", null, true);
	}

	@Test
	public void nullDatesOnSingleDateFilter() {
		runFilterTest(null, null, true);
	}

	@Test
	public void nullCompareDateOnSingleDateFilter() {
		runFilterTest(null, "5/1/1900", false);
	}

	private void runFilterTest(String dateToCheck, String filterDate, boolean expected) {
		Dateable item = createAppointment(null, dateToCheck);
		boolean actual = DateFilterer.meetsDateFilter(item, DateFilterFactory.createFilterFromDate(filterDate));
		assertEquals(expected, actual);
	}

	@Test
	public void meetsBetweenFilter() {
		runFilterTest("6/1/2012 11:00:00", "6/1/2012 10:00:00", "6/1/2012 12:00:00", true);
	}

	@Test
	public void afterBetweenFilter() {
		runFilterTest("6/1/2012 11:00:00", "6/1/2012 10:00:00", "6/1/2012 10:30:00", false);
	}

	@Test
	public void equalsBeforeBetweenFilter() {
		runFilterTest("6/1/2012 11:00:00", "6/1/2012 11:00:00", "6/1/2012 11:30:00", true);
	}

	@Test
	public void equalsAfterBetweenFilter() {
		runFilterTest("6/1/2012 11:00:00", "6/1/2012 10:00:00", "6/1/2012 11:00:00", true);
	}

	@Test
	public void beforeBetweenFilter() {
		runFilterTest("6/1/2012 11:00:00", "6/1/2012 11:00:01", "6/1/2012 12:00:00", false);
	}

	@Test
	public void nullDateOnBetweenFilter() {
		runFilterTest(null, "6/1/2012 11:00:01", "6/1/2012 12:00:00", false);
	}

	@Test
	public void nullBeforeDateOnBetweenFilter() {
		runFilterTest("6/1/2012 11:00:00", null, "6/1/2012 12:00:00", true);
	}

	@Test
	public void nullAfterDateOnBetweenFilter() {
		runFilterTest("6/1/2012 11:00:00", "6/1/2012 09:00:00", null, true);
	}

	@Test
	public void nullFilterDateOnBetweenFilter() {
		runFilterTest("6/1/2012 11:00:00", null, null, true);
	}

	@Test
	public void nullBeforeDateOnBetweenNoMatchFilter() {
		runFilterTest("6/1/2012 11:00:00", null, "6/1/2012 10:00:00", false);
	}

	@Test
	public void nullAfterDateOnBetweenNoMatchFilter() {
		runFilterTest("6/1/2012 11:00:00", "6/1/2012 12:00:00", null, false);
	}

	private void runFilterTest(String dateToCheck, String beginFilterDate, String endFilterDate, boolean expected) {
		boolean actual = DateFilterer.meetsDateFilter(DateHelper.parseDateTime(dateToCheck), DateHelper.parseDateTime(beginFilterDate), DateHelper.parseDateTime(endFilterDate));
		assertEquals(expected, actual);
	}

	@Test
	public void filterItemBySingleDate() {
		Dateable item = createAppointment(null, "1/1/2010");
		boolean actual = DateFilterer.meetsDateFilter(item, DateFilterFactory.createFilterFromDate("1/1/2010"));
		assertEquals(true, actual);
	}

	@Test
	public void filterItemByDateRange() {
		Dateable item = createAppointment(null, "1/1/2010");
		boolean actual = DateFilterer.meetsDateFilter(item, DateFilterFactory.createFilterFromDate("1/1/2009", "1/1/2011"));
		assertEquals(true, actual);
	}

	@Test
	public void filterItemBySingleDateNoMatch() {
		Dateable item = createAppointment(null, "1/1/2010");
		boolean actual = DateFilterer.meetsDateFilter(item, DateFilterFactory.createFilterFromDate("1/2/2010"));
		assertEquals(false, actual);
	}

	private Appointment createAppointment(String id, String dt) {
		Appointment appointment = new Appointment();
		appointment.setAppointmentStartDate(DateHelper.parseDate(dt));
		appointment.setUniqueId(id);
		return appointment;
	}

	@Test
	public void filterCollectionBySingleDate() {
		Appointments appointments = createTestAppointmentCollection();
		appointments = DateFilterer.filterByDate(appointments, DateFilterFactory.createFilterFromDate("1/2/2000"));
		assertEquals(1, appointments.size());
		assertEquals("b", appointments.get(0).getUniqueId());
	}

	@Test
	public void filterCollectionByDateRange() {
		Appointments appointments = createTestAppointmentCollection();
		appointments = DateFilterer.filterByDate(appointments, DateFilterFactory.createFilterFromDate("1/2/2000", "1/3/2000"));
		assertEquals(2, appointments.size());
		assertEquals("b", appointments.get(0).getUniqueId());
		assertEquals("c", appointments.get(1).getUniqueId());
	}

	@Test
	public void filterCollectionLeavesOriginal() {
		Appointments allAppointments = createTestAppointmentCollection();
		assertEquals(5, allAppointments.getSize());

		Appointments appointments = DateFilterer.filterByDate(allAppointments, DateFilterFactory.createFilterFromDate("1/2/2000", "1/3/2000"));
		assertEquals(2, appointments.size());

		appointments = DateFilterer.filterByDate(allAppointments, DateFilterFactory.createEmptyFilter());
		assertEquals(5, appointments.getSize());

		// Class.forName(appointments.getClass().getName()).newInstance();
	}

	@Test
	public void filterCollectionByNull() {
		Appointments appointments = createTestAppointmentCollection();
		DateFilter filter = null;
		appointments = DateFilterer.filterByDate(appointments, filter);
		assertEquals(5, appointments.size());
	}

	@Test
	public void filterCollectionByNullFilter() {
		Appointments appointments = createTestAppointmentCollection();
		Date dt = null;
		appointments = DateFilterer.filterByDate(appointments, DateFilterFactory.createFilterFromDate(dt));
		assertEquals(5, appointments.size());
	}

	private Appointments createTestAppointmentCollection() {
		Appointments items = new Appointments();
		items.add(createAppointment("a", "1/1/2000"));
		items.add(createAppointment("b", "1/2/2000"));
		items.add(createAppointment("c", "1/3/2000"));
		items.add(createAppointment("d", "1/4/2000"));
		items.add(createAppointment("e", "1/5/2000"));
		return items;
	}

}
