package com.agilex.healthcare.mobilehealthplatform.validator.vitals;

import static org.junit.Assert.assertEquals;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.UUID;

import org.junit.Before;
import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.domain.ValidationError;
import com.agilex.healthcare.mobilehealthplatform.domain.ValidationResult;
import com.agilex.healthcare.mobilehealthplatform.domain.VitalEntry;
import com.agilex.healthcare.mobilehealthplatform.domain.VitalObservation;
import com.agilex.healthcare.mobilehealthplatform.domain.code.ObservationTypeCode;
import com.agilex.healthcare.utility.DateHelper;

public class VitalsTemperatureValidatorTest {

	private VitalTemperatureValidator validator;

	@Before
	public void setup() {
		validator = new VitalTemperatureValidator();
	}

	@Test
	public void validateInRangeData() {
		VitalEntry vitalEntry = createTempatureVitalEntry("98.6", "");
		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		assertEquals(true, result.isValid());
	}

	@Test
	public void validateErrorContainsActualVitalEntry() {
		VitalEntry vitalEntry = createTempatureVitalEntry("98.6", "");
		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		assertEquals(vitalEntry, result.getRequestObject());
	}

	@Test
	public void validateNullEntryReturnsError() {
		VitalEntry vitalEntry = null;
		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		assertEquals(false, result.isValid());
	}

	@Test
	public void validateLowTemperateEntryReturnsError() {
		VitalEntry vitalEntry = createTempatureVitalEntry("94", "");
		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		assertEquals(false, result.isValid());
		assertError(result.getErrors().get(0), VitalTemperatureValidator.TEMPERATURE_FIELD, "Temperature field must be between 95.0 and 106.0");
	}

	@Test
	public void validateHighTemperateEntryReturnsError() {
		VitalEntry vitalEntry = createTempatureVitalEntry("107", "");
		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		assertEquals(false, result.isValid());
		assertError(result.getErrors().get(0), VitalTemperatureValidator.TEMPERATURE_FIELD, "Temperature field must be between 95.0 and 106.0");
	}

	@Test
	public void validateEmptyTemperateEntryReturnsError() {
		VitalEntry vitalEntry = createTempatureVitalEntry("", "");
		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		assertEquals(false, result.isValid());
		assertError(result.getErrors().get(0), VitalTemperatureValidator.TEMPERATURE_FIELD, "Temperature field is required.");
	}

	@Test
	public void validateNullTemperateEntryReturnsError() {
		VitalEntry vitalEntry = createTempatureVitalEntry(null, "");
		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		assertEquals(false, result.isValid());

		assertError(result.getErrors().get(0), VitalTemperatureValidator.TEMPERATURE_FIELD, "Temperature field is required.");
	}

	@Test
	public void validateNullReturnsError() {
		ValidationResult<VitalEntry> result = validator.validate(null);
		assertEquals(false, result.isValid());

		assertError(result.getErrors().get(0), VitalTemperatureValidator.VitalType, "Temperature vital entry object is required.");
	}

	@Test
	public void validateLengthOfNotes() {
		VitalEntry vitalEntry = createTempatureVitalEntry("98.6", "a");
		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		assertEquals(true, result.isValid());
	}

	@Test
	public void validateNullNotes() {
		VitalEntry vitalEntry = createTempatureVitalEntry("98.6", null);
		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		assertEquals(true, result.isValid());
	}

	@Test
	public void validateDate() {
		VitalEntry vitalEntry = createTempatureVitalEntry("98.6", "", DateHelper.parseDate("1/1/2012"));
		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		assertEquals(true, result.isValid());
	}

	@Test
	public void validateNullDate() {
		VitalEntry vitalEntry = createTempatureVitalEntry("98.6", "", null);
		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		assertEquals(false, result.isValid());

		assertError(result.getErrors().get(0), "Entry Date", "Entry date is required.");
	}

	private void assertError(ValidationError error, String expectedFieldName, String expectedMessage) {
		assertEquals(expectedFieldName, error.getFieldName());
		assertEquals(expectedMessage, error.getErrorMessage());
	}

	private VitalEntry createTempatureVitalEntry(String temperature, String notes) {
		return createTempatureVitalEntry(temperature, notes, new Date());
	}

	private VitalEntry createTempatureVitalEntry(String temperature, String notes, Date dt) {
		VitalEntry vitalEntry = new VitalEntry();
		vitalEntry.setNotes(notes);
		vitalEntry.setEntryDate(dt);
		vitalEntry.setId(UUID.randomUUID().toString());

		List<VitalObservation> vitalObservations = new ArrayList<VitalObservation>();
		vitalEntry.setVitalObservations(vitalObservations);

		VitalObservation temperatureObservation = new VitalObservation();
		temperatureObservation.setType(ObservationTypeCode.BODY_TEMPERATURE);
		temperatureObservation.setValue(temperature);
		vitalObservations.add(temperatureObservation);

		return vitalEntry;
	}

}
