package com.agilex.healthcare.mobilehealthplatform.validator.vitals;

import static org.junit.Assert.assertEquals;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.UUID;

import org.junit.Before;
import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.domain.ValidationError;
import com.agilex.healthcare.mobilehealthplatform.domain.ValidationResult;
import com.agilex.healthcare.mobilehealthplatform.domain.VitalEntry;
import com.agilex.healthcare.mobilehealthplatform.domain.VitalObservation;
import com.agilex.healthcare.mobilehealthplatform.domain.code.ObservationTypeCode;
import com.agilex.healthcare.utility.DateHelper;

public class VitalsRepirationValidatorTest {

	private static final String RESPIRATION_FIELD = "Respiration";

	private VitalRespirationValidator validator;

	@Before
	public void setup() {
		validator = new VitalRespirationValidator();
	}

	@Test
	public void validateInRangeData() {
		VitalEntry vitalEntry = createVitalEntry("40", "");
		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		assertEquals(true, result.isValid());
	}

	@Test
	public void validateErrorContainsActualVitalEntry() {
		VitalEntry vitalEntry = createVitalEntry("40", "");
		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		assertEquals(vitalEntry, result.getRequestObject());
	}

	@Test
	public void validateNullEntryReturnsError() {
		VitalEntry vitalEntry = null;
		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		assertEquals(false, result.isValid());
	}

	@Test
	public void validateLowRespirationEntryReturnsError() {
		VitalEntry vitalEntry = createVitalEntry("1", "");
		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		assertEquals(false, result.isValid());
		assertError(result.getErrors().get(0), RESPIRATION_FIELD, "Respiration field must be between 4.0 and 60.0");
	}

	@Test
	public void validateOnLowEntryReturnsValid() {
		VitalEntry vitalEntry = createVitalEntry("4", "");
		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		assertEquals(true, result.isValid());
	}

	@Test
	public void validateOnHighEntryReturnsValid() {
		VitalEntry vitalEntry = createVitalEntry("60", "");
		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		assertEquals(true, result.isValid());
	}

	@Test
	public void validateHighRespirationEntryReturnsError() {
		VitalEntry vitalEntry = createVitalEntry("100", "");
		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		assertEquals(false, result.isValid());
		assertError(result.getErrors().get(0), RESPIRATION_FIELD, "Respiration field must be between 4.0 and 60.0");
	}

	@Test
	public void validateEmptyTemperateEntryReturnsError() {
		VitalEntry vitalEntry = createVitalEntry("", "");
		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		assertEquals(false, result.isValid());
		assertError(result.getErrors().get(0), RESPIRATION_FIELD, "Respiration field is required.");
	}

	@Test
	public void validateNullTemperateEntryReturnsError() {
		VitalEntry vitalEntry = createVitalEntry(null, "");
		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		assertEquals(false, result.isValid());
		assertError(result.getErrors().get(0), RESPIRATION_FIELD, "Respiration field is required.");
	}

	@Test
	public void validateNullReturnsError() {
		ValidationResult<VitalEntry> result = validator.validate(null);
		assertEquals(false, result.isValid());
		assertError(result.getErrors().get(0), "Respiration", "Respiration vital entry object is required.");
	}

	@Test
	public void validateLengthOfNotes() {
		VitalEntry vitalEntry = createVitalEntry("40", "a");
		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		assertEquals(true, result.isValid());
	}

	@Test
	public void validateNullNotes() {
		VitalEntry vitalEntry = createVitalEntry("40", null);
		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		assertEquals(true, result.isValid());
	}

	@Test
	public void validateDate() {
		VitalEntry vitalEntry = createVitalEntry("40", "", DateHelper.parseDate("1/1/2012"));
		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		assertEquals(true, result.isValid());
	}

	@Test
	public void validateNullDate() {
		VitalEntry vitalEntry = createVitalEntry("40", "", null);
		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		assertEquals(false, result.isValid());

		assertError(result.getErrors().get(0), "Entry Date", "Entry date is required.");
	}

	private void assertError(ValidationError error, String expectedFieldName, String expectedMessage) {
		assertEquals(expectedFieldName, error.getFieldName());
		assertEquals(expectedMessage, error.getErrorMessage());
	}

	private VitalEntry createVitalEntry(String respiration, String notes) {
		return createVitalEntry(respiration, notes, new Date());
	}

	private VitalEntry createVitalEntry(String respiration, String notes, Date dt) {
		VitalEntry vitalEntry = new VitalEntry();
		vitalEntry.setNotes(notes);
		vitalEntry.setEntryDate(dt);
		vitalEntry.setId(UUID.randomUUID().toString());

		List<VitalObservation> vitalObservations = new ArrayList<VitalObservation>();
		vitalEntry.setVitalObservations(vitalObservations);

		VitalObservation temperatureObservation = new VitalObservation();
		temperatureObservation.setType(ObservationTypeCode.RESPIRATION);
		temperatureObservation.setValue(respiration);
		vitalObservations.add(temperatureObservation);

		return vitalEntry;
	}

}
