package com.agilex.healthcare.mobilehealthplatform.validator.vitals;

import static org.junit.Assert.assertEquals;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.UUID;

import org.junit.Before;
import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.domain.ValidationError;
import com.agilex.healthcare.mobilehealthplatform.domain.ValidationResult;
import com.agilex.healthcare.mobilehealthplatform.domain.VitalEntry;
import com.agilex.healthcare.mobilehealthplatform.domain.VitalObservation;
import com.agilex.healthcare.mobilehealthplatform.domain.code.ObservationTypeCode;
import com.agilex.healthcare.utility.DateHelper;

public class VitalsPainValidatorTest {

	private static final String PAIN_FIELD_NAME = "Pain";
	private static final String VALID_PAIN_ENTRY = "4";

	private VitalPainValidator validator;

	@Before
	public void setup() {
		validator = new VitalPainValidator();
	}

	@Test
	public void validateInRangeData() {
		VitalEntry vitalEntry = createVitalEntry(VALID_PAIN_ENTRY, "");
		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		assertEquals(true, result.isValid());
	}

	@Test
	public void validateErrorContainsActualVitalEntry() {
		VitalEntry vitalEntry = createVitalEntry(VALID_PAIN_ENTRY, "");
		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		assertEquals(vitalEntry, result.getRequestObject());
	}

	@Test
	public void validateNullEntryReturnsError() {
		VitalEntry vitalEntry = null;
		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		assertEquals(false, result.isValid());
	}

	@Test
	public void validateLowEntryReturnsError() {
		VitalEntry vitalEntry = createVitalEntry("-1", "");
		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		assertEquals(false, result.isValid());
		assertError(result.getErrors().get(0), "Pain", "Pain field must be between 0.0 and 10.0");
	}

	@Test
	public void validateOnLowEntryReturnsValid() {
		VitalEntry vitalEntry = createVitalEntry("0", "");
		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		assertEquals(true, result.isValid());
	}

	@Test
	public void validateOnHighEntryReturnsValid() {
		VitalEntry vitalEntry = createVitalEntry("10", "");
		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		assertEquals(true, result.isValid());
	}

	@Test
	public void validateHighEntryReturnsError() {
		VitalEntry vitalEntry = createVitalEntry("11", "");
		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		assertEquals(false, result.isValid());
		assertError(result.getErrors().get(0), PAIN_FIELD_NAME, "Pain field must be between 0.0 and 10.0");
	}

	@Test
	public void validateEmptyTemperateEntryReturnsError() {
		VitalEntry vitalEntry = createVitalEntry("", "");
		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		assertEquals(false, result.isValid());
		assertError(result.getErrors().get(0), PAIN_FIELD_NAME, "Pain field is required.");
	}

	@Test
	public void validateNullTemperateEntryReturnsError() {
		VitalEntry vitalEntry = createVitalEntry(null, "");
		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		assertEquals(false, result.isValid());
		assertError(result.getErrors().get(0), PAIN_FIELD_NAME, "Pain field is required.");
	}

	@Test
	public void validateNullReturnsError() {
		ValidationResult<VitalEntry> result = validator.validate(null);
		assertEquals(false, result.isValid());
		assertError(result.getErrors().get(0), "Pain", "Pain vital entry object is required.");
	}

	@Test
	public void validateLengthOfNotes() {
		VitalEntry vitalEntry = createVitalEntry(VALID_PAIN_ENTRY, "a");
		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		assertEquals(true, result.isValid());
	}

	@Test
	public void validateNullNotes() {
		VitalEntry vitalEntry = createVitalEntry(VALID_PAIN_ENTRY, null);
		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		assertEquals(true, result.isValid());
	}

	@Test
	public void validateDate() {
		VitalEntry vitalEntry = createVitalEntry(VALID_PAIN_ENTRY, "", DateHelper.parseDate("1/1/2012"));
		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		assertEquals(true, result.isValid());
	}

	@Test
	public void validateNullDate() {
		VitalEntry vitalEntry = createVitalEntry(VALID_PAIN_ENTRY, "", null);
		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		assertEquals(false, result.isValid());
		assertError(result.getErrors().get(0), "Entry Date", "Entry date is required.");
	}

	@Test
	public void twoValdiationExceptions() {
		VitalEntry vitalEntry = createVitalEntry("1000", "", null);
		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		assertEquals(false, result.isValid());
		assertEquals(2, result.getErrors().size());
		assertError(result.getErrors().get(0), "Entry Date", "Entry date is required.");
		assertError(result.getErrors().get(1), "Pain", "Pain field must be between 0.0 and 10.0");
	}

	private void assertError(ValidationError error, String expectedFieldName, String expectedMessage) {
		assertEquals(expectedFieldName, error.getFieldName());
		assertEquals(expectedMessage, error.getErrorMessage());
	}

	private VitalEntry createVitalEntry(String pain, String notes) {
		return createVitalEntry(pain, notes, new Date());
	}

	private VitalEntry createVitalEntry(String pain, String notes, Date dt) {
		VitalEntry vitalEntry = new VitalEntry();
		vitalEntry.setNotes(notes);
		vitalEntry.setEntryDate(dt);
		vitalEntry.setId(UUID.randomUUID().toString());

		List<VitalObservation> vitalObservations = new ArrayList<VitalObservation>();
		vitalEntry.setVitalObservations(vitalObservations);

		VitalObservation temperatureObservation = new VitalObservation();
		temperatureObservation.setType(ObservationTypeCode.PAIN);
		temperatureObservation.setValue(pain);
		vitalObservations.add(temperatureObservation);

		
		return vitalEntry;
	}

}
