package com.agilex.healthcare.mobilehealthplatform.validator.vitals;

import static org.junit.Assert.assertEquals;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.domain.ValidationResult;
import com.agilex.healthcare.mobilehealthplatform.domain.VitalEntry;
import com.agilex.healthcare.mobilehealthplatform.domain.VitalObservation;
import com.agilex.healthcare.mobilehealthplatform.domain.code.ObservationTypeCode;
import com.agilex.healthcare.utility.DateHelper;

public class BloodPressureValidatorTest {

	BloodPressureValidator validator = new BloodPressureValidator();
	private String notes = "Java is a programming language originally developed by James Gosling at Sun Microsystems (which has since merged into"
			+ " Oracle Corporation) and released in 1995 as a core component of Sun Microsystems' Java platform. The language derives much of its "
			+ "syntax from C and C++ but has a simpler object model and fewer low-level facilities. Java applications are typically "
			+ "compiled to bytecode (class file) that can run on any Java Virtual Machine (JVM) regardless of computer architecture. "
			+ "Java is a general-purpose, concurrent, class-based, object-oriented language that is specifically designed " + "to have as few implementation dependencies as possible. ";

	@Test
	public void verifyBPWithSystolic120_Diastolic80_Pulse90isValid() {

		VitalEntry vitalEntry = createVitalEntry("120", "80", "90", null);
		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		assertEquals(true, result.isValid());
	}

	@Test
	public void verifyBPWithSystolic60_Diastolic80_Pulse90isValid() {
		VitalEntry vitalEntry = createVitalEntry("60", "80", "90", null);
		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		assertEquals(true, result.isValid());
	}

	@Test
	public void verifyBPWithSystolic220_Diastolic80_Pulse90isValid() {
		VitalEntry vitalEntry = createVitalEntry("220", "80", "90", null);
		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		assertEquals(true, result.isValid());
	}

	@Test
	public void verifyBPWithSystolicNull_Diastolic80_Pulse90isInValid() {
		VitalEntry vitalEntry = createVitalEntry(null, "80", "90", null);

		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		validateSystolicValidationError(result);
	}

	private void validateSystolicValidationError(ValidationResult<VitalEntry> result) {
		assertEquals(1, result.getErrors().size());
		assertEquals("Systolic blood pressure", result.getErrors().get(0).getFieldName());
		// assertEquals("?", result.getErrors().get(0).getErrorMessage());
	}

	@Test
	public void verifyBPWithSystolic30_Diastolic80_Pulse90isInValid() {
		VitalEntry vitalEntry = createVitalEntry("30", "80", "90", null);

		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		validateSystolicValidationError(result);
	}

	@Test
	public void verifyBPWithSystolic59_Diastolic80_Pulse90isInValid() {
		VitalEntry vitalEntry = createVitalEntry("59", "80", "90", null);

		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		validateSystolicValidationError(result);
	}

	@Test
	public void verifyBPWithSystolic221_Diastolic80_Pulse90isInValid() {
		VitalEntry vitalEntry = createVitalEntry("221", "80", "90", null);

		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		validateSystolicValidationError(result);
	}

	@Test
	public void verifyBPWithSystolicXYZ_Diastolic80_Pulse90isInValid() {
		VitalEntry vitalEntry = createVitalEntry("XYZ", "80", "90", null);

		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		validateSystolicValidationError(result);
	}

	@Test
	public void verifyBPWithSystolic120_Diastolic40_Pulse90isValid() {
		VitalEntry vitalEntry = createVitalEntry("120", "40", "90", null);
		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		assertEquals(true, result.isValid());
	}

	@Test
	public void verifyBPWithSystolic120_Diastolic160_Pulse90isValid() {
		VitalEntry vitalEntry = createVitalEntry("120", "160", "90", null);
		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		assertEquals(true, result.isValid());
	}

	@Test
	public void verifyBPWithSystolic120_DiastolicNull_Pulse90isInValid() {
		VitalEntry vitalEntry = createVitalEntry("120", null, "90", null);

		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		validateDiastolicValidationError(result);
	}

	private void validateDiastolicValidationError(ValidationResult<VitalEntry> result) {
		assertEquals(1, result.getErrors().size());
		assertEquals("Diastolic blood pressure", result.getErrors().get(0).getFieldName());
	}

	@Test
	public void verifyBPWithSystolic120_Diastolic80_Pulse30isInValid() {
		VitalEntry vitalEntry = createVitalEntry("120", "30", "90", null);

		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		validateDiastolicValidationError(result);
	}

	@Test
	public void verifyBPWithSystolic120_Diastolic39_Pulse90isInValid() {
		VitalEntry vitalEntry = createVitalEntry("120", "39", "90", null);

		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		validateDiastolicValidationError(result);
	}

	@Test
	public void verifyBPWithSystolic120_Diastolic161_Pulse90isInValid() {
		VitalEntry vitalEntry = createVitalEntry("120", "161", "90", null);

		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		validateDiastolicValidationError(result);
	}

	@Test
	public void verifyBPWithSystolic120_DiastolicXYZ_Pulse90isInValid() {
		VitalEntry vitalEntry = createVitalEntry("120", "XYZ", "90", null);

		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		validateDiastolicValidationError(result);
	}

	@Test
	public void verifyBPWithSystolic120_Diastolic90_Pulse40isValid() {
		VitalEntry vitalEntry = createVitalEntry("120", "90", "40", null);
		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		assertEquals(true, result.isValid());
	}

	@Test
	public void verifyBPWithSystolic120_Diastolic90_Pulse250isValid() {
		VitalEntry vitalEntry = createVitalEntry("120", "90", "250", null);
		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		assertEquals(true, result.isValid());
	}

	@Test
	public void verifyBPWithSystolic120_Diastolic90_PulseNullisInValid() {
		VitalEntry vitalEntry = createVitalEntry("120", "90", null, null);

		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		validatePulseValidationError(result);
	}

	private void validatePulseValidationError(ValidationResult<VitalEntry> result) {
		assertEquals(1, result.getErrors().size());
		assertEquals("Pulse", result.getErrors().get(0).getFieldName());
	}

	@Test
	public void verifyBPWithSystolic120_Diastolic90_Pulse30isInValid() {
		VitalEntry vitalEntry = createVitalEntry("120", "90", "30", null);

		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		validatePulseValidationError(result);
	}

	@Test
	public void verifyBPWithSystolic120_Diastolic90_Pulse39isInValid() {
		VitalEntry vitalEntry = createVitalEntry("120", "90", "39", null);

		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		validatePulseValidationError(result);
	}

	@Test
	public void verifyBPWithSystolic120_Diastolic90_Pulse251isInValid() {
		VitalEntry vitalEntry = createVitalEntry("120", "90", "251", null);

		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		validatePulseValidationError(result);
	}

	@Test
	public void verifyBPWithSystolic120_Diastolic90_PulseXYZisInValid() {
		VitalEntry vitalEntry = createVitalEntry("120", "90", "XYZ", null);

		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		validatePulseValidationError(result);
	}

	@Test
	public void verifyBPWithSystolic30_Diastolic90_PulseXYZisInValid() {
		VitalEntry vitalEntry = createVitalEntry("30", "90", "XYZ", null);

		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		assertEquals(2, result.getErrors().size());
	}

	@Test
	public void verifyBPWithSystolic30_Diastolic400_PulseXYZisInValid() {
		VitalEntry vitalEntry = createVitalEntry("30", "400", "XYZ", null);

		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		assertEquals(3, result.getErrors().size());
	}

	@Test
	public void verifyBPWithSystolic120_Diastolic90_Pulse80NotesWith30CharsisValid() {
		VitalEntry vitalEntry = createVitalEntry("120", "90", "90", notes.substring(0, 29));

		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		assertEquals(true, result.isValid());

	}

	@Test
	public void verifyBPWithSystolic120_Diastolic90_Pulse80NotesWith250CharsisValid() {
		VitalEntry vitalEntry = createVitalEntry("120", "90", "90", notes.substring(0, 250));

		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		assertEquals(true, result.isValid());
	}

	@Test
	public void verifyBPWithSystolic120_Diastolic90_Pulse80WithCurrentEntryTimeisValid() {
		VitalEntry vitalEntry = createVitalEntry("120", "90", "90", null);
		vitalEntry.setEntryDate(new Date());

		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		assertEquals(true, result.isValid());
	}

	@Test
	public void verifyBPWithSystolic120_Diastolic90_Pulse80WithPastEntryTimeisValid() {
		VitalEntry vitalEntry = createVitalEntry("120", "90", "90", null);
		vitalEntry.setEntryDate(DateHelper.parse("10/12/2011 12:30:00", "MM/dd/yyyy HH:mm:ss"));

		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		assertEquals(true, result.isValid());
	}

	@Test
	public void verifyBPWithSystolic120_Diastolic90_Pulse80WithOutEntryTimeisInValid() {
		VitalEntry vitalEntry = createVitalEntry("120", "90", "90", null);
		vitalEntry.setEntryDate(null);

		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		assertEquals(1, result.getErrors().size());
		assertEquals("Entry Date", result.getErrors().get(0).getFieldName());
	}

	@Test
	public void verifyVitalsWithOutObservationsIsInvalid() {
		VitalEntry vitalEntry = new VitalEntry();
		vitalEntry.setEntryDate(new Date());

		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		assertEquals(3, result.getErrors().size());
	}

	@Test
	public void verifyVitalsWithOutPulseIsInvalid() {
		VitalEntry vitalEntry = new VitalEntry();
		vitalEntry.setEntryDate(new Date());

		List<VitalObservation> vitalObservations = new ArrayList<VitalObservation>();
		vitalEntry.setVitalObservations(vitalObservations);

		VitalObservation systolicObs = new VitalObservation();
		systolicObs.setType(ObservationTypeCode.INTRAVASCULAR_SYSTOLIC);
		systolicObs.setValue("120");
		vitalObservations.add(systolicObs);

		VitalObservation diastolicObs = new VitalObservation();
		diastolicObs.setType(ObservationTypeCode.INTRAVASCULAR_DIASTOLIC);
		diastolicObs.setValue("90");
		vitalObservations.add(diastolicObs);

		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		assertEquals(1, result.getErrors().size());
		assertEquals("Pulse", result.getErrors().get(0).getFieldName());
		assertEquals("Pulse field is required.", result.getErrors().get(0).getErrorMessage());
	}

	@Test
	public void verifyVitalsWithOutSystolicIsInvalid() {
		VitalEntry vitalEntry = new VitalEntry();
		vitalEntry.setEntryDate(new Date());

		List<VitalObservation> vitalObservations = new ArrayList<VitalObservation>();
		vitalEntry.setVitalObservations(vitalObservations);

		VitalObservation diastolicObs = new VitalObservation();
		diastolicObs.setType(ObservationTypeCode.INTRAVASCULAR_DIASTOLIC);
		diastolicObs.setValue("90");
		vitalObservations.add(diastolicObs);

		VitalObservation pulseObs = new VitalObservation();
		pulseObs.setType(ObservationTypeCode.HEART_BEAT);
		pulseObs.setValue("60");
		vitalObservations.add(pulseObs);

		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		assertEquals(1, result.getErrors().size());
		assertEquals("Systolic blood pressure", result.getErrors().get(0).getFieldName());
		assertEquals("Systolic blood pressure field is required.", result.getErrors().get(0).getErrorMessage());
	}

	@Test
	public void verifyVitalsWithOutDiastolicIsInvalid() {
		VitalEntry vitalEntry = new VitalEntry();
		vitalEntry.setEntryDate(new Date());

		List<VitalObservation> vitalObservations = new ArrayList<VitalObservation>();
		vitalEntry.setVitalObservations(vitalObservations);

		VitalObservation systolicObs = new VitalObservation();
		systolicObs.setType(ObservationTypeCode.INTRAVASCULAR_SYSTOLIC);
		systolicObs.setValue("120");
		vitalObservations.add(systolicObs);

		VitalObservation pulseObs = new VitalObservation();
		pulseObs.setType(ObservationTypeCode.HEART_BEAT);
		pulseObs.setValue("60");
		vitalObservations.add(pulseObs);

		ValidationResult<VitalEntry> result = validator.validate(vitalEntry);
		assertEquals(1, result.getErrors().size());
		assertEquals("Diastolic blood pressure", result.getErrors().get(0).getFieldName());
	}

	private VitalEntry createVitalEntry(String systolic, String diastolic, String pulse, String notes) {
		VitalEntry vitalEntry = new VitalEntry();
		vitalEntry.setNotes(notes);
		vitalEntry.setEntryDate(new Date());

		List<VitalObservation> vitalObservations = new ArrayList<VitalObservation>();
		vitalEntry.setVitalObservations(vitalObservations);

		VitalObservation systolicObs = new VitalObservation();
		systolicObs.setType(ObservationTypeCode.INTRAVASCULAR_SYSTOLIC);
		systolicObs.setValue(systolic);
		vitalObservations.add(systolicObs);

		VitalObservation diastolicObs = new VitalObservation();
		diastolicObs.setType(ObservationTypeCode.INTRAVASCULAR_DIASTOLIC);
		diastolicObs.setValue(diastolic);
		vitalObservations.add(diastolicObs);

		VitalObservation pulseObs = new VitalObservation();
		pulseObs.setType(ObservationTypeCode.HEART_BEAT);
		pulseObs.setValue(pulse);
		vitalObservations.add(pulseObs);
		return vitalEntry;
	}

}
