package com.agilex.healthcare.mobilehealthplatform.validator.appointment;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;

import java.util.Calendar;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;

import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.domain.AppointmentRequest;
import com.agilex.healthcare.mobilehealthplatform.domain.Facility;
import com.agilex.healthcare.mobilehealthplatform.domain.ValidationError;
import com.agilex.healthcare.mobilehealthplatform.domain.ValidationResult;
import com.agilex.healthcare.mobilehealthplatform.enumeration.AppointmentRequestPurposeOfVisit;
import com.agilex.healthcare.mobilehealthplatform.enumeration.AppointmentRequestType;
import com.agilex.healthcare.mobilehealthplatform.enumeration.AppointmentRequestVisitType;
import com.agilex.healthcare.utility.DateHelper;

public class AppointmentRequestValidatorTest {
    AppointmentRequestValidator validator = new AppointmentRequestValidator(96, 120);
    private HashSet<String> bestTimetoCall;

    @Test
    public void verifyAppointmentRequestValid() {
        AppointmentRequest appointmentRequest = createValidAppointmentRequest();

        ValidationResult<AppointmentRequest> result = validator.validate(appointmentRequest, true);
        assertEquals(true, result.isValid());
    }

    @Test
    public void verifyNull() {
        AppointmentRequest appointmentRequest = createValidAppointmentRequest();
        appointmentRequest.setAppointmentType(null);
        appointmentRequest.setVisitType(null);
        appointmentRequest.setFacility(null);
        appointmentRequest.setEmail(null);
        appointmentRequest.setPhoneNumber(null);
        appointmentRequest.setOptionDate1(null);
        appointmentRequest.setOptionTime1(null);
        appointmentRequest.setOptionDate2(null);
        appointmentRequest.setOptionTime2(null);
        appointmentRequest.setOptionDate3(null);
        appointmentRequest.setOptionTime3(null);
        appointmentRequest.setBestTimetoCall(null);
        appointmentRequest.setPurposeOfVisit(null);
        appointmentRequest.setProviderId(null);

        ValidationResult<AppointmentRequest> result = validator.validate(appointmentRequest, false);
        assertEquals(false, result.isValid());
        assertEquals(10, result.getErrors().size());
    }

    @Test
    public void verifyNotEmpty() {
        String facilityAddress = "";
        String facilityCity = "";
        String facilityState = "";
        String facilityCode = "";
        String facilityName = "";
        String facilityParentSiteCode = "";
        String facilityType = "";

        Facility facility = new Facility();
        facility.setAddress(facilityAddress);
        facility.setCity(facilityCity);
        facility.setFacilityCode(facilityCode);
        facility.setName(facilityName);
        facility.setParentSiteCode(facilityParentSiteCode);
        facility.setState(facilityState);
        facility.setType(facilityType);

        AppointmentRequest appointmentRequest = createValidAppointmentRequest();
        appointmentRequest.setAppointmentType("");
        appointmentRequest.setVisitType("");
        appointmentRequest.setFacility(new Facility());
        appointmentRequest.setEmail("");
        appointmentRequest.setPhoneNumber("");
        appointmentRequest.setOptionDate1("");
        appointmentRequest.setOptionTime1("");
        appointmentRequest.setOptionDate2("");
        appointmentRequest.setOptionTime2("");
        appointmentRequest.setOptionDate3("");
        appointmentRequest.setOptionTime3("");
        appointmentRequest.setBestTimetoCall(bestTimetoCall);
        appointmentRequest.setPurposeOfVisit("");
        appointmentRequest.setProviderId("");

        ValidationResult<AppointmentRequest> result = validator.validate(appointmentRequest, false);
        assertEquals(false, result.isValid());
        assertEquals(9, result.getErrors().size());
    }

    @Test
    public void verifyInvalidMinLengthTextMessaging() {
        AppointmentRequest appointmentRequest = createValidAppointmentRequest();
        appointmentRequest.setTextMessagingPhoneNumber("5555-5555");

        ValidationResult<AppointmentRequest> result = validator.validate(appointmentRequest, false);
        assertEquals(false, result.isValid());
    }

    @Test
    public void verifyInvalidMaxLengthTextMessaging() {
        AppointmentRequest appointmentRequest = createValidAppointmentRequest();
        appointmentRequest.setTextMessagingPhoneNumber("5555-5555-5555-5555-5555-5555-5555-5555-5555-5555-5");

        ValidationResult<AppointmentRequest> result = validator.validate(appointmentRequest, false);
        assertEquals(false, result.isValid());
    }

    @Test
    public void verifyInvalidMinDigitTextMessaging() {
        AppointmentRequest appointmentRequest = createValidAppointmentRequest();
        appointmentRequest.setTextMessagingPhoneNumber("5---------");

        ValidationResult<AppointmentRequest> result = validator.validate(appointmentRequest, false);
        assertEquals(false, result.isValid());
    }

    @Test
    public void verifyInvalidCharactersInTextMessaging() {
        List<String> PhoneNumbersWithInvalidCharacters = new LinkedList<String>();
        PhoneNumbersWithInvalidCharacters.add("+1 (555) 555-5555!");
        PhoneNumbersWithInvalidCharacters.add("+1 (555) 555-5555@");
        PhoneNumbersWithInvalidCharacters.add("+1 (555) 555-5555#");
        PhoneNumbersWithInvalidCharacters.add("+1 (555) 555-5555$");
        PhoneNumbersWithInvalidCharacters.add("+1 (555) 555-5555%");
        PhoneNumbersWithInvalidCharacters.add("+1 (555) 555-5555^");
        PhoneNumbersWithInvalidCharacters.add("+1 (555) 555-5555&");
        PhoneNumbersWithInvalidCharacters.add("+1 (555) 555-5555*");
        PhoneNumbersWithInvalidCharacters.add("+1 (555) 555-5555=");
        PhoneNumbersWithInvalidCharacters.add("+1 (555) 555-5555_");
        PhoneNumbersWithInvalidCharacters.add("+1 (555) 555-5555{");
        PhoneNumbersWithInvalidCharacters.add("+1 (555) 555-5555}");
        PhoneNumbersWithInvalidCharacters.add("+1 (555) 555-5555[");
        PhoneNumbersWithInvalidCharacters.add("+1 (555) 555-5555]");
        PhoneNumbersWithInvalidCharacters.add("+1 (555) 555-5555\\");
        PhoneNumbersWithInvalidCharacters.add("+1 (555) 555-5555|");
        PhoneNumbersWithInvalidCharacters.add("+1 (555) 555-5555?");
        PhoneNumbersWithInvalidCharacters.add("+1 (555) 555-5555,");
        PhoneNumbersWithInvalidCharacters.add("+1 (555) 555-5555<");
        PhoneNumbersWithInvalidCharacters.add("+1 (555) 555-5555>");

        for (String PhoneNumberWithInvalidCharacters : PhoneNumbersWithInvalidCharacters) {
            AppointmentRequest appointmentRequest = createValidAppointmentRequest();
            appointmentRequest.setTextMessagingPhoneNumber(PhoneNumberWithInvalidCharacters);

            ValidationResult<AppointmentRequest> result = validator.validate(appointmentRequest, false);
            assertEquals(false, result.isValid());
        }
    }

    @Test
    public void verifyEmptyTextMessagingOk() {
        AppointmentRequest appointmentRequest = createValidAppointmentRequest();
        appointmentRequest.setTextMessagingPhoneNumber("");

        ValidationResult<AppointmentRequest> result = validator.validate(appointmentRequest, false);
        assertEquals(true, result.isValid());
    }

    @Test
    public void verifyNullTextMessagingOk() {
        AppointmentRequest appointmentRequest = createValidAppointmentRequest();
        appointmentRequest.setTextMessagingPhoneNumber(null);

        ValidationResult<AppointmentRequest> result = validator.validate(appointmentRequest, false);
        assertEquals(true, result.isValid());
    }

    @Test
    public void verifyInvalidMinLengthPhoneNumber() {
        AppointmentRequest appointmentRequest = createValidAppointmentRequest();
        appointmentRequest.setPhoneNumber("5555-5555");

        ValidationResult<AppointmentRequest> result = validator.validate(appointmentRequest, false);
        assertEquals(false, result.isValid());
    }

    @Test
    public void verifyInvalidMaxLengthPhoneNumber() {
        AppointmentRequest appointmentRequest = createValidAppointmentRequest();
        appointmentRequest.setPhoneNumber("5555-5555-5555-5555-5555-5555-5555-5555-5555-5555-5");

        ValidationResult<AppointmentRequest> result = validator.validate(appointmentRequest, false);
        assertEquals(false, result.isValid());
    }

    @Test
    public void verifyInvalidMinDigitPhoneNumber() {
        AppointmentRequest appointmentRequest = createValidAppointmentRequest();
        appointmentRequest.setPhoneNumber("5---------");

        ValidationResult<AppointmentRequest> result = validator.validate(appointmentRequest, false);
        assertEquals(false, result.isValid());
    }

    @Test
    public void verifyInvalidCharactersInPhoneNumber() {
        List<String> PhoneNumbersWithInvalidCharacters = new LinkedList<String>();
        PhoneNumbersWithInvalidCharacters.add("+1 (555) 555-5555!");
        PhoneNumbersWithInvalidCharacters.add("+1 (555) 555-5555@");
        PhoneNumbersWithInvalidCharacters.add("+1 (555) 555-5555#");
        PhoneNumbersWithInvalidCharacters.add("+1 (555) 555-5555$");
        PhoneNumbersWithInvalidCharacters.add("+1 (555) 555-5555%");
        PhoneNumbersWithInvalidCharacters.add("+1 (555) 555-5555^");
        PhoneNumbersWithInvalidCharacters.add("+1 (555) 555-5555&");
        PhoneNumbersWithInvalidCharacters.add("+1 (555) 555-5555*");
        PhoneNumbersWithInvalidCharacters.add("+1 (555) 555-5555=");
        PhoneNumbersWithInvalidCharacters.add("+1 (555) 555-5555_");
        PhoneNumbersWithInvalidCharacters.add("+1 (555) 555-5555{");
        PhoneNumbersWithInvalidCharacters.add("+1 (555) 555-5555}");
        PhoneNumbersWithInvalidCharacters.add("+1 (555) 555-5555[");
        PhoneNumbersWithInvalidCharacters.add("+1 (555) 555-5555]");
        PhoneNumbersWithInvalidCharacters.add("+1 (555) 555-5555\\");
        PhoneNumbersWithInvalidCharacters.add("+1 (555) 555-5555|");
        PhoneNumbersWithInvalidCharacters.add("+1 (555) 555-5555?");
        PhoneNumbersWithInvalidCharacters.add("+1 (555) 555-5555,");
        PhoneNumbersWithInvalidCharacters.add("+1 (555) 555-5555<");
        PhoneNumbersWithInvalidCharacters.add("+1 (555) 555-5555>");

        for (String PhoneNumberWithInvalidCharacters : PhoneNumbersWithInvalidCharacters) {
            AppointmentRequest appointmentRequest = createValidAppointmentRequest();
            appointmentRequest.setPhoneNumber(PhoneNumberWithInvalidCharacters);

            ValidationResult<AppointmentRequest> result = validator.validate(appointmentRequest, false);
            assertEquals(false, result.isValid());
        }
    }

    //Just a simple test to ensure email validators are invoked as part of appointment request validation
    //more detailed email tests are found in ValidatorHelperTest
    @Test
    public void verifyValidEmail() {
        String validEmail = "niceandsimple@example.com";

        AppointmentRequest appointmentRequest = createValidAppointmentRequest();
        appointmentRequest.setEmail(validEmail);

        ValidationResult<AppointmentRequest> result = validator.validate(appointmentRequest, false);
        assertTrue(result.isValid());
    }

    //Just a simple test to ensure email validators are invoked as part of appointment request validation
    //more detailed email tests are found in ValidatorHelperTest
    @Test
    public void verifyInvalidEmail() {
        String invalidEmail = "Abc.example.com";

        AppointmentRequest appointmentRequest = createValidAppointmentRequest();
        appointmentRequest.setEmail(invalidEmail);

        ValidationResult<AppointmentRequest> result = validator.validate(appointmentRequest, false);
        assertFalse(result.isValid());
    }

    @Test
    public void verifyInvalidAppointmentType() {
        AppointmentRequest appointmentRequest = createValidAppointmentRequest();
        appointmentRequest.setAppointmentType("INVALID_TYPE");

        ValidationResult<AppointmentRequest> result = validator.validate(appointmentRequest, false);
        assertEquals(false, result.isValid());
    }

    @Test
    public void verifyNonUniqueOptions12() {
        AppointmentRequest appointmentRequest = createValidAppointmentRequest();

        Calendar cal = Calendar.getInstance();
        cal.add(Calendar.DATE, 7);
        String nonUniqueDateString = DateHelper.formatDate(cal.getTime());
        appointmentRequest.setOptionDate1(nonUniqueDateString);
        appointmentRequest.setOptionTime1("AM");
        appointmentRequest.setOptionDate2(nonUniqueDateString);
        appointmentRequest.setOptionTime2("AM");

        ValidationResult<AppointmentRequest> result = validator.validate(appointmentRequest, false);
        assertEquals(false, result.isValid());
    }

    @Test
    public void verifyNonUniqueOptions23() {
        AppointmentRequest appointmentRequest = createValidAppointmentRequest();

        Calendar cal = Calendar.getInstance();
        cal.add(Calendar.DATE, 7);
        String nonUniqueDateString = DateHelper.formatDate(cal.getTime());
        appointmentRequest.setOptionDate2(nonUniqueDateString);
        appointmentRequest.setOptionTime2("AM");
        appointmentRequest.setOptionDate3(nonUniqueDateString);
        appointmentRequest.setOptionTime3("AM");

        ValidationResult<AppointmentRequest> result = validator.validate(appointmentRequest, false);
        assertEquals(false, result.isValid());
    }

    @Test
    public void verifyNonUniqueOptions13() {
        AppointmentRequest appointmentRequest = createValidAppointmentRequest();

        Calendar cal = Calendar.getInstance();
        cal.add(Calendar.DATE, 7);
        String nonUniqueDateString = DateHelper.formatDate(cal.getTime());
        appointmentRequest.setOptionDate1(nonUniqueDateString);
        appointmentRequest.setOptionTime1("AM");
        appointmentRequest.setOptionDate3(nonUniqueDateString);
        appointmentRequest.setOptionTime3("AM");

        ValidationResult<AppointmentRequest> result = validator.validate(appointmentRequest, false);
        assertEquals(false, result.isValid());
    }

    @Test
    public void verifyInvalidMinTimeOption() {
        AppointmentRequest appointmentRequest = createValidAppointmentRequest();

        Calendar cal = Calendar.getInstance();
        cal.add(Calendar.DATE, 2);
        appointmentRequest.setOptionDate1(DateHelper.formatDate(cal.getTime()));
        appointmentRequest.setOptionTime1("AM");

        ValidationResult<AppointmentRequest> result = validator.validate(appointmentRequest, true);
        assertEquals(false, result.isValid());
    }

    @Test
    public void verifyInvalidMinTimeOptionIsValidWithNonFullValidation() {
        AppointmentRequest appointmentRequest = createValidAppointmentRequest();

        Calendar cal = Calendar.getInstance();
        cal.add(Calendar.DATE, 2);
        appointmentRequest.setOptionDate1(DateHelper.formatDate(cal.getTime()));
        appointmentRequest.setOptionTime1("AM");

        ValidationResult<AppointmentRequest> result = validator.validate(appointmentRequest, false);
        assertEquals(true, result.isValid());
    }

    @Test
    public void verifyDateOptionOfFiveDaysFromTodayIsValid() {
        AppointmentRequest appointmentRequest = createValidAppointmentRequest();

        Calendar cal = Calendar.getInstance();
        cal.add(Calendar.DATE, 5);
        appointmentRequest.setOptionDate1(DateHelper.formatDate(cal.getTime()));
        appointmentRequest.setOptionTime1("PM");

        ValidationResult<AppointmentRequest> result = validator.validate(appointmentRequest, true);
        assertEquals(true, result.isValid());
    }

    @Test
    public void verifyInvalidMaxTimeOption() {
        AppointmentRequest appointmentRequest = createValidAppointmentRequest();

        Calendar cal = Calendar.getInstance();
        cal.add(Calendar.DATE, 121);
        appointmentRequest.setOptionDate1(DateHelper.formatDate(cal.getTime()));
        appointmentRequest.setOptionTime1("AM");

        ValidationResult<AppointmentRequest> result = validator.validate(appointmentRequest, true);
        assertEquals(false, result.isValid());
    }

    @Test
    public void verifyInvalidMaxTimeOptionWithNonFullValidation() {
        AppointmentRequest appointmentRequest = createValidAppointmentRequest();

        Calendar cal = Calendar.getInstance();
        cal.add(Calendar.DATE, 121);
        appointmentRequest.setOptionDate1(DateHelper.formatDate(cal.getTime()));
        appointmentRequest.setOptionTime1("AM");

        ValidationResult<AppointmentRequest> result = validator.validate(appointmentRequest, false);
        assertEquals(true, result.isValid());
    }

    @Test
    public void verifyInvalidOtherPurposeOfVisitOption() {
        AppointmentRequest appointmentRequest = createValidAppointmentRequest();

        String textOver100Characters = "12345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901";
        appointmentRequest.setOtherPurposeOfVisit(textOver100Characters);

        ValidationResult<AppointmentRequest> result = validator.validate(appointmentRequest, false);
        assertEquals(false, result.isValid());
    }

    private AppointmentRequest createValidAppointmentRequest() {
        String appointmentType = AppointmentRequestType.MENTAL_HEALTH.getName();
        String visitType = AppointmentRequestVisitType.OFFICE_VISIT.getName();
        String email = "test@agilex.com";
        String phoneNumber = "+1 (555) 555-5555 x55.55";
        String purposeOfVisit = AppointmentRequestPurposeOfVisit.OTHER.getName();
        String otherPurposeOfVisit = "test other purpose of visit";
        String providerId = "PROV1";

        String facilityAddress = "test address";
        String facilityCity = "test city";
        String facilityState = "test state";
        String facilityCode = "test facility code";
        String facilityName = "Test Facility";
        String facilityParentSiteCode = "test parent site code";
        String facilityType = "test facility type";

        Facility facility = new Facility();
        facility.setAddress(facilityAddress);
        facility.setCity(facilityCity);
        facility.setFacilityCode(facilityCode);
        facility.setName(facilityName);
        facility.setParentSiteCode(facilityParentSiteCode);
        facility.setState(facilityState);
        facility.setType(facilityType);

        Calendar calendar = Calendar.getInstance();
        calendar.add(Calendar.DATE, 7);
        String optionDate1 = DateHelper.formatDate(calendar.getTime());
        String optionTime1 = "AM";
        calendar.add(Calendar.DATE, 1);
        String optionDate2 = DateHelper.formatDate(calendar.getTime());
        String optionTime2 = "PM";
        calendar.add(Calendar.DATE, 2);
        String optionDate3 = DateHelper.formatDate(calendar.getTime());
        String optionTime3 = "AM";

        bestTimetoCall = new HashSet<String>();
        bestTimetoCall.add("9 AM - 11 AM");
        bestTimetoCall.add("11 PM - 1 PM");


        AppointmentRequest a = new AppointmentRequest();

        a.setAppointmentType(appointmentType);
        a.setVisitType(visitType);
        a.setFacility(facility);
        a.setEmail(email);
        a.setPhoneNumber(phoneNumber);
        a.setOptionDate1(optionDate1);
        a.setOptionTime1(optionTime1);
        a.setOptionDate2(optionDate2);
        a.setOptionTime2(optionTime2);
        a.setOptionDate3(optionDate3);
        a.setOptionTime3(optionTime3);
        a.setBestTimetoCall(bestTimetoCall);
        a.setPurposeOfVisit(purposeOfVisit);
        a.setOtherPurposeOfVisit(otherPurposeOfVisit);
        a.setProviderId(providerId);

        return a;
    }

    /**
     * Call before assertion to print error messages for debugging
     *
     * @param result
     */
    @SuppressWarnings("unused")
    private void printErrors(ValidationResult<AppointmentRequest> result) {
        for (ValidationError error : result.getErrors()) {
            System.out.println(String.format("%s - %s", error.getFieldName(), error.getErrorMessage()));
        }
    }
}
