package com.agilex.healthcare.mobilehealthplatform.validator;

import com.agilex.healthcare.mobilehealthplatform.domain.ValidationError;
import org.junit.Test;

import java.util.Date;
import java.util.LinkedList;
import java.util.List;

import static org.junit.Assert.*;

public class ValidatorHelperTest {

	@Test
	public void verifyNotNullValidator() {
		String field = "NullTest";
		Object o = null;
		ValidationError error = ValidatorHelper.validateIsNotNull(o, new ValidationError(field, field));
		assertNotNull(error);
		assertEquals(field, error.getFieldName());
		assertEquals(field, error.getFieldName());
	}
	
	@Test
	public void verifyNotNullDateValidator() {
		String field = "DateTest";
		Date d = null;
		
		ValidationError error = ValidatorHelper.validateDateIsNotNull(d, new ValidationError(field, field));
		assertNotNull(error);
		assertEquals(field, error.getFieldName());
		assertEquals(field, error.getFieldName());
	}
	
	@Test
	public void verifyInclusiveTextLength() {
		String field = "InclusiveText";
		String lengthOfFive = "aaaaa";
		
		ValidationError shouldBeNull = ValidatorHelper.validateInclusiveTextLength(lengthOfFive, 5, new ValidationError(field, field));
		assertNull(shouldBeNull);
		
		ValidationError shouldBeError = ValidatorHelper.validateInclusiveTextLength(lengthOfFive, 4, new ValidationError(field, field));
		assertNotNull(shouldBeError);
		assertEquals(field, shouldBeError.getErrorMessage());
		assertEquals(field, shouldBeError.getFieldName());
	}
	
	@Test
	public void verifyInclusiveNumericBoundsIntBasic() {
		String field = "InclusiveBounds";
		int value = 100;
		int upperBound = 250;
		int lowerBound = 1;
		
		ValidationError shouldBeNull = ValidatorHelper.validateInclusiveNumericBounds(value, upperBound, lowerBound, new ValidationError(field, field));
		assertNull(shouldBeNull);
	}
	
	@Test
	public void verifyInclusiveNumericBoundsDoublesBasic() {
		String field = "InclusiveBounds";
		double value = 100.915;
		double upperBound = 180;
		double lowerBound = 50;
		
		ValidationError shouldBeNull = ValidatorHelper.validateInclusiveNumericBounds(value, upperBound, lowerBound, new ValidationError(field, field));
		assertNull(shouldBeNull);
	}
	
	@Test
	public void verifyInclusiveNumericBoundsIntUpperLimit() {
		String field = "InclusiveBounds";
		int value = 250;
		int upperBound = 250;
		int lowerBound = 1;
		
		ValidationError shouldBeNull = ValidatorHelper.validateInclusiveNumericBounds(value, upperBound, lowerBound, new ValidationError(field, field));
		assertNull(shouldBeNull);
		
		value = 251;
		ValidationError shouldBeError = ValidatorHelper.validateInclusiveNumericBounds(value, upperBound, lowerBound, new ValidationError(field, field));
		assertNotNull(shouldBeError);
		assertEquals(field, shouldBeError.getErrorMessage());
		assertEquals(field, shouldBeError.getFieldName());
	}
	
	@Test
	public void verifyInclusiveNumericBoundsDoublesUpperLimit() {
		String field = "InclusiveBounds";
		double value = 179.999999999999;
		double upperBound = 180;
		double lowerBound = 50;
		
		ValidationError shouldBeNull = ValidatorHelper.validateInclusiveNumericBounds(value, upperBound, lowerBound, new ValidationError(field, field));
		assertNull(shouldBeNull);
	}
	
	@Test
	public void verifyInclusiveNumericBoundsIntLowerLimit() {
		String field = "InclusiveBounds";
		int value = 1;
		int upperBound = 250;
		int lowerBound = 1;
		
		ValidationError shouldBeNull = ValidatorHelper.validateInclusiveNumericBounds(value, upperBound, lowerBound, new ValidationError(field, field));
		assertNull(shouldBeNull);
		
		value = 0;
		ValidationError shouldBeError = ValidatorHelper.validateInclusiveNumericBounds(value, upperBound, lowerBound, new ValidationError(field, field));
		assertNotNull(shouldBeError);
		assertEquals(field, shouldBeError.getErrorMessage());
		assertEquals(field, shouldBeError.getFieldName());
	}
	
	@Test
	public void verifyInclusiveNumericBoundsDoublesLowerLimit() {
		String field = "InclusiveBounds";
		double value = 50.0000001;
		double upperBound = 180;
		double lowerBound = 50;
		
		ValidationError shouldBeNull = ValidatorHelper.validateInclusiveNumericBounds(value, upperBound, lowerBound, new ValidationError(field, field));
		assertNull(shouldBeNull);
	}
	
	@Test
	public void verifyNonNegativeValidator() {
		String field = "NonNegative";
		int value = -10;
		
		ValidationError shouldBeError = ValidatorHelper.validateNonNegative(value, new ValidationError(field, field));
		assertNotNull(shouldBeError);
		assertEquals(field, shouldBeError.getErrorMessage());
		assertEquals(field, shouldBeError.getFieldName());
	}
	
	@Test
	public void verifyIsNotBlankOrNull() {
		String field = "BlankOrNull";
		String blank = "";
		
		ValidationError shouldBeError = ValidatorHelper.validateIsNotBlankOrNull(blank, new ValidationError(field, field));
		assertNotNull(shouldBeError);
		assertEquals(field, shouldBeError.getErrorMessage());
		assertEquals(field, shouldBeError.getFieldName());
	}

    @Test
    public void verifyPlainEmail(){
        String email = "name@domain.com";

        final boolean emailAddressFormatValid = ValidatorHelper.isEmailAddressFormatValid(email);
        assertTrue(emailAddressFormatValid);
    }
    @Test
    public void verifyEmailNameWithDash(){
        String email = "name-lastname@domain.com";

        final boolean emailAddressFormatValid = ValidatorHelper.isEmailAddressFormatValid(email);
        assertTrue(emailAddressFormatValid);
    }
    @Test
    public void verifyEmailDomainWithDash(){
        String email = "name@domain-dom.com";

        final boolean emailAddressFormatValid = ValidatorHelper.isEmailAddressFormatValid(email);
        assertTrue(emailAddressFormatValid);
    }

    @Test
    public void verifyValidEmail() {
        List<String> validEmails = new LinkedList<String>();
        validEmails.add("niceandsimple@example.com");
        validEmails.add("simplewith+symbol@example.com");
        validEmails.add("less.common@example.com");
        validEmails.add("a.little.more.unusual@dept.example.com");
        validEmails.add("user@[IPv6:2001:db8:1ff::a0b:dbd0]");
        validEmails.add("\"much.more unusual\"@example.com");
        validEmails.add("\"very.unusual.@.unusual.com\"@example.com");
        validEmails.add("\"v.(),:;<>[]\\\".V.\\\"v@\\\\ \\\"v\\\".u\"@s.e.com");
        validEmails.add("0@a");
        validEmails.add("!#$%&'*+@example.org");
        validEmails.add("-@example.org");
        validEmails.add("/@example.org");
        validEmails.add("=@example.org");
        validEmails.add("?^_`@example.org");
        validEmails.add("{}|~@example.org");
        validEmails.add("!#$%&'*+-/=?^_`{}|~@example.org");
        validEmails.add("\"/=?^_`{}|~ ? ^_`{}|~.a\"@e.org");
        validEmails.add("\"()<>[]:,;@\\\\\\\"!#$%&'*+-\"@e.org");
        validEmails.add("\"\"@example.org");
        validEmails.add("postbox@com");
        validEmails.add("name-lastname@domain.com");
        validEmails.add("name@domain-dom.com");

        for (String validEmail : validEmails) {
            assertTrue(ValidatorHelper.isEmailAddressFormatValid(validEmail));
        }
    }

    @Test
    public void verifyInvalidEmail() {
        List<String> invalidEmails = new LinkedList<String>();
        invalidEmails.add("Abc.example.com");
        invalidEmails.add("Abc.@example.com");
        invalidEmails.add("Abc..123@example.com");
        invalidEmails.add(".test@example.com");
        // JUSTIFICATION: Currently decided on the lenient approach.
        // The regular expression may get too complex to filter out these invalid addresses
        //invalidEmails.add("A@b@c@example.com");
        //invalidEmails.add("a\"b(c)d,e:f;g<h>i[j\\k]l@example.com");
        //invalidEmails.add("just\"not\"right@example.com");
        //invalidEmails.add("this is\"not\\allowed@example.com");
        //invalidEmails.add("this\\ still\"not\\\\allowed@example.com");

        for (String invalidEmail : invalidEmails) {
            assertFalse(ValidatorHelper.isEmailAddressFormatValid(invalidEmail));
        }
    }

}
