package com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

import java.net.URI;

import javax.ws.rs.core.UriBuilder;

import org.junit.Test;

import com.agilex.healthcare.utility.DateHelper;

public class DateFilterFactoryTest {

	private static final String BASEURI = "http://server/resource";

	@Test
	public void createFilterFromStartDate() {
		DateFilter filter = DateFilterFactory.createFilterFromDate("1/1/2000", null);
		assertDateFilter("1/1/2000", null, filter);
	}

	@Test
	public void createFilterFromEndDate() {
		DateFilter filter = DateFilterFactory.createFilterFromDate(null, "1/1/2001");
		assertDateFilter(null, "1/1/2001", filter);
	}

	@Test
	public void createFilterFromStartEndDate() {
		DateFilter filter = DateFilterFactory.createFilterFromDate("1/1/2000", "1/1/2001");
		assertDateFilter("1/1/2000", "1/1/2001", filter);
	}

	@Test
	public void createFilterFromUriWithStartDateLowercase() {
		URI uri = UriBuilder.fromUri(BASEURI).queryParam("startdate", "1/1/2000").build();
		DateFilter filter = DateFilterFactory.createFilterFromUri(uri);
		assertDateFilter("1/1/2000", null, filter);
	}

	@Test
	public void createFilterFromUriWithStartDateMixedCase() {
		URI uri = UriBuilder.fromUri(BASEURI).queryParam("StaRtDate", "1/1/2000").build();
		DateFilter filter = DateFilterFactory.createFilterFromUri(uri);
		assertDateFilter("1/1/2000", null, filter);
	}

	@Test
	public void createFilterFromUriWithDate() {
		URI uri = UriBuilder.fromUri(BASEURI).queryParam("enddate", "1/1/2001").queryParam("startdate", "1/1/2000").build();
		DateFilter filter = DateFilterFactory.createFilterFromUri(uri);
		assertEquals(DateHelper.endOfDay(DateHelper.parseDate("1/1/2001")), filter.getEndDate());
		assertEquals(DateHelper.parseDate("1/1/2000"), filter.getStartDate());
	}

	@Test
	public void createFilterWithStartSlidingScale() {
		URI uri = UriBuilder.fromUri(BASEURI).queryParam("numberOfPastMonths", "1").build();
		DateFilter filter = DateFilterFactory.createFilterFromUri(uri);
		assertTrue(DateHelper.parseDate("3/20/2012").before(filter.getStartDate()));
	}

	@Test
	public void createFilterWithStartEndSlidingScaleAndReferenceDate() {
		URI uri = UriBuilder.fromUri(BASEURI).queryParam("numberOfPastMonths", "1").queryParam("numberOfFutureMonths", "2").queryParam("referenceDate", "3/1/2012").build();
		DateFilter filter = DateFilterFactory.createFilterFromUri(uri);
		assertDateFilter("2/1/2012", "5/1/2012", filter);
	}

	@Test
	public void createFutureFilterWithReferenceDateAndFutureMonths() {
		URI uri = UriBuilder.fromUri(BASEURI).queryParam("numberOfFutureMonths", "2").queryParam("referenceDate", "3/1/2012").build();
		DateFilter filter = DateFilterFactory.createFutureFilterFromUri(uri);
		assertDateFilter("3/1/2012", "5/1/2012", filter);
	}

	@Test
	public void createFutureFilterWithReferenceDateAndPastMonths() {
		URI uri = UriBuilder.fromUri(BASEURI).queryParam("numberOfPastMonths", "1").queryParam("referenceDate", "3/1/2012").build();
		DateFilter filter = DateFilterFactory.createPastFilterFromUri(uri);
		assertDateFilter("2/1/2012", "3/1/2012", filter);
	}

	private void assertDateFilter(String startDate, String endDate, DateFilter filter) {
		assertEquals(DateHelper.parseDate(startDate), filter.getStartDate());
		assertEquals(DateHelper.parseDate(endDate), filter.getEndDate());
	}

}
