package com.agilex.healthcare.mobilehealthplatform.domain;

import static org.junit.Assert.assertEquals;

import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.domain.formatters.AddressDispayFormatter;

public class AddressFormatterTest {

	private Address getFullAddress() {
		Address a = new Address();
		a.setStreetAddressLine1("123 Elm St.");
		a.setStreetAddressLine2("Apt #3");
		a.setStreetAddressLine3("PO BOX 1234");
		a.setCity("Chantilly");
		a.setState("VA");
		a.setZipCode("20165");
		return a;
	}

	private void assertFormatCorrect(String expectedLine1, String expectedLine2, Address a) {
		assertEquals("Incorrect line 1", expectedLine1, a.getFormattedDisplayAddress1());
		assertEquals("Incorrect line 2", expectedLine2, a.getFormattedDisplayAddress2());
	}

	private void runTest(String expectedLine1, String expectedLine2, Address a) {
		AddressDispayFormatter formatter = new AddressDispayFormatter();
		formatter.createFormatDisplay(a);
		assertFormatCorrect(expectedLine1, expectedLine2, a);
	}

	@Test
	public void formatSimpleAddress() {
		String expected1 = "123 Elm St.";
		String expected2 = "Chantilly, VA 20165";

		Address a = new Address();
		a.setStreetAddressLine1("123 Elm St.");
		a.setCity("Chantilly");
		a.setState("VA");
		a.setZipCode("20165");

		runTest(expected1, expected2, a);
	}

	@Test
	public void formatFullAddress() {
		String expected1 = "123 Elm St., Apt #3, PO BOX 1234";
		String expected2 = "Chantilly, VA 20165";
		Address a = getFullAddress();

		runTest(expected1, expected2, a);
	}

	@Test
	public void formatAddressNoAddress3() {
		String expected1 = "123 Elm St., Apt #3";
		String expected2 = "Chantilly, VA 20165";
		Address a = getFullAddress();
		a.setStreetAddressLine3(null);

		runTest(expected1, expected2, a);
	}

	@Test
	public void formatAddressNoAddress2() {
		String expected1 = "123 Elm St., PO BOX 1234";
		String expected2 = "Chantilly, VA 20165";
		Address a = getFullAddress();
		a.setStreetAddressLine2(null);

		runTest(expected1, expected2, a);
	}

	@Test
	public void formatAddressNoLine1() {
		String expected1 = "Apt #3, PO BOX 1234";
		String expected2 = "Chantilly, VA 20165";
		Address a = getFullAddress();
		a.setStreetAddressLine1(null);

		runTest(expected1, expected2, a);
	}

	@Test
	public void formatAddressNoCity() {
		String expected1 = "123 Elm St., Apt #3, PO BOX 1234";
		String expected2 = "VA 20165";
		Address a = getFullAddress();
		a.setCity(null);

		runTest(expected1, expected2, a);
	}

	@Test
	public void formatAddressNoState() {
		String expected1 = "123 Elm St., Apt #3, PO BOX 1234";
		String expected2 = "Chantilly 20165";
		Address a = getFullAddress();
		a.setState(null);

		runTest(expected1, expected2, a);
	}

	@Test
	public void formatFullAddressNoZip() {
		String expected1 = "123 Elm St., Apt #3, PO BOX 1234";
		String expected2 = "Chantilly, VA";
		Address a = getFullAddress();
		a.setZipCode(null);

		runTest(expected1, expected2, a);
	}

	@Test
	public void formatNoStreetInfo() {
		String expected1 = "Chantilly, VA 20151";
		String expected2 = null;
		Address a = new Address();
		a.setCity("Chantilly");
		a.setState("VA");
		a.setZipCode("20151");

		runTest(expected1, expected2, a);
	}

	@Test
	public void formatNoCityStateInfo() {
		String expected1 = "123 Elm St.";
		String expected2 = null;

		Address a = new Address();
		a.setStreetAddressLine1("123 Elm St.");

		runTest(expected1, expected2, a);
	}

}
