package com.agilex.healthcare.mobilehealthplatform.datalayer.vital;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;

import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.domain.GraphData;
import com.agilex.healthcare.mobilehealthplatform.domain.GraphDataSeries;
import com.agilex.healthcare.mobilehealthplatform.domain.VitalEntries;
import com.agilex.healthcare.mobilehealthplatform.domain.VitalEntry;
import com.agilex.healthcare.mobilehealthplatform.domain.VitalObservation;
import com.agilex.healthcare.mobilehealthplatform.domain.code.ObservationTypeCode;
import com.agilex.healthcare.mobilehealthplatform.domain.code.VitalSectionCode;
import com.agilex.healthcare.utility.DateHelper;

public class VitalGraphGeneratorTest {

	private static final double epsilon = 1E-1;

	@Test
	public void convertEmpty() {
		ProviderVitalGraphGenerator graphGenerator = new ProviderVitalGraphGenerator();
		VitalEntries vitals = new VitalEntries();
		GraphData graphData = graphGenerator.createGraphDataBySection(vitals, VitalSectionCode.Height);
		assertNotNull(graphData);
	}

	@Test
	public void convertSimpleDataSet() {
		ProviderVitalGraphGenerator graphGenerator = new ProviderVitalGraphGenerator();

		VitalEntries vitals = createSampleData();

		GraphData graphData = graphGenerator.createGraphDataBySection(vitals, VitalSectionCode.Height);
		assertEquals(1, graphData.getDataSeriesCollection().get(0).size());
		assertEquals(58, graphData.getDataSeriesCollection().get(0).get(0).getY(), epsilon);
		assertEquals(DateHelper.parseDate("12/4/2011"), DateHelper.dateFromSeconds(graphData.getDataSeriesCollection().get(0).get(0).getX()));
	}

	@Test
	public void convertBp() {
		ProviderVitalGraphGenerator graphGenerator = new ProviderVitalGraphGenerator();

		VitalEntries vitals = createSampleData();

		GraphData graphData = graphGenerator.createGraphDataByObservationType(vitals, ObservationTypeCode.INTRAVASCULAR_SYSTOLIC, ObservationTypeCode.INTRAVASCULAR_DIASTOLIC);
		assertEquals(2, graphData.getDataSeriesCollection().size());

		GraphDataSeries dataSeries = graphData.getDataSeriesCollection().get(0);
		assertEquals(3, dataSeries.size());
		assertEquals(130, dataSeries.get(0).getY(), epsilon);
		assertEquals(DateHelper.parseDate("12/1/2011"), DateHelper.dateFromSeconds(dataSeries.get(0).getX()));
		assertEquals("130/80", dataSeries.get(0).getLabel());
		assertEquals(135, dataSeries.getMaxY(), epsilon);
		assertEquals(120, dataSeries.getMinY(), epsilon);
		assertEquals(90, dataSeries.getReference(), epsilon);

		dataSeries = graphData.getDataSeriesCollection().get(1);
		assertEquals(3, dataSeries.size());
		assertEquals(80, dataSeries.get(0).getY(), epsilon);
		assertEquals(DateHelper.parseDate("12/1/2011"), DateHelper.dateFromSeconds(dataSeries.get(0).getX()));
		assertEquals("130/80", dataSeries.get(0).getLabel());
		assertEquals(85, dataSeries.getMaxY(), epsilon);
		assertEquals(75, dataSeries.getMinY(), epsilon);
		assertEquals(140, dataSeries.getReference(), epsilon);
		assertNull(dataSeries.getReferenceHigh());
		assertNull(dataSeries.getReferenceLow());
	}
	
	@Test
	public void convertPulse() {
		ProviderVitalGraphGenerator graphGenerator = new ProviderVitalGraphGenerator();

		VitalEntries vitals = createSampleData();

		GraphData graphData = graphGenerator.createGraphDataByObservationType(vitals, ObservationTypeCode.HEART_BEAT);
		assertEquals(1, graphData.getDataSeriesCollection().size());

		GraphDataSeries dataSeries = graphData.getDataSeriesCollection().get(0);
		assertEquals(3, dataSeries.size());
		assertEquals(60, dataSeries.get(0).getY(), epsilon);
		assertEquals(DateHelper.parseDate("12/1/2011"), DateHelper.dateFromSeconds(dataSeries.get(0).getX()));
		assertEquals("60 beats", dataSeries.get(0).getLabel());
		assertEquals(65, dataSeries.getMaxY(), epsilon);
		assertEquals(55, dataSeries.getMinY(), epsilon);

		assertEquals(100, dataSeries.getReferenceHigh(), epsilon);
		assertEquals(60, dataSeries.getReferenceLow(), epsilon);
		assertNull(dataSeries.getReference());
	}

	private VitalEntries createSampleData() {
		VitalEntries vitalEntries = new VitalEntries();
		String gallowPatientId = "PATID26";

		VitalEntry vitalEntry = new VitalEntry();
		vitalEntry.setSection(VitalSectionCode.BloodPressure);
		vitalEntry.setId("A");
		vitalEntry.setEntryDate(DateHelper.parseDate("12/1/2011"));
		vitalEntry.setQualifier("Standing");
		vitalEntry.setPatientId(gallowPatientId);
		VitalObservation observation = new VitalObservation();
		observation.setType(ObservationTypeCode.HEART_BEAT); // pulse
		observation.setValue("60");
		observation.setValueUnits("beats");
		vitalEntry.getVitalObservations().add(observation);
		observation = new VitalObservation();
		observation.setType(ObservationTypeCode.INTRAVASCULAR_DIASTOLIC); 
		observation.setValue("80");
		observation.setValueUnits("mm(hg)");
		vitalEntry.getVitalObservations().add(observation);
		observation = new VitalObservation();
		observation.setType(ObservationTypeCode.INTRAVASCULAR_SYSTOLIC); 
		observation.setValue("130");
		observation.setValueUnits("mm(hg)");
		vitalEntry.getVitalObservations().add(observation);
		vitalEntry.setNotes("test note");
		vitalEntries.add(vitalEntry);

		vitalEntry = new VitalEntry();
		vitalEntry.setSection(VitalSectionCode.BloodPressure);
		vitalEntry.setId("A");
		vitalEntry.setEntryDate(DateHelper.parseDate("12/1/2011"));
		vitalEntry.setQualifier("Standing");
		vitalEntry.setPatientId(gallowPatientId);
		observation = new VitalObservation();
		observation.setType(ObservationTypeCode.HEART_BEAT); // pulse
		observation.setValue("65");
		observation.setValueUnits("beats");
		vitalEntry.getVitalObservations().add(observation);
		observation = new VitalObservation();
		observation.setType(ObservationTypeCode.INTRAVASCULAR_DIASTOLIC);  
		observation.setValue("85");
		observation.setValueUnits("mm(hg)");
		vitalEntry.getVitalObservations().add(observation);
		observation = new VitalObservation();
		observation.setType(ObservationTypeCode.INTRAVASCULAR_SYSTOLIC);  
		observation.setValue("120");
		observation.setValueUnits("mm(hg)");
		vitalEntry.getVitalObservations().add(observation);
		vitalEntry.setNotes("test note");
		vitalEntries.add(vitalEntry);

		vitalEntry = new VitalEntry();
		vitalEntry.setSection(VitalSectionCode.BloodPressure);
		vitalEntry.setId("A");
		vitalEntry.setEntryDate(DateHelper.parseDate("12/1/2011"));
		vitalEntry.setQualifier("Standing");
		vitalEntry.setPatientId(gallowPatientId);
		observation = new VitalObservation();
		observation.setType(ObservationTypeCode.HEART_BEAT); // pulse
		observation.setValue("55");
		observation.setValueUnits("beats");
		vitalEntry.getVitalObservations().add(observation);
		observation = new VitalObservation();
		observation.setType(ObservationTypeCode.INTRAVASCULAR_DIASTOLIC);  
		observation.setValue("75");
		observation.setValueUnits("mm(hg)");
		vitalEntry.getVitalObservations().add(observation);
		observation = new VitalObservation();
		observation.setType(ObservationTypeCode.INTRAVASCULAR_SYSTOLIC);  
		observation.setValue("135");
		observation.setValueUnits("mm(hg)");
		vitalEntry.getVitalObservations().add(observation);
		vitalEntry.setNotes("test note");
		vitalEntries.add(vitalEntry);

		vitalEntry = new VitalEntry();
		vitalEntry.setId("C");
		vitalEntry.setSection(VitalSectionCode.Height);
		vitalEntry.setEntryDate(DateHelper.parseDate("12/4/2011"));
		vitalEntry.setPatientId(gallowPatientId);
		observation = new VitalObservation();
		observation.setType(ObservationTypeCode.HEIGHT);
		observation.setValue("58");
		observation.setValueUnits("inches");
		vitalEntry.getVitalObservations().add(observation);
		vitalEntries.add(vitalEntry);

		return vitalEntries;
	}

}
