package com.agilex.healthcare.mobilehealthplatform.datalayer.medication;

import java.util.Date;

import org.junit.*;

import com.agilex.healthcare.mobilehealthplatform.domain.Medication;
import com.agilex.healthcare.mobilehealthplatform.domain.code.MedicationRefillCode;
import com.agilex.healthcare.utility.DateHelper;

public class MedicationRefillHelperTest {

	MedicationRefillHelper helper = new MedicationRefillHelper();
	Medication medication;

	@Before
	public void before() {
		medication = new Medication();
		medication.setRefills("1");
	}

	@Test 
	public void verifyMeetsDateRequirementsFor90DayRx() {
		medication.setDaysSupply("90");
		
		medication.setLastFilledDate(DateHelper.minusDays(new Date(), MedicationRefillHelper.NINETY_DAY_REFILL_PERIOD-1));
		Assert.assertTrue(false == helper.meetsDateRequirements(medication));
		
		medication.setLastFilledDate(DateHelper.minusDays(new Date(), MedicationRefillHelper.NINETY_DAY_REFILL_PERIOD));
		Assert.assertTrue(true == helper.meetsDateRequirements(medication));
		
		medication.setLastFilledDate(DateHelper.minusDays(new Date(), MedicationRefillHelper.NINETY_DAY_REFILL_PERIOD+1));
		Assert.assertTrue(true == helper.meetsDateRequirements(medication));
	}

	@Test
	public void verifyMeetsDateRequirementsFor30DayRx() {
		medication.setDaysSupply("30");
		
		medication.setLastFilledDate(DateHelper.minusDays(new Date(), MedicationRefillHelper.THIRTY_DAY_REFILL_PERIOD-1));
		Assert.assertTrue(false == helper.meetsDateRequirements(medication));
		
		medication.setLastFilledDate(DateHelper.minusDays(new Date(), MedicationRefillHelper.THIRTY_DAY_REFILL_PERIOD));
		Assert.assertTrue(true == helper.meetsDateRequirements(medication));
		
		medication.setLastFilledDate(DateHelper.minusDays(new Date(), MedicationRefillHelper.THIRTY_DAY_REFILL_PERIOD+1));
		Assert.assertTrue(true == helper.meetsDateRequirements(medication));
	}

	@Test
	public void verifyNextRefillDateFor90DayRx() {
		medication.setLastFilledDate(DateHelper.parseDate("01/01/2012"));
		medication.setDaysSupply("90");

		int period = 90 - MedicationRefillHelper.NINETY_DAY_REFILL_PERIOD;
		Date expectedDate = DateHelper.plusDays(medication.getLastFilledDate(), period);
		helper.calculateAndSetNextRefillDate(medication);

		Assert.assertTrue(medication.getCalculatedNextFillDate() != null);
		Assert.assertTrue(medication.getCalculatedNextFillDate().equals(expectedDate));
	}

	@Test
	public void verifyNextRefillDateFor30DayRx() {
		medication.setLastFilledDate(DateHelper.parseDate("01/01/2012"));
		medication.setDaysSupply("30");

		int period = 30 - MedicationRefillHelper.THIRTY_DAY_REFILL_PERIOD;
		Date expectedDate = DateHelper.plusDays(medication.getLastFilledDate(), period);
		helper.calculateAndSetNextRefillDate(medication);

		Assert.assertTrue(medication.getCalculatedNextFillDate() != null);
		Assert.assertTrue(medication.getCalculatedNextFillDate().equals(expectedDate));
	}

	@Test
	public void verifyDetermineRefillAction() {
		medication.setDaysSupply("30");
		medication.setLastFilledDate(DateHelper.minusDays(new Date(), MedicationRefillHelper.THIRTY_DAY_REFILL_PERIOD-1));
		helper.determineRefillAction(medication);
		Assert.assertTrue(MedicationRefillCode.REFILL_ACTION_NONE.equals(medication.getRefillAction()));
		
		medication.setLastFilledDate(DateHelper.minusDays(new Date(), MedicationRefillHelper.THIRTY_DAY_REFILL_PERIOD));
		helper.determineRefillAction(medication);
		Assert.assertTrue(MedicationRefillCode.REFILL_ACTION_ELIGIBLE.equals(medication.getRefillAction()));
	}
	
	@Test
	public void validateStatusWithZeroRefills() {
		Medication noRefills = new Medication();
		noRefills.setLastFilledDate(DateHelper.minusDays(new Date(), 30));
		noRefills.setRefills("0");
		
		helper.determineRefillAction(noRefills);
		Assert.assertTrue(MedicationRefillCode.REFILL_ACTION_NONE.equals(noRefills.getRefillAction()));
	}
	
	@Test
	public void validateStatusWithNullRefills() {
		Medication noRefills = new Medication();
		noRefills.setLastFilledDate(DateHelper.minusDays(new Date(), 30));
		
		noRefills.setRefills(null);
		helper.determineRefillAction(noRefills);
		Assert.assertTrue(MedicationRefillCode.REFILL_ACTION_NONE.equals(noRefills.getRefillAction()));
		
		noRefills.setRefills("");
		helper.determineRefillAction(noRefills);
		Assert.assertTrue(MedicationRefillCode.REFILL_ACTION_NONE.equals(noRefills.getRefillAction()));
		
	}
}
