package com.agilex.healthcare.mobilehealthplatform.datalayer.facility;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;

import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.domain.Facilities;
import com.agilex.healthcare.mobilehealthplatform.domain.Facility;

public class FacilityFiltererTest {

	Facilities facilities = new Facilities();
	FacilityFilterer filterer = new FacilityFilterer();
	
	public FacilityFiltererTest() {
		init();
	}
	
	private void init() {
		Facility facility = new Facility();
		facility.setAddress("2615 East      on Ave\nFresno, CA 93703");
		facility.setFacilityCode("570BC");
		facility.setName("Central California HCS");
		facility.setState("CA");
		facility.setType("VAMC");
		facility.setCity("Fresno");
		facility.setParentSiteCode("570");
		facilities.add(facility);
		
		facility = new Facility();
		facility.setAddress("8438 Hello World Way\nFresno, CA 93703");
		facility.setFacilityCode("570AB");
		facility.setName("Hello World California HCS");
		facility.setState("CA");
		facility.setType("VAMC");
		facility.setCity("Fresno");
		facility.setParentSiteCode("570");
		facilities.add(facility);
		
		facility = new Facility();
		facility.setAddress("434 Some Street\nFresno, CA 93703");
		facility.setFacilityCode("680V");
		facility.setName("ACT California HCS");
		facility.setState("VA");
		facility.setType("VAMC");
		facility.setCity("Fresno");
		facility.setParentSiteCode("680");
		facilities.add(facility);
		
		facility = new Facility();
		facility.setAddress("5155 Parkstone Dr\nChantilly, VA 20151");
		facility.setFacilityCode("999");
		facility.setName("Agilex HCS");
		facility.setState("VA");
		facility.setType("VAMC");
		facility.setCity("Chantilly");
		facility.setParentSiteCode("999");
		facilities.add(facility);
	}
	
	@Test
	public void verifyEmptyFilter() {
		FacilityFilter filter = new FacilityFilter();
		
		validateFilter(filter, facilities.size());
	}
	
	@Test
	public void verifyTermFilter() {
		FacilityFilter filter = new FacilityFilter();
		filter.setTerm("CA");
		
		validateFilter(filter, 3);
	}
	
	@Test
	public void verifyCommaDelimitedTermFilter() {
		FacilityFilter filter = new FacilityFilter();
		filter.setTerm("CA Fresno");
		
		validateFilter(filter, 3);
	}
	
	@Test
	public void verifyCaseInsensitiveTermFilter() {
		FacilityFilter filter = new FacilityFilter();
		filter.setTerm("ca");
		
		validateFilter(filter, 3);
	}
	
	@Test
	public void validateWithoutMatch() {
		FacilityFilter filter = new FacilityFilter();
		filter.setTerm("zzzz");
		
		validateFilter(filter, 0);
	}
	
	@Test
	public void validateParentSiteOnly() {
		FacilityFilter filter = new FacilityFilter();
		filter.setParentFacilitiesOnly(true);
		
		validateFilter(filter, 1);
	}
	
	@Test
	public void validateSingleFacilityCode() {
		FacilityFilter filter = new FacilityFilter();
		filter.addFacilityCode("570BC");
		
		validateFilter(filter, 1);
	}

	@Test
	public void validateMultipleFacilityCodeList() {
		FacilityFilter filter = new FacilityFilter();
		filter.addFacilityCode("570BC");
		filter.addFacilityCode("999");
		
		validateFilter(filter, 2);
	}
	
	@Test
	public void validateSingleParentFacilityCode() {
		FacilityFilter filter = new FacilityFilter();
		filter.addParentSiteCode("570");
		
		validateFilter(filter, 2);
	}

	@Test
	public void validateMultipleParentFacilityCodeList() {
		FacilityFilter filter = new FacilityFilter();
		filter.addParentSiteCode("570");
		filter.addParentSiteCode("999");
		
		validateFilter(filter, 3);
	}
	
	@Test
	public void validateFacilityName() {
		FacilityFilter filter = new FacilityFilter();
		filter.setFacilityName("Central California HCS");
		
		validateFilter(filter, 1);
	}
	
	@Test
	public void validateCaseInsensitiveFacilityName() {
		FacilityFilter filter = new FacilityFilter();
		filter.setFacilityName("central california hcs");
		
		validateFilter(filter, 1);
	}
	
	private void validateFilter(FacilityFilter filter, int expectedSize) {
		Facilities filtered = filterer.filter(facilities, filter);
		
		assertEquals(expectedSize, filtered.size());
	}
	
	@Test
	public void testCacheKeys() {
		FacilityFilter filterA = new FacilityFilter();
		FacilityFilter filterB = new FacilityFilter();
		filterA.addFacilityCode("570BC");
		filterA.addFacilityCode("999");

		filterB.addFacilityCode("570BC");
		filterB.addFacilityCode("999");
		filterA.setFacilityName("me");
		
		filterB.setFacilityName("ME".toLowerCase());
		assertTrue(filterA.getKeyForCache().equals(filterB.getKeyForCache()));
		
		String name = "this is me";
		int index = name.indexOf("me");
		filterB.setFacilityName(name.substring(index, index+2));
		assertTrue(filterA.getKeyForCache().equals(filterB.getKeyForCache()));

		filterB.setTerm("abc");
		assertFalse(filterA.getKeyForCache().equals(filterB.getKeyForCache()));
		
		filterB.setTerm("me");
		filterB.addParentSiteCode("parent");
		assertFalse(filterA.getKeyForCache().equals(filterB.getKeyForCache()));
	}
}
