package com.agilex.healthcare.utility.objectmapper;

import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import com.agilex.healthcare.mobilehealthplatform.domain.VitalEntries;
import com.agilex.healthcare.mobilehealthplatform.domain.VitalEntry;
import com.agilex.healthcare.mobilehealthplatform.domain.VitalObservation;
import com.agilex.healthcare.mobilehealthplatform.domain.VitalTableData;
import com.agilex.healthcare.mobilehealthplatform.domain.VitalTableRow;
import com.agilex.healthcare.mobilehealthplatform.domain.code.ObservationTypeCode;
import com.agilex.healthcare.mobilehealthplatform.domain.code.VitalSectionCode;
import com.agilex.healthcare.utility.NullChecker;

public class VitalTableDataGenerator {
	public VitalTableData generateVitalTableData(VitalEntries vitalEntries) {
		VitalTableData tableData = new VitalTableData();

		if (NullChecker.isNullish(vitalEntries)) {
			return tableData;
		}

		Map<Date, VitalEntries> vitalsByDate = getMostRecentVitalsByDate(vitalEntries);

		Set<Date> datesWithVitals = vitalsByDate.keySet();

		for (Date date : datesWithVitals) {
			VitalEntries vitalEntriesForDate = vitalsByDate.get(date);
			VitalTableRow tableRow = createVitalTableRow(vitalEntriesForDate);
			tableRow.setEntryTime(date);
			tableData.add(tableRow);
		}

		return tableData;
	}

	private VitalTableRow createVitalTableRow(VitalEntries vitalEntries) {
		VitalTableRow tableRow = new VitalTableRow();

		for (VitalEntry vitalEntry : vitalEntries) {
			tableRow.setPatientIdentifier(vitalEntry.getPatientIdentifier());

			String section = vitalEntry.getSection();

			if (VitalSectionCode.BloodPressure.equalsIgnoreCase(section)) {
				tableRow = setBloodPressure(vitalEntry, tableRow);
			} else if (VitalSectionCode.Temperature.equalsIgnoreCase(section)) {
				tableRow = setTemperature(vitalEntry, tableRow);
			} else if (VitalSectionCode.Respiration.equalsIgnoreCase(section)) {
				tableRow = setRespiration(vitalEntry, tableRow);
			} else if (VitalSectionCode.Weight.equalsIgnoreCase(section)) {
				tableRow = setWeight(vitalEntry, tableRow);
			} else if (VitalSectionCode.Height.equalsIgnoreCase(section)) {
				tableRow = setHeight(vitalEntry, tableRow);
			} else if (VitalSectionCode.Pain.equalsIgnoreCase(section)) {
				tableRow = setPain(vitalEntry, tableRow);
			} else if (VitalSectionCode.PulseOximetry.equalsIgnoreCase(section)) {
				tableRow = setPulseOximetry(vitalEntry, tableRow);
			} else if (VitalSectionCode.Pulse.equalsIgnoreCase(section)) {
				tableRow = setPulse(vitalEntry, tableRow);
			}
		}

		return tableRow;
	}

	private VitalTableRow setPulseOximetry(VitalEntry vitalEntry, VitalTableRow tableRow) {
		VitalObservation vitalObservation = vitalEntry.getObservationByType(ObservationTypeCode.PULSE_OXIMETRY);
        if(vitalObservation != null)
		    tableRow.setPulseOximetry(vitalObservation.getValue());
		return tableRow;
	}

	private VitalTableRow setPain(VitalEntry vitalEntry, VitalTableRow tableRow) {
		VitalObservation vitalObservation = vitalEntry.getObservationByType(ObservationTypeCode.PAIN);
        if(vitalObservation != null)
		    tableRow.setPain(vitalObservation.getValue());
		return tableRow;
	}

	private VitalTableRow setWeight(VitalEntry vitalEntry, VitalTableRow tableRow) {
		VitalObservation vitalObservation = vitalEntry.getObservationByType(ObservationTypeCode.WEIGHT);
        if(vitalObservation != null)
		    tableRow.setWeight(vitalObservation.getValue());
		return tableRow;
	}

	private VitalTableRow setHeight(VitalEntry vitalEntry, VitalTableRow tableRow) {
		VitalObservation vitalObservation = vitalEntry.getObservationByType(ObservationTypeCode.HEIGHT);
        if(vitalObservation != null)
            tableRow.setHeight(vitalObservation.getValue());
		return tableRow;
	}

	private VitalTableRow setRespiration(VitalEntry vitalEntry, VitalTableRow tableRow) {
		VitalObservation vitalObservation = vitalEntry.getObservationByType(ObservationTypeCode.RESPIRATION);
        if(vitalObservation != null)
		    tableRow.setRespiration(vitalObservation.getValue());
		return tableRow;
	}

	private VitalTableRow setTemperature(VitalEntry vitalEntry, VitalTableRow tableRow) {
		VitalObservation vitalObservation = vitalEntry.getObservationByType(ObservationTypeCode.BODY_TEMPERATURE);
        if(vitalObservation != null)
            tableRow.setTemperature(vitalObservation.getValue());
		return tableRow;
	}

	private VitalTableRow setBloodPressure(VitalEntry vitalEntry, VitalTableRow tableRow) {
		VitalObservation vitalObservation = vitalEntry.getObservationByType(ObservationTypeCode.INTRAVASCULAR_SYSTOLIC);
        if(vitalObservation != null)
		    tableRow.setSystolic(vitalObservation.getValue());

		vitalObservation = vitalEntry.getObservationByType(ObservationTypeCode.INTRAVASCULAR_DIASTOLIC);
        if(vitalObservation != null)
		    tableRow.setDiastolic(vitalObservation.getValue());

		vitalObservation = vitalEntry.getObservationByType(ObservationTypeCode.HEART_BEAT);
		if (vitalObservation != null)
			tableRow.setPulse(vitalObservation.getValue());

		return tableRow;
	}
	
	private VitalTableRow setPulse(VitalEntry vitalEntry, VitalTableRow tableRow) {
		VitalObservation vitalObservation = vitalEntry.getObservationByType(ObservationTypeCode.HEART_BEAT);
        if(vitalObservation != null)
		    tableRow.setPulse(vitalObservation.getValue());
		return tableRow;
	}

	private Map<Date, VitalEntries> getMostRecentVitalsByDate(VitalEntries vitalEntries) {
		Map<Date, VitalEntries> mostRecentVitals = new HashMap<Date, VitalEntries>();

		for (VitalEntry vitalEntry : vitalEntries) {

			Date date = vitalEntry.getEntryDate() ;
			if( date == null ) { 
				
				continue ; 
			}

			VitalEntries vitalEntriesByDate = mostRecentVitals.get(date);
			if (vitalEntriesByDate == null) {
				vitalEntriesByDate = new VitalEntries();
				vitalEntriesByDate.add(vitalEntry);
				mostRecentVitals.put(date, vitalEntriesByDate);
			} else {
				
				VitalEntry vitalEntryByDate = getVitalEntryBySection(vitalEntriesByDate, vitalEntry);
				vitalEntriesByDate.add(vitalEntry);
				if (vitalEntryByDate == null) {
					vitalEntriesByDate.add(vitalEntry);
				} 
				else if (vitalEntry.getEntryDate().compareTo(vitalEntryByDate.getEntryDate()) > 0) {
					vitalEntriesByDate.remove(vitalEntryByDate);
					vitalEntriesByDate.add(vitalEntry);
				}
			}
		}

		return mostRecentVitals;
	}

	private VitalEntry getVitalEntryBySection(VitalEntries vitalEntriesByDate, VitalEntry vitalEntry) {

		if (NullChecker.isNotNullish(vitalEntry.getSection())) {
			for (VitalEntry entry : vitalEntriesByDate) {
				if (vitalEntry.getSection().equalsIgnoreCase(entry.getSection())) {
					return entry;
				}
			}
		}

		return null;
	}

}
