package com.agilex.healthcare.utility;

import java.io.IOException;
import java.io.StringReader;

import javax.xml.XMLConstants;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

/**
 * Supports the mapping of XML to an object. This is designed to be 
 * extended to create custom XML mappings.
 * 
 */
public class XmlMapper {
	
	private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(XmlMapper.class);

	protected NodeList nodeList;
	
	/**
	 * Given an xml document as a string, this will build a {@link Document} and 
	 * instantiate a {@link NodeList} using the baseTagName provided.
	 * 
	 */
	public XmlMapper(String responseXml, String baseTagName) {
		
        Document doc = null;

        try {
            DocumentBuilder db = getDocumentBuilder();
            doc = getDocument(responseXml, db);
            nodeList = doc.getElementsByTagName(baseTagName);
        } catch (Exception e) {
            logger.error("Unable to parse and map response XML received from external system", e);
            throw new RuntimeException("Unable to parse the response xml while mapping.", e);
        }
	}
	
	
	private Document getDocument(String responseXML, DocumentBuilder db) throws SAXException, IOException {
        InputSource inputStream = new InputSource();
        inputStream.setCharacterStream(new StringReader(responseXML));

        return db.parse(inputStream);
    }
	
	private DocumentBuilder getDocumentBuilder() throws ParserConfigurationException {
        DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
        dbf.setNamespaceAware(true);
        try {
            dbf.setFeature("http://xml.org/sax/features/external-parameter-entities", false);
            dbf.setFeature("http://xml.org/sax/features/external-general-entities", false);
        } 
        catch (ParserConfigurationException e) {
            System.err.println("could not set parser feature");
        }
        dbf.setFeature(XMLConstants.FEATURE_SECURE_PROCESSING, true);
        return dbf.newDocumentBuilder();
    }
	
	/**
	 * Given an ordered array of nodeNames, this traverses the child elements 
	 * of each the node and return to return a node's value.
	 * 
	 * @param node
	 * @param nodeNames
	 * @return
	 */
	protected String getNodeValue(Node node, String[] nodeNames) {
        if(node.getNodeType() != Node.ELEMENT_NODE)
            return null;

        Element element = (Element) node;

        for (String nodeName : nodeNames) {
            try {
                element = getNextElement(element, nodeName);
            } catch (Exception ex) {
                //Element not found
                return null;
            }
        }

        return getNodeValue(element);
    }
	
	protected Element getNextElement(Element element, String nodeName) {
        NodeList valueNode = element.getElementsByTagName(nodeName);
        Element valueElement = (Element) valueNode.item(0);

        return  (Element) valueElement.getChildNodes();
    }
	
	
	/**
	 * Returns the text (with whitespace trimmed) of a single element.
	 * 
	 * @param element
	 * @return
	 */
	protected String getNodeValue(Element element) {
        try {
            NodeList textNode = element.getChildNodes();
            return ((Node)textNode.item(0)).getNodeValue().trim();
        } catch (Exception ex) {
            //Text not found
            return null;
        }
    }
}
