package com.agilex.healthcare.mobilehealthplatform.validator.vitals;

import java.util.List;

import com.agilex.healthcare.mobilehealthplatform.domain.ValidationError;
import com.agilex.healthcare.mobilehealthplatform.domain.ValidationResult;
import com.agilex.healthcare.mobilehealthplatform.domain.VitalEntry;
import com.agilex.healthcare.mobilehealthplatform.domain.VitalObservation;
import com.agilex.healthcare.mobilehealthplatform.validator.ValidatorHelper;
import com.agilex.healthcare.utility.NullChecker;

public class VitalWeightValidator extends VitalValidator {

	private static final double WEIGHT_LOWER_BOUND = 0;
	private static final double WEIGHT_UPPER_BOUND = 999.9;

	private static final String VitalType = "Weight";
	private static final String DATE_FIELD = "Date";
	private static final String WEIGHT_NULL = "Weight Vital Entry cannot be null";
	private static final String DATE_NULL_ERROR = "Date must be provided before saving";
	private static final String WEIGHT_FIELD_NULL = "Weight must be provided before saving";
	private static final String WEIGHT_BOUNDS_ERROR = String.format("Weight must be between %s and %s ", WEIGHT_LOWER_BOUND, WEIGHT_UPPER_BOUND);

	@Override
	protected String getVitalType() {
		return VitalType;
	}

	@Override
	protected void performSpecificValidation(ValidationResult<VitalEntry> validationResult, VitalEntry vitalEntry) {
		if (vitalEntry == null) {
			validationResult.addIfNotNull(ValidatorHelper.validateIsNotNull(vitalEntry, new ValidationError(WEIGHT_NULL,VitalType)));
		} else {
			
			List<VitalObservation> observationsToVerify = vitalEntry.getVitalObservations();
			for (VitalObservation observation : observationsToVerify) {
				ValidationError weightExistsValidation = ValidatorHelper.validateIsNotBlankOrNull(observation.getValue(), new ValidationError(WEIGHT_FIELD_NULL, VitalType));
				validationResult.addIfNotNull(weightExistsValidation);
				
				if (isWeightProvided(observation)) {
					ValidationError weightRangeValidation = ValidatorHelper.validateInclusiveNumericBounds(observation.getValueAsDouble(), WEIGHT_UPPER_BOUND, WEIGHT_LOWER_BOUND, new ValidationError(WEIGHT_BOUNDS_ERROR, VitalType));
					validationResult.addIfNotNull(weightRangeValidation);
				}
			}
			ValidationError dateExistsValidation = ValidatorHelper.validateDateIsNotNull(vitalEntry.getEntryDate(), new ValidationError(DATE_FIELD, DATE_NULL_ERROR));
			validationResult.addIfNotNull(dateExistsValidation);
		}
	}
	
	private boolean isWeightProvided(VitalObservation observation) {
		return NullChecker.isNotNullish(observation.getValue());
	}
}
