package com.agilex.healthcare.mobilehealthplatform.validator.vitals;

import java.util.List;

import com.agilex.healthcare.mobilehealthplatform.domain.ValidationError;
import com.agilex.healthcare.mobilehealthplatform.domain.ValidationResult;
import com.agilex.healthcare.mobilehealthplatform.domain.VitalEntry;
import com.agilex.healthcare.mobilehealthplatform.domain.VitalObservation;
import com.agilex.healthcare.mobilehealthplatform.validator.ValidatorHelper;

public class VitalHeightValidator extends VitalValidator {

	private static final int HEIGHT_MAX = 96;
	private static final int HEIGHT_MIN = 0;

	private static final String VitalType = "Height";

	private static final String HEIGHT_EMPTY_ERROR = "Height must be provided before saving";
	private static final String HEIGHT_BOUNDS_ERROR = String.format("Height must be between %s and %s inches", HEIGHT_MIN, HEIGHT_MAX);
	private static final String HEIGHT_NULL_ERROR = "Height Vital Entry cannot be null";

	@Override
	protected String getVitalType() {
		return VitalType;
	}

	@Override
	protected void performSpecificValidation(ValidationResult<VitalEntry> validationResult, VitalEntry vitalEntry) {
		if (vitalEntry == null) {
			validationResult.addIfNotNull(ValidatorHelper.validateIsNotNull(vitalEntry, new ValidationError(HEIGHT_NULL_ERROR, VitalType)));
		} else {
			List<VitalObservation> observationsToVerify = vitalEntry.getVitalObservations();
			for (VitalObservation observation : observationsToVerify) {
				validationResult.addIfNotNull(ValidatorHelper.validateIsNotBlankOrNull(observation.getValue(), new ValidationError(HEIGHT_EMPTY_ERROR, VitalType)));
				if (isHeightProvided(observation)) {
					Integer valueAsInteger = observation.getValueAsIntegerByType();
					if (valueAsInteger != null) {
						validationResult.addIfNotNull(ValidatorHelper.validateInclusiveNumericBounds(observation.getValueAsIntegerByType(), HEIGHT_MAX, HEIGHT_MIN, new ValidationError(HEIGHT_BOUNDS_ERROR, VitalType)));
					}
				}
			}
		}
	}

	private boolean isHeightProvided(VitalObservation observation) {
		return observation.getValue() != null;
	}
}
