package com.agilex.healthcare.mobilehealthplatform.validator.preference;

import com.agilex.healthcare.mobilehealthplatform.domain.PatientPreference;
import com.agilex.healthcare.mobilehealthplatform.domain.ValidationError;
import com.agilex.healthcare.mobilehealthplatform.domain.ValidationResult;
import com.agilex.healthcare.mobilehealthplatform.validator.ValidatorHelper;
import com.agilex.healthcare.utility.NullChecker;

import java.util.Arrays;
import java.util.List;

public class PreferenceValidator {
	
	public ValidationResult<PatientPreference> validate(PatientPreference pref) {
		
		ValidationResult<PatientPreference> validationResult = new ValidationResult<PatientPreference>();
		
		validateEmail(pref, validationResult);
		
		validatePhNumber(pref, validationResult);

        validatePreferenceFrequency(pref, validationResult);
		
		return validationResult;
	}

    private void validatePreferenceFrequency(PatientPreference pref, ValidationResult<PatientPreference> validationResult) {
        String[] validValues = {"Each new message", "Daily", "Never"};
        List<String> validValuesList =  Arrays.asList(validValues);
        if (!validValuesList.contains(pref.getNotificationFrequency())){
             validationResult.addIfNotNull(new ValidationError("Invalid Notification Frequency", "notificationFrequency"));
        }
    }

    private void validatePhNumber(PatientPreference pref,
			ValidationResult<PatientPreference> validationResult) {
		String phoneNumber = pref.getTextMsgPhNumber();
        if(NullChecker.isNotNullish(phoneNumber)){
            if (!ValidatorHelper.isPhoneNumberLengthValid(phoneNumber) ||
                    !ValidatorHelper.isPhoneNumberHasMinDigits(phoneNumber) ||
                    !ValidatorHelper.isPhoneNumberValidCharacters(phoneNumber)){

                validationResult.addIfNotNull(new ValidationError("Phone must have a minimum of 10 digits and only contain digits and these characters: + ( ) - . x space", "textMsgPhNumber"));
            }
        }
	}

	private void validateEmail(PatientPreference pref,
			ValidationResult<PatientPreference> validationResult) {
		validationResult.setRequestObject(pref);
		String email = pref.getEmailAddress();
        if(NullChecker.isNotNullish(email)){

            if (!ValidatorHelper.isEmailLengthValid(email)
                    || !ValidatorHelper.isEmailAddressFormatValid(email)) {
                validationResult
                        .addIfNotNull(new ValidationError(
                                "Email must be between 3 and 50 characters in a valid email format",
                                "emailAddress"));
            }
        }
    }
}
