package com.agilex.healthcare.mobilehealthplatform.validator.diet;

import java.util.Date;

import com.agilex.healthcare.mobilehealthplatform.domain.DietEntry;
import com.agilex.healthcare.mobilehealthplatform.domain.ValidationError;
import com.agilex.healthcare.mobilehealthplatform.validator.ValidatorHelper;
import com.agilex.healthcare.utility.NullChecker;
import com.agilex.healthcare.utility.NullSafeStringComparer;

public class DietValidatorHelper {

	public static final String ENTRYDATE_FIELD = "Entry Date";

	private static final double MINIMUM_VALUE = 0;
	private static final double MAXIMUM_VALUE = 9999;

	public static ValidationError validateEntryDate(Date entryDate) {
		ValidationError error = null;
		if (entryDate == null) {
			error = new ValidationError();
			error.setFieldName(ENTRYDATE_FIELD);
			error.setErrorMessage("Entry date is required.");
		}
		return error;
	}

	public static ValidationError validateMealEntryValueWithinBounds(DietEntry mealEntry, String fieldName) {
		ValidationError error = null;
		String value = getFieldValueFromMealEntry(mealEntry, fieldName);

		if (mealEntry == null) {
			error = new ValidationError("Meal is required.", fieldName);
		} else {
			try {
				if (NullChecker.isNotNullish(value)) {
					Double doubleValue = Double.parseDouble(value);
					error = new ValidationError(String.format("%s field must be between %s and %s.", fieldName, MINIMUM_VALUE, MAXIMUM_VALUE), fieldName);
					return ValidatorHelper.validateInclusiveNumericBounds(doubleValue, MAXIMUM_VALUE, MINIMUM_VALUE, error);
				}
			} catch (NumberFormatException ex) {
				error = new ValidationError(String.format("%s field must be numeric", fieldName), fieldName);
			}
		}
		return error;
	}

	private static String getFieldValueFromMealEntry(DietEntry mealEntry, String fieldName) {
		String fieldValue = "";
		if (NullSafeStringComparer.areEqual(DietValidator.FAT_FIELDNAME, fieldName)) {
			fieldValue = mealEntry.getFat();
		} else if (NullSafeStringComparer.areEqual(DietValidator.CARBS_FIELDNAME, fieldName)) {
			fieldValue = mealEntry.getCarbs();
		} else if (NullSafeStringComparer.areEqual(DietValidator.CALORIES_FIELDNAME, fieldName)) {
			fieldValue = mealEntry.getCalories();
		} else if (NullSafeStringComparer.areEqual(DietValidator.PROTEIN_FIELDNAME, fieldName)) {
			fieldValue = mealEntry.getProtein();
		}

		return fieldValue;
	}

	public static ValidationError validateNotNull(DietEntry mealEntry, String entryType) {
		ValidationError error = null;
		if (mealEntry == null) {
			error = new ValidationError(String.format("%s object is required.", entryType), entryType);
		}
		return error;
	}
}
