package com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter;

import java.net.URI;
import java.util.Date;

import javax.ws.rs.core.UriInfo;

import org.joda.time.DateTime;

import com.agilex.healthcare.mobilehealthplatform.utils.QueryParameters;
import com.agilex.healthcare.utility.DateHelper;

public class DateFilterFactory {
	private static final org.apache.commons.logging.Log LOGGER = org.apache.commons.logging.LogFactory.getLog(DateFilterFactory.class);

	public static DateFilter createEmptyFilter() {
		return new DateFilterImpl();
	}

	public static DateFilter createFilterFromUri(URI requestUri) {
		QueryParameters queryParameters = new QueryParameters(requestUri.getQuery());
		DateFilter filter = createFilter(queryParameters, null, null, null);
		LOGGER.debug("created filter from uri");
		return filter;
	}

	public static DateFilter createFilterFromUri(UriInfo uriInfo) {
		return createFilterFromUri(uriInfo.getRequestUri());
	}

	public static DateFilter createPastFilterFromUri(URI u) {
		QueryParameters queryParameters = new QueryParameters(u.getQuery());
		DateFilter filter = createFilter(queryParameters, null, null, determineReferenceDate(queryParameters).toDate());
		LOGGER.debug("created filter from uri");
		return filter;
	}

	public static DateFilter createFutureFilterFromUri(URI u) {
		QueryParameters queryParameters = new QueryParameters(u.getQuery());

		DateFilter filter = createFilter(queryParameters, null, determineReferenceDate(queryParameters).toDate(), null);
		LOGGER.debug("created filter from uri");
		return filter;
	}

	public static DateFilter createFutureFilterNext30DaysFromUri(URI u) {
		QueryParameters queryParameters = new QueryParameters(u.getQuery());

		DateFilter filter = createFilter(queryParameters, null, determineReferenceDate(queryParameters).toDate(), DateHelper.get30DaysFromNow());
		LOGGER.debug("created filter from uri 30 days");
		return filter;
	}

	public static DateFilter createFutureFilterNext120DaysFromUri(URI u) {
		QueryParameters queryParameters = new QueryParameters(u.getQuery());

		DateFilter filter = createFilter(queryParameters, null, determineReferenceDate(queryParameters).toDate(), DateHelper.get120DaysFromNow());
		LOGGER.debug("created filter from uri 120 days");
		return filter;
	}
	
	public static DateFilter createCalendarApplicationDateFilter(URI u) {
		QueryParameters queryParameters = new QueryParameters(u.getQuery());
		DateFilter filter = createFilter(queryParameters, null, DateHelper.getYesterday(), DateHelper.get3MonthsFromNow());
		LOGGER.debug("created filter from calapp uri ");
		return filter;
	}

	private static DateTime determineReferenceDate(QueryParameters parameters) {
		return determineReferenceDate(new DateFilterQueryParameters(parameters));

	}

	private static DateTime determineReferenceDate(DateFilterQueryParameters parameters) {
		Date referenceDate;
		if (parameters.getReferenceDate().isNotNullish()) {
			referenceDate = parameters.getReferenceDate().asDate();
		} else {
			referenceDate = DateHelper.getToday();
		}
		DateTime referenceDateTime = new DateTime(referenceDate);
		return referenceDateTime;
	}

	private static DateFilter createFilter(QueryParameters queryParameters, Date defaultEntryDate, Date defaultStartDate, Date defaultEndDate) {
		DateFilterQueryParameters parameters = new DateFilterQueryParameters(queryParameters);

		DateTime referenceDate = determineReferenceDate(parameters);
		Date entryDate = determineEntryDate(defaultEntryDate, parameters);
		Date startDate = determineStartDate(defaultStartDate, parameters, referenceDate);
		Date endDate = determineEndDate(defaultEndDate, parameters, referenceDate);

		DateFilter filter;
		if (entryDate != null) {
			filter = createFilterFromDate(parameters.getEntryDate().asDate());
		} else if (startDate != null || endDate != null) {
			filter = createFilterFromDate(startDate, endDate);
		} else {
			filter = createEmptyFilter();
		}

		return filter;
	}

	private static Date determineEndDate(Date defaultEndDate, DateFilterQueryParameters parameters, DateTime referenceDate) {
		Date endDate;
		if (parameters.getEndDate().isNotNullish()) {
			endDate = parameters.getEndDate().asDate();
			endDate = DateHelper.endOfDay(endDate);
		} else if (parameters.getSlidingScaleMonthsInFuture().isNotNullish()) {
			endDate = referenceDate.plusMonths(parameters.getSlidingScaleMonthsInFuture().asInteger()).toDate();
		} else {
			endDate = defaultEndDate;
		}
		return endDate;
	}

	private static Date determineStartDate(Date defaultStartDate, DateFilterQueryParameters parameters, DateTime referenceDate) {
		Date startDate;
		if (parameters.getStartDate().isNotNullish()) {
			startDate = parameters.getStartDate().asDate();
		} else if (parameters.getSlidingScaleMonthsInPast().isNotNullish()) {
			startDate = referenceDate.minusMonths(parameters.getSlidingScaleMonthsInPast().asInteger()).toDate();
		} else {
			startDate = defaultStartDate;
		}
		return startDate;
	}

	private static Date determineEntryDate(Date defaultEntryDate, DateFilterQueryParameters parameters) {
		Date entryDate;
		if (parameters.getEntryDate().isNotNullish()) {
			entryDate = parameters.getEntryDate().asDate();
		} else {
			entryDate = defaultEntryDate;
		}
		return entryDate;
	}

	public static DateFilter createFilterFromDate(Date startDate, Date endDate) {
		DateFilterImpl filter = new DateFilterImpl();
		filter.setStartDate(startDate);
		filter.setEndDate(endDate);

		return filter;
	}

	public static DateFilter createFilterFromDate(String startDate, String endDate) {
		return createFilterFromDate(DateHelper.parseDate(startDate), DateHelper.parseDate(endDate));
	}

	public static DateFilter createFilterFromDate(Date filterDate) {
		Date startDate = filterDate;
		Date endDate = DateHelper.endOfDay(startDate);
		return createFilterFromDate(startDate, endDate);
	}

	public static DateFilter createFilterFromToday() {
		Date today = new Date();
		Date beginningOfDay = DateHelper.beginOfDate(today);
		Date endOfDay = DateHelper.endOfDay(beginningOfDay);
		return createFilterFromDate(beginningOfDay, endOfDay);
	}

	public static DateFilter createFilterFromDate(String filterDate) {
		return createFilterFromDate(DateHelper.parseDate(filterDate));
	}

}
