package com.agilex.healthcare.mobilehealthplatform.datalayer.vital;

import java.util.ArrayList;
import java.util.Collection;

import com.agilex.healthcare.mobilehealthplatform.domain.GraphData;
import com.agilex.healthcare.mobilehealthplatform.domain.GraphDataSeries;
import com.agilex.healthcare.mobilehealthplatform.domain.GraphItem;
import com.agilex.healthcare.mobilehealthplatform.domain.VitalEntries;
import com.agilex.healthcare.mobilehealthplatform.domain.VitalEntry;
import com.agilex.healthcare.mobilehealthplatform.domain.VitalObservation;
import com.agilex.healthcare.mobilehealthplatform.domain.code.ObservationTypeCode;
import com.agilex.healthcare.mobilehealthplatform.domain.code.VitalSectionCode;
import com.agilex.healthcare.utility.DateHelper;
import com.agilex.healthcare.utility.NullChecker;

public abstract class VitalGraphGenerator {

	private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(VitalGraphGenerator.class);
	
	public abstract GraphData createGraphDataBySection(VitalEntries vitals, String graphType);
	
	public GraphData createGraphDataByObservationType(VitalEntries vitals, String... observationTypeToGraphArray) {
		ArrayList<String> observationTypesCollection = new ArrayList<String>();
		for (String observationType : observationTypeToGraphArray) {
			observationTypesCollection.add(observationType);
		}
		return createGraphDataByObservationType(vitals, observationTypesCollection);
	}

	public GraphData createGraphDataByObservationType(VitalEntries vitals, Collection<String> observationTypeToGraphArray) {
		return createGraphDataByObservationType(vitals, observationTypeToGraphArray, "Default");
	}

	public GraphData createGraphDataByObservationType(VitalEntries vitals, Collection<String> observationTypeToGraphArray, String graphType) {
		GraphData graphData = new GraphData();

		for (String observationTypeToGraph : observationTypeToGraphArray) {
			GraphDataSeries dataSeries = new GraphDataSeries();
			dataSeries.setTitle(observationTypeToGraph);

			for (VitalEntry vitalEntry : vitals) {
				for (VitalObservation vitalObservation : vitalEntry.getVitalObservations()) {
					if (vitalObservation.getType().contentEquals(observationTypeToGraph)) {
						GraphItem item = buildGraphItem(vitalEntry, vitalObservation);
                        if(item != null) {
                            dataSeries.add(item);
                            updateMinMax(dataSeries, item);
                        }
					}
				}
			}

			if (ObservationTypeCode.HEART_BEAT.equals(observationTypeToGraph)) {
				dataSeries.setReferenceLow(60);
				dataSeries.setReferenceHigh(100);
			} else if (ObservationTypeCode.INTRAVASCULAR_DIASTOLIC.equals(observationTypeToGraph)) {
				dataSeries.setReference(140);
			} else if (ObservationTypeCode.INTRAVASCULAR_SYSTOLIC.equals(observationTypeToGraph)) {
				dataSeries.setReference(90);
			}
			
			graphData.getDataSeriesCollection().add(dataSeries);
		}
		graphData.setGraphTitle(VitalSectionCode.getTitle(graphType));

		return graphData;
	}
	
	private GraphItem buildGraphItem(VitalEntry vitalEntry, VitalObservation vitalObservation) {
        Double valueAsDouble = vitalObservation.getValueAsDouble();
        if(valueAsDouble == null) // The result was unparseable or null, exclude
            return null;

        GraphItem item = new GraphItem();
		item.setX(DateHelper.dateToSeconds(vitalEntry.getEntryDate()));

        item.setY(valueAsDouble);

		String escapedString = vitalObservation.getValueUnits() == null ? "" : vitalObservation.getValueUnits().replaceAll("%","%%");
		item.setUnits(String.format(escapedString));

		item.setDate(vitalEntry.getEntryDate());
		item.setLabel(buildLabel(vitalEntry, vitalObservation));

		logger.debug("processing data as graph item ");
		return item;
	}
	
	private String buildLabel(VitalEntry vitalEntry, VitalObservation observation) {
		String label;
		if (isBloodPressureObservation(vitalEntry, observation)) {
			String systolic = vitalEntry.getObservationValueByType(ObservationTypeCode.INTRAVASCULAR_SYSTOLIC);
			String diastolic = vitalEntry.getObservationValueByType(ObservationTypeCode.INTRAVASCULAR_DIASTOLIC);
			label = String.format("%s/%s", systolic, diastolic);
		} else {
			label = observation.getValue();
			if(NullChecker.isNotNullish(observation.getValueUnits()))
				label += String.format(" %s", observation.getValueUnits());
		}
		return label;
	}

	private boolean isBloodPressureObservation(VitalEntry vitalEntry, VitalObservation observation) {
		return vitalEntry.getSection().contentEquals(VitalSectionCode.BloodPressure) && (observation.getType().contentEquals(ObservationTypeCode.INTRAVASCULAR_DIASTOLIC) || observation.getType().contentEquals(ObservationTypeCode.INTRAVASCULAR_SYSTOLIC));
	}

	private void updateMinMax(GraphDataSeries dataSeries, GraphItem item) {
		dataSeries.setMaxX(Math.max(dataSeries.getMaxX(), item.getX()));
		dataSeries.setMaxY(Math.max(dataSeries.getMaxY(), item.getY()));
		if (dataSeries.size() == 1) {
			dataSeries.setMinX(item.getX());
			dataSeries.setMinY(item.getY());
		} else {
			dataSeries.setMinX(Math.min(dataSeries.getMinX(), item.getX()));
			dataSeries.setMinY(Math.min(dataSeries.getMinY(), item.getY()));
		}
	}
}
