package com.agilex.healthcare.mobilehealthplatform.datalayer.paindiary;

import java.util.Date;
import java.util.HashMap;
import java.util.Map;

import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentProperty;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentQuestion;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentQuestionChoice;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentQuestionChoices;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentResponse;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentResult;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentResults;
import com.agilex.healthcare.mobilehealthplatform.domain.GraphData;
import com.agilex.healthcare.mobilehealthplatform.domain.GraphDataSeries;
import com.agilex.healthcare.mobilehealthplatform.domain.GraphItem;
import com.agilex.healthcare.mobilehealthplatform.domain.PainDiaryEntries;
import com.agilex.healthcare.mobilehealthplatform.domain.PainDiaryEntry;
import com.agilex.healthcare.mobilehealthplatform.domain.PainDiaryReport;
import com.agilex.healthcare.mobilehealthplatform.domain.code.AssessmentCode;
import com.agilex.healthcare.utility.DateHelper;

public class PainSummaryGraphGenerator {

	public GraphData createGraphData(PainDiaryEntries painDiaryEntries, AssessmentResults assessmentResults) {
		GraphData graphData = new GraphData();

        painDiaryEntries = filterPainDiaryEntries(painDiaryEntries);

		GraphDataSeries dataSeries = createPainDiaryDataSeries(painDiaryEntries);
		graphData.getDataSeriesCollection().add(dataSeries);
		
		dataSeries = createPainAssessmentDataSeries(assessmentResults);
		graphData.getDataSeriesCollection().add(dataSeries);
		
		graphData.setGraphTitle("Pain Management Severity over Time");

		return graphData;
	}

    private PainDiaryEntries filterPainDiaryEntries(PainDiaryEntries painDiaryEntries) {
        //only return the entry with latest pain value on any given day

        Map<Date, PainDiaryEntry> entriesByDate = new HashMap<Date, PainDiaryEntry>();

        for (PainDiaryEntry painDiaryEntry : painDiaryEntries) {
            PainDiaryReport diaryReport = painDiaryEntry.getPainQuestionValues();
            String painIntensity = diaryReport.getAveragePainIntensity();

            //we do not want entries with no pain Intensity
            if (painIntensity != null && !"".equals(painIntensity)) {
              Date entryDate = painDiaryEntry.getDateCreated();
              Date entryDateMidnight = DateHelper.beginOfDate(entryDate);

    //            //Get the date component (without time component)
                if (entriesByDate.containsKey(entryDateMidnight)){
                    PainDiaryEntry existingEntry = entriesByDate.get(entryDateMidnight);
                    if (entryDate.after(existingEntry.getDateCreated())){
                        entriesByDate.remove(entryDateMidnight);
                        entriesByDate.put(entryDateMidnight, painDiaryEntry);
                    }
                    else{
                        //do nothing, leave existing item
                    }
    //            if entriesByDate already contains the date, compare
    //                keep the latter entry


                }
                else{
                    entriesByDate.put(entryDateMidnight, painDiaryEntry);
                }
            }
        }

        //Now rebuild the list
        PainDiaryEntries newEntries = new PainDiaryEntries();
        newEntries.addAll(entriesByDate.values());
        return newEntries;
    }

    private GraphDataSeries createPainDiaryDataSeries(
			PainDiaryEntries painDiaryEntries) {
		GraphDataSeries dataSeries = new GraphDataSeries();
		dataSeries.setTitle("Pain Diary Severity");

		for (PainDiaryEntry entry : painDiaryEntries){
			GraphItem item = buildGraphItem(entry);
			dataSeries.add(item);
			updateMinMax(dataSeries, item);
		}
		return dataSeries;
	}

	private GraphDataSeries createPainAssessmentDataSeries(AssessmentResults assessmentResults) {
		GraphDataSeries dataSeries = new GraphDataSeries();
		dataSeries.setTitle("Pain Diary Severity");

        if (assessmentResults != null) {
            for (AssessmentResult result : assessmentResults){
                Date dateTaken = result.getDateTaken();
                for (AssessmentResponse response : result.getResponses()){
                    AssessmentQuestion question = response.getQuestion();
                    if (question.getId().equals("Q1")){
                        AssessmentQuestionChoices choices = response.getSelectedChoices();
                        for (AssessmentQuestionChoice choice : choices){
                            for (AssessmentProperty property : choice.getProperties()){
                                if (property.getName().equals(AssessmentCode.RESULT_PROPERTY_SELECTED_VALUE)){
                                    String value = property.getValue();

                                    GraphItem item = buildGraphItem(value, dateTaken);
                                    dataSeries.add(item);
                                    updateMinMax(dataSeries, item);
                                }
                            }
                        }
                    }
                }
            }
        }
		return dataSeries;
	}


	private GraphItem buildGraphItem(String value, Date dateTaken){
		GraphItem item = new GraphItem();
		
		item.setX(DateHelper.dateToSeconds(dateTaken));
		item.setY(Double.parseDouble(value));
		
		item.setDate(dateTaken);
		item.setLabel(buildAssessmentResultLabel(value));
		return item;
	}
	
	private GraphItem buildGraphItem(PainDiaryEntry entry) {
		GraphItem item = new GraphItem();
		Date dateCompleted = entry.getDateCreated();
		item.setX(DateHelper.dateToSeconds(dateCompleted));
		
		double y = Double.parseDouble( entry.getPainQuestionValues().getAveragePainIntensity());
		item.setY(y);

		item.setDate(dateCompleted);
		item.setLabel(buildPainDiaryLabel(entry));

		return item;
	}

	private String buildAssessmentResultLabel(String value){
		return value;
	}
	private String buildPainDiaryLabel(PainDiaryEntry entry) {
		return entry.getPainQuestionValues().getAveragePainIntensity();
	}

	private void updateMinMax(GraphDataSeries dataSeries, GraphItem item) {
		dataSeries.setMaxX(Math.max(dataSeries.getMaxX(), item.getX()));
		dataSeries.setMaxY(Math.max(dataSeries.getMaxY(), item.getY()));
		if (dataSeries.size() == 1) {
			dataSeries.setMinX(item.getX());
			dataSeries.setMinY(item.getY());
		} else {
			dataSeries.setMinX(Math.min(dataSeries.getMinX(), item.getX()));
			dataSeries.setMinY(Math.min(dataSeries.getMinY(), item.getY()));
		}
	}

}