package com.agilex.healthcare.mobilehealthplatform.datalayer.medication;

import java.util.Date;

import com.agilex.healthcare.mobilehealthplatform.domain.Medication;
import com.agilex.healthcare.mobilehealthplatform.domain.Medications;
import com.agilex.healthcare.mobilehealthplatform.domain.code.MedicationRefillCode;
import com.agilex.healthcare.utility.DateHelper;

public class MedicationRefillHelper {
	
	// Medications are eligible x number of days before the prescription is due to run out
	static final int NINETY_DAY_REFILL_PERIOD = 14;
	static final int THIRTY_DAY_REFILL_PERIOD = 10;

	public void determineRefillAction(Medication medication) {
		boolean isEligibleForRefill =  meetsRefillCountRequirements(medication) && meetsDateRequirements(medication);

		if (isEligibleForRefill) {
			medication.setRefillAction(MedicationRefillCode.REFILL_ACTION_ELIGIBLE);
		} else {
			medication.setRefillAction(MedicationRefillCode.REFILL_ACTION_NONE);
		}
	}
	
	private boolean meetsRefillCountRequirements(Medication medication) {
		String refills = medication.getRefills();
		int refillCount = 0;
		try {
			refillCount = Integer.parseInt(refills);
		} catch (Exception cannotParseNumberOfRefills) {
			// no information available on the number of refills. Safely assume no refills are available.
		}
		
		return refillCount > 0;
	}

	public boolean meetsDateRequirements(Medication medication) {
		Date filledDate = DateHelper.beginOfDate(medication.getLastFilledDate());
		Date today = DateHelper.endOfDay(new Date());
		
		int delta = DateHelper.calculateDeltaInDays(filledDate, today);
		int refillEligibilityPeriod = medication.getDaysSupply().equals("90") ? NINETY_DAY_REFILL_PERIOD : THIRTY_DAY_REFILL_PERIOD;
		calculateAndSetNextRefillDate(medication);
		
		return (delta >= refillEligibilityPeriod);
	}
	
	void calculateAndSetNextRefillDate(Medication medication) {
		if("90".equals(medication.getDaysSupply()))
			medication.setCalculatedNextFillDate(DateHelper.plusDays(medication.getLastFilledDate(), 90 - NINETY_DAY_REFILL_PERIOD));
		else
			medication.setCalculatedNextFillDate(DateHelper.plusDays(medication.getLastFilledDate(), 30 - THIRTY_DAY_REFILL_PERIOD));
	}
	
	public Date nextRefillDate(Date date, String daysSupply) {
		if("90".equals(daysSupply))
			return DateHelper.plusDays(date, 90 - NINETY_DAY_REFILL_PERIOD);
		else
			return DateHelper.plusDays(date, 30 - THIRTY_DAY_REFILL_PERIOD);
	}

	public void determineRefillAction(Medications medications) {
		for (Medication medication : medications) {
			determineRefillAction(medication);
		}
	}

}
