package com.agilex.healthcare.mobilehealthplatform.datalayer.medication;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.agilex.healthcare.mobilehealthplatform.domain.Medication;
import com.agilex.healthcare.mobilehealthplatform.domain.Medications;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilterer;
import com.agilex.healthcare.utility.DateHelper;
import com.agilex.healthcare.utility.MedicationDateComparator;
import com.agilex.healthcare.utility.NullChecker;

public class MedicationFilterer {
	private static final Log logger = LogFactory.getLog(MedicationFilterer.class);
	private static int MIN_PAGE_VALUE = 0;
	private static int MIN_RESULTCOUNT_VALUE = 1;
	private static int RECENTLY_EXPIRED_DURATION = 90;
	private static List<String> activeKeys = createStatusMappingKeys("ACTIVE", "REFILL");
	private static List<String> expiredKeys = createStatusMappingKeys("EXPIRED", "DISCONTINUED", "SUSPENDED", "DELETED", "DONE", "DISCONTINUED BY PROVIDER", "DISCONTINUED (EDIT)");

	public static Medications filter(Medications medications, MedicationFilter filter) {
		boolean meetsSourceSystem = false;
		boolean meetsMedicationSource = false;
		boolean meetsStatus = false;
		boolean meetsExpiration = false;

		Medications filteredMedications = new Medications();
		if (NullChecker.isNullish(medications)) {
			return filteredMedications;
		}

		if (filter == null) {
			return medications;
		}

		Medications medicationsInDateRange = DateFilterer.filterByDate(medications, filter.getDateFilter());

		for (Medication medication : medicationsInDateRange) {
			if (filter != null) {

				String translatedStatus = translateStatus(medication.getStatus());
				List<String> translatedSourceSystems = translateSourceSystems(filter.getSourceSystems());

				meetsSourceSystem = doesMeetFilterWithMultipleCriteria(isSourceSystemFilterGiven(filter), translateSourceSystem(medication.getSourceSystem()), translatedSourceSystems);
				meetsMedicationSource = doesMeetFilterWithMultipleCriteria(isMedicationSourceFilterGiven(filter), medication.getMedicationSource(), filter.getMedicationSources());
				meetsStatus = doesMeetFilterWithMultipleCriteria(isStatusFilterGiven(filter), translatedStatus, filter.getStatuses());
				meetsExpiration = doesMeetExpirationCriteria(medication.getExpiredDate(), translatedStatus, filter);

				logger.debug(String.format("Filter criteria state for Medication %s: [sourceSystem: %s][medicationSource: %s][status: %s][expiration %s]", medication.getDrugName(), meetsSourceSystem, meetsMedicationSource, meetsStatus, meetsExpiration));

				if (meetsSourceSystem && meetsMedicationSource && meetsStatus && meetsExpiration) {
					filteredMedications.add(medication);
				}
			} else {
				filteredMedications.add(medication);
			}
		}

		filteredMedications = sortMedicationsByStatusThenDate(filteredMedications);
        filteredMedications.setPrePaginationFilterSize(filteredMedications.getSize());
        return paginateIfNeeded(filteredMedications, filter);
	}

	private static List<String> translateSourceSystems(List<String> sourceSystems) {
		List<String> translatedSystems = new ArrayList<String>();
		if (NullChecker.isNotNullish(sourceSystems)) {
			for (String sourceSystem : sourceSystems) {
				translatedSystems.add(translateSourceSystem(sourceSystem));
			}
		}
		return translatedSystems;
	}

	private static String translateSourceSystem(String sourceSystem) {
		if (NullChecker.isNotNullish(sourceSystem) && sourceSystem.equalsIgnoreCase("VA")) {
			return "VA";
		} else {
			return "Non-VA";
		}
	}

	private static boolean doesMeetFilterWithMultipleCriteria(boolean isFilterGiven, String field, List<String> criterias) {
		if (isFilterGiven == false) {
			return true;
		} else {
			if (NullChecker.isNullish(field)) {
				return false;
			} else {
				return doesMedicationFieldMeetACriteria(field, criterias);
			}
		}
	}

	private static boolean doesMeetExpirationCriteria(Date expirationDate, String status, MedicationFilter filter) {
		if(isStatusFilterGiven(filter) == false)
			return true;

		if(doesFilterContainExpired(filter) && "EXPIRED".equals(status)) {
			return true;
		} else if(doesFilterContainActive(filter) && "ACTIVE".equalsIgnoreCase(status)) {
			return true;
		} else if ("EXPIRED".equalsIgnoreCase(status) && NullChecker.isNotNullish(expirationDate)) {
			if (meetsRecentlyExpiredMedicationCriteria(expirationDate)) {
				return true;
			}
		}
		return false;
	}

	static boolean isStatusFilterGiven(MedicationFilter filter) {
		return NullChecker.isNotNullish(filter.getStatuses());
	}

	static boolean meetsRecentlyExpiredMedicationCriteria(Date expirationDate) {
		return (DateHelper.calculateDeltaInDays(DateHelper.endOfDay(expirationDate), DateHelper.endOfDay(new Date())) <= RECENTLY_EXPIRED_DURATION);
	}

	static boolean doesFilterContainExpired(MedicationFilter filter) {
		if(!isStatusFilterGiven(filter))
			return false;

		List<String> statuses = filter.getStatuses();
		for(String status : statuses) {
			if("EXPIRED".equalsIgnoreCase(status))
				return true;
		}
		return false;
	}

	public static boolean doesFilterContainActive(MedicationFilter filter) {
		if(!isStatusFilterGiven(filter))
			return false;

		List<String> statuses = filter.getStatuses();
		for(String status : statuses) {
			if("ACTIVE".equalsIgnoreCase(status))
				return true;
		}
		return false;
	}

	private static boolean isSourceSystemFilterGiven(MedicationFilter filter) {
		return NullChecker.isNotNullish(filter.getSourceSystems());
	}

	private static boolean isMedicationSourceFilterGiven(MedicationFilter filter) {
		return NullChecker.isNotNullish(filter.getMedicationSources());
	}

	private static boolean isPaginationFilterNotGiven(MedicationFilter filter) {
		return NullChecker.isNullish(filter.getPage()) && NullChecker.isNullish(filter.getResultCount());
	}

	private static boolean doesMedicationFieldMeetACriteria(String field, List<String> criterias) {
		if (NullChecker.isNotNullish(criterias)) {
			for (String criteria : criterias) {
				criteria = preventTrailingCommaBugs(criteria);
				if (field.equalsIgnoreCase(criteria)) {
					return true;
				}
			}
			return false;
		} else {
			return true;
		}
	}

	private static String preventTrailingCommaBugs(String value) {
		if (value.charAt(value.length() - 1) == ',') {
			return value.substring(0, value.length() - 1);
		}
		return value;
	}

	private static Medications paginateIfNeeded(Medications medications, MedicationFilter filter) {
		if (isPaginationFilterNotGiven(filter) == true) {
			return medications;
		} else if (parseNumericValue(filter.getPage()) < MIN_PAGE_VALUE || parseNumericValue(filter.getResultCount()) < MIN_RESULTCOUNT_VALUE) {
			throw new UnsupportedOperationException(String.format("page value must be at least %s and resultCount must be at least %s", MIN_PAGE_VALUE, MIN_RESULTCOUNT_VALUE));
		}

		int page = Integer.parseInt(filter.getPage());
		int pageSize = Integer.parseInt(filter.getResultCount());
		Medications pageOfMedications = getPage(medications, page, pageSize);
        pageOfMedications.setPrePaginationFilterSize(medications.getPrePaginationFilterSize());
		return pageOfMedications;
	}

	private static int parseNumericValue(String value) {
		int parsedValue = -1;
		try {
			parsedValue = Integer.parseInt(value);
		} catch (NumberFormatException notANumber) {
			throw new UnsupportedOperationException("Expected a numeric value but could not parse it as a number");
		}
		return parsedValue;
	}

	private static Medications getPage(Medications medications, int page, int pageSize) {
		int startIndex = pageSize * (page - 1);
		int endIndex = medications.size();
		if (pageSize * page < endIndex) {
			endIndex = pageSize * page;
		}
		List<Medication> medicationSubList = medications.subList(startIndex, endIndex);

		return new Medications(medicationSubList);
	}

	private static List<String> createStatusMappingKeys(String... keys) {
		List<String> builtKeys = new ArrayList<String>();
		for (int i = 0; i < keys.length; i++) {
			builtKeys.add(keys[i]);
		}
		return builtKeys;
	}

	private static String translateStatus(String key) {
		if (NullChecker.isNotNullish(key)) {
			if (activeKeys.contains(key.toUpperCase())) {
				return "ACTIVE";
			} else if (expiredKeys.contains(key.toUpperCase())) {
				return "EXPIRED";
			}
		} else {
			return "EXPIRED";
		}
		return key;
	}

	private static Medications sortMedicationsByStatusThenDate(Medications filteredMedications) {
		Medications activeMedications = new Medications();
		Medications expiredMedications = new Medications();
		for(Medication medication : filteredMedications) {
			if("active".equalsIgnoreCase(medication.getStatus())) {
				activeMedications.add(medication);
			} else {
				expiredMedications.add(medication);
			}
		}

		Medications sortedMedications = new Medications();
		MedicationDateComparator comparator = new MedicationDateComparator();
		Collections.sort(activeMedications, comparator);
		Collections.sort(expiredMedications, comparator);
		Collections.reverse(expiredMedications);
		sortedMedications.addAll(activeMedications);
		sortedMedications.addAll(expiredMedications);

		return sortedMedications;
	}
}
