package com.agilex.healthcare.mobilehealthplatform.pdf.securemessage;

import static org.junit.Assert.assertTrue;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Date;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.regex.Pattern;

import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.MessageSource;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;

import com.agilex.healthcare.mbb.datalayer.PdfDataLayer;
import com.agilex.healthcare.mbb.datalayer.PdfDataLayerItext;
import com.agilex.healthcare.mobilehealthplatform.domain.SecureMessage;
import com.itextpdf.text.pdf.PRTokeniser;
import com.itextpdf.text.pdf.PdfReader;

@ContextConfiguration(locations = { "classpath:applicationContext.xml"})
@RunWith(SpringJUnit4ClassRunner.class)
public abstract class SecureMessagePDFTest {
  private static final org.apache.commons.logging.Log LOGGER = org.apache.commons.logging.LogFactory.getLog(SecureMessagePDFTest.class);
  
  @Autowired
  MessageSource messageSource;

  private final String FROM = "From";
  private final String TO = "To";
  private final String SUBJECT_CATEGORY = "0";
  private final String SUBJECT = "Subject";
  private final Date DATE = new Date();
  private final String MESSAGE = "Message";
  
  @Test
  public void testReport() throws IOException {
    SecureMessage secureMessage = buildSecureMessage(FROM, TO, SUBJECT_CATEGORY, SUBJECT, DATE, MESSAGE);
    ByteArrayOutputStream outputStream = generateReport(secureMessage);
    
    byte[] bytes = outputStream.toByteArray();
    outputStream.close();
    
    Iterator<String> iterator = getAssertionTextIterator();
    
    while (iterator.hasNext()) {
      String text = iterator.next();
      InputStream inputStream = new ByteArrayInputStream(bytes);
      
      List<PRTokeniser> tokenisers = parseReport(inputStream);
      
      boolean match = false;
      for (PRTokeniser tokeniser : tokenisers) {
        if (!match && containsText(tokeniser, text)) {
          match = true;
        }

        try {
          tokeniser.close();
        } catch (IOException e) {
          LOGGER.error("Error closing tokeniser", e);
        }
      }

      assertTrue(String.format("(%s) not found", text), match);
    }
  }
  
  protected boolean containsText(PRTokeniser tokeniser, String text) throws IOException {
    boolean match = false;
    while(tokeniser.nextToken()) {
      if (tokeniser.getTokenType() == PRTokeniser.TK_STRING
          && Pattern.compile(Pattern.quote(text), Pattern.CASE_INSENSITIVE).matcher(tokeniser.getStringValue()).find()) {
        match = true;
        break;
      }
    }
    
    return match;
  }
  
  private ByteArrayOutputStream generateReport(SecureMessage secureMessage) {
    ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
    
    PdfDataLayer pdfDataLayer = new PdfDataLayerItext();
    pdfDataLayer.generateSecureMessagePdf(outputStream, messageSource, secureMessage);
    
    return outputStream;
  }
  
  private List<PRTokeniser> parseReport(InputStream inputStream) throws IOException {
    List<PRTokeniser> tokenisers = new LinkedList<PRTokeniser>();
    
    PdfReader reader = new PdfReader(inputStream);
        
    for (int i = 1; i <= reader.getNumberOfPages(); i++) {
      byte[] streamBytes = reader.getPageContent(i);
      PRTokeniser tokenizer = new PRTokeniser(streamBytes);
          tokenisers.add(tokenizer);
    }
        
    inputStream.close();
    reader.close();
        
    return tokenisers;
  }
  
  public SecureMessage buildSecureMessage(String from, String to, String subjectCategory, String subject, Date dateSent, String messageBody) {
    SecureMessage secureMessage = new SecureMessage();
    secureMessage.setFrom(from);
    secureMessage.setTo(to);
    secureMessage.setSubjectCategory(subjectCategory);
    secureMessage.setSubject(subject);
    secureMessage.setDateSent(dateSent);
    secureMessage.setMessageBody(messageBody);
    
    return secureMessage;
  }
  
  public Iterator<String> getAssertionTextIterator() {
    List<String> assertTextList = new LinkedList<String>();
    assertTextList.add(FROM);
    assertTextList.add(TO);
    assertTextList.add(SUBJECT);
    assertTextList.add(MESSAGE);
    // date sent
    // message category
    return assertTextList.iterator();
  }
}
