package com.agilex.healthcare.mobilehealthplatform.pdf.mygoals;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Calendar;
import java.util.Date;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.regex.Pattern;

import junit.framework.Assert;

import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.MessageSource;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;

import com.agilex.healthcare.mobilehealthplatform.domain.DOBDate;
import com.agilex.healthcare.mobilehealthplatform.domain.Mygoals;
import com.agilex.healthcare.mobilehealthplatform.domain.MygoalsList;
import com.agilex.healthcare.mobilehealthplatform.domain.Patient;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilter;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilterImpl;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.MutableDateFilter;
import com.agilex.healthcare.mobilehealthplatform.enumeration.MygoalsInventoryType;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.Domain;
import com.agilex.healthcare.mygoals.datalayer.PdfDataLayer;
import com.agilex.healthcare.mygoals.datalayer.PdfDataLayerItext;
import com.agilex.healthcare.pdf.PdfGenerationContext;
import com.agilex.healthcare.utility.DateHelper;
import com.itextpdf.text.pdf.PRTokeniser;
import com.itextpdf.text.pdf.PdfReader;

@ContextConfiguration(locations = { "classpath:applicationContext.xml"})
@RunWith(SpringJUnit4ClassRunner.class)
public class MygoalsPDFTest {
  private static final org.apache.commons.logging.Log LOGGER = org.apache.commons.logging.LogFactory.getLog(MygoalsPDFTest.class);
  
  @Autowired
  MessageSource messageSource;

  protected Locale locale = Locale.US;
  
  @Test
  public void testReport() throws IOException {
    PdfGenerationContext context = specifyContext(buildContext());
    ByteArrayOutputStream outputStream = generateReport(context);
//    OutputStream f1 = new FileOutputStream("file2.pdf");  
//    f1.write(outputStream.toByteArray()); 
//    f1.close();   
    byte[] bytes = outputStream.toByteArray();
    outputStream.close();
    
    Iterator<String> iterator = getAssertionTextIterator();
    
    while (iterator.hasNext()) {
      String text = iterator.next();
      InputStream inputStream = new ByteArrayInputStream(bytes);
      
      List<PRTokeniser> tokenisers = parseReport(inputStream);
      
      boolean match = false;
      for (PRTokeniser tokeniser : tokenisers) {
        if (!match && containsText(tokeniser, text)) {
          match = true;
        }

        try {
          tokeniser.close();
        } catch (IOException e) {
          LOGGER.error("Error closing tokeniser", e);
        }
      }

      Assert.assertTrue(String.format("(%s) not found", text), match);
    }
  }

  @Test
  public void testProviderReport() throws IOException {
    PdfGenerationContext context = specifyContext(buildContext());
    context.setProviderReport(true);
    ByteArrayOutputStream outputStream = generateReport(context);
//    OutputStream f1 = new FileOutputStream("file3.pdf");  
//    f1.write(outputStream.toByteArray()); 
//    f1.close();   
    byte[] bytes = outputStream.toByteArray();
    outputStream.close();
    
    Iterator<String> iterator = getAssertionTextIterator();
    
    while (iterator.hasNext()) {
      String text = iterator.next();
      InputStream inputStream = new ByteArrayInputStream(bytes);
      
      List<PRTokeniser> tokenisers = parseReport(inputStream);
      
      boolean match = false;
      for (PRTokeniser tokeniser : tokenisers) {
        if (!match && containsText(tokeniser, text)) {
          match = true;
        }

        try {
          tokeniser.close();
        } catch (IOException e) {
          LOGGER.error("Error closing tokeniser", e);
        }
      }

      Assert.assertTrue(String.format("(%s) not found", text), match);
    }
  }
  
  protected boolean containsText(PRTokeniser tokeniser, String text) throws IOException {
    boolean match = false;
    while(tokeniser.nextToken()) {
      if (tokeniser.getTokenType() == PRTokeniser.TK_STRING
          && Pattern.compile(Pattern.quote(text), Pattern.CASE_INSENSITIVE).matcher(tokeniser.getStringValue()).find()) {
        match = true;
        break;
      }
    }
    
    return match;
  }
  
  private ByteArrayOutputStream generateReport(PdfGenerationContext context) {
    ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
    
    PdfDataLayer pdfDataLayer = new PdfDataLayerItext();
    pdfDataLayer.generateReport(outputStream, context);
    
    return outputStream;
  }
  
  private List<PRTokeniser> parseReport(InputStream inputStream) throws IOException {
    List<PRTokeniser> tokenisers = new LinkedList<PRTokeniser>();
    
    PdfReader reader = new PdfReader(inputStream);
        
    for (int i = 1; i <= reader.getNumberOfPages(); i++) {
      byte[] streamBytes = reader.getPageContent(i);
      PRTokeniser tokenizer = new PRTokeniser(streamBytes);
          tokenisers.add(tokenizer);
    }
        
    inputStream.close();
    reader.close();
        
    return tokenisers;
  }
  
  public PdfGenerationContext specifyContext(PdfGenerationContext context) {
    context.setInformation(Domain.mygoals + "," +  context.getPatientIdentifier().getUniqueId(), createInventoryResponses());
    
    return context;
  }
  
  public Iterator<String> getAssertionTextIterator() {
    List<String> assertTextList = new LinkedList<String>();
    assertTextList.add(messageSource.getMessage("mygoals.report.reflections", null, locale));
    assertTextList.add(messageSource.getMessage("reflection-matters", null, locale));
    assertTextList.add(messageSource.getMessage("reflection-joy", null, locale));
//    assertTextList.add(messageSource.getMessage("reflection-sorrow", null, locale));
//    assertTextList.add(messageSource.getMessage("reflection-vision", null, locale));
    assertTextList.add(messageSource.getMessage("mygoals.report.feelings", null, locale));
//    assertTextList.add(messageSource.getMessage("feelings-physically", null, locale));
//    assertTextList.add(messageSource.getMessage("feelings-mentally", null, locale));
//    assertTextList.add(messageSource.getMessage("feelings-day-to-day", null, locale));
    assertTextList.add(messageSource.getMessage("mygoals.report.feelings.scale", null, locale));
    assertTextList.add(messageSource.getMessage("mygoals.report.currentanddesirestate", null, locale));
    assertTextList.add(messageSource.getMessage("mygoals.report.currentstate", null, locale));
    assertTextList.add(messageSource.getMessage("mygoals.report.desirestate", null, locale));
//    assertTextList.add(messageSource.getMessage("mygoals.report.currentanddesirestate.scale", null, locale));
    assertTextList.add(messageSource.getMessage("Current-body-slider", null, locale));
    assertTextList.add(messageSource.getMessage("Current-recharge-slider", null, locale));
    assertTextList.add(messageSource.getMessage("Current-fuel-slider", null, locale));
    assertTextList.add(messageSource.getMessage("Current-life-slider", null, locale));
    assertTextList.add(messageSource.getMessage("Current-family-slider", null, locale));
    assertTextList.add(messageSource.getMessage("Current-soul-slider", null, locale));
    assertTextList.add(messageSource.getMessage("Current-surroundings-slider", null, locale));
    assertTextList.add(messageSource.getMessage("Current-mind-slider", null, locale));
//    assertTextList.add(messageSource.getMessage("Current-prevention-slider", null, locale));
    assertTextList.add(messageSource.getMessage("Current-intervention-slider", null, locale));
    assertTextList.add(messageSource.getMessage("mygoals.report.priorities", null, locale));
    assertTextList.add(messageSource.getMessage("mygoals.report.priorities.p1", null, locale));
    assertTextList.add(messageSource.getMessage("mygoals.report.priorities.p2", null, locale));
    return assertTextList.iterator();
  }
  
  private MygoalsList createInventoryResponses() {
    MygoalsList mygoalsList = new MygoalsList();
    Mygoals mygoals = new Mygoals();
    mygoals.setQuestionKey("reflection-matters");
    mygoals.setAnswer("This is reflection matters");
    mygoals.setDateUpdated(new Date());
    mygoals.setInventoryType(MygoalsInventoryType.REFLECTIONS.name());
    mygoalsList.add(mygoals);
  
    mygoals = new Mygoals();
    mygoals.setQuestionKey("reflection-joy");
    mygoals.setAnswer("This is reflection joy");
    mygoals.setDateUpdated(new Date());
    mygoals.setInventoryType(MygoalsInventoryType.REFLECTIONS.name());
    mygoalsList.add(mygoals);   
    
    mygoals = new Mygoals();
    mygoals.setQuestionKey("reflection-sorrow");
    mygoals.setAnswer("This is reflection sorrow");
    mygoals.setDateUpdated(new Date());
    mygoals.setInventoryType(MygoalsInventoryType.REFLECTIONS.name());
    mygoalsList.add(mygoals); 

    mygoals = new Mygoals();
    mygoals.setQuestionKey("reflection-vision");
    mygoals.setAnswer("This is reflection vision");
    mygoals.setDateUpdated(new Date());
    mygoals.setInventoryType(MygoalsInventoryType.REFLECTIONS.name());
    mygoalsList.add(mygoals); 
  
    mygoals = new Mygoals();
    mygoals.setQuestionKey("feelings-physically");
    mygoals.setAnswer("1");
    mygoals.setDateUpdated(new Date());
    mygoals.setInventoryType(MygoalsInventoryType.FEELINGS.name());
    mygoalsList.add(mygoals);
  
    mygoals = new Mygoals();
    mygoals.setQuestionKey("feelings-mentally");
    mygoals.setAnswer("2");
    mygoals.setDateUpdated(new Date());
    mygoals.setInventoryType(MygoalsInventoryType.FEELINGS.name());
    mygoalsList.add(mygoals);   
    
    mygoals = new Mygoals();
    mygoals.setQuestionKey("feelings-day-to-day");
    mygoals.setAnswer("3");
    mygoals.setDateUpdated(new Date());
    mygoals.setInventoryType(MygoalsInventoryType.FEELINGS.name());
    mygoalsList.add(mygoals);     

    mygoals = new Mygoals();
    mygoals.setQuestionKey("Current-body-slider");
    mygoals.setAnswer("1");
    mygoals.setDateUpdated(new Date());
    mygoals.setInventoryType(MygoalsInventoryType.CURRENT_AND_DESIRED_STATE.name());
    mygoalsList.add(mygoals);
  
    mygoals = new Mygoals();
    mygoals.setQuestionKey("Desired-body-slider");
    mygoals.setAnswer("2");
    mygoals.setDateUpdated(new Date());
    mygoals.setInventoryType(MygoalsInventoryType.CURRENT_AND_DESIRED_STATE.name());
    mygoalsList.add(mygoals); 

    mygoals = new Mygoals();
    mygoals.setQuestionKey("Current-recharge-slider");
    mygoals.setAnswer("2");
    mygoals.setDateUpdated(new Date());
    mygoals.setInventoryType(MygoalsInventoryType.CURRENT_AND_DESIRED_STATE.name());
    mygoalsList.add(mygoals);
  
    mygoals = new Mygoals();
    mygoals.setQuestionKey("Desired-recharge-slider");
    mygoals.setAnswer("3");
    mygoals.setDateUpdated(new Date());
    mygoals.setInventoryType(MygoalsInventoryType.CURRENT_AND_DESIRED_STATE.name());
    mygoalsList.add(mygoals); 

    mygoals = new Mygoals();
    mygoals.setQuestionKey("Current-fuel-slider");
    mygoals.setAnswer("3");
    mygoals.setDateUpdated(new Date());
    mygoals.setInventoryType(MygoalsInventoryType.CURRENT_AND_DESIRED_STATE.name());
    mygoalsList.add(mygoals);
  
    mygoals = new Mygoals();
    mygoals.setQuestionKey("Desired-fuel-slider");
    mygoals.setAnswer("4");
    mygoals.setDateUpdated(new Date());
    mygoals.setInventoryType(MygoalsInventoryType.CURRENT_AND_DESIRED_STATE.name());
    mygoalsList.add(mygoals); 

    mygoals = new Mygoals();
    mygoals.setQuestionKey("Current-life-slider");
    mygoals.setAnswer("4");
    mygoals.setDateUpdated(new Date());
    mygoals.setInventoryType(MygoalsInventoryType.CURRENT_AND_DESIRED_STATE.name());
    mygoalsList.add(mygoals);
  
    mygoals = new Mygoals();
    mygoals.setQuestionKey("Desired-life-slider");
    mygoals.setAnswer("5");
    mygoals.setDateUpdated(new Date());
    mygoals.setInventoryType(MygoalsInventoryType.CURRENT_AND_DESIRED_STATE.name());
    mygoalsList.add(mygoals); 

    mygoals = new Mygoals();
    mygoals.setQuestionKey("Current-family-slider");
    mygoals.setAnswer("5");
    mygoals.setDateUpdated(new Date());
    mygoals.setInventoryType(MygoalsInventoryType.CURRENT_AND_DESIRED_STATE.name());
    mygoalsList.add(mygoals);
  
    mygoals = new Mygoals();
    mygoals.setQuestionKey("Desired-family-slider");
    mygoals.setAnswer("6");
    mygoals.setDateUpdated(new Date());
    mygoals.setInventoryType(MygoalsInventoryType.CURRENT_AND_DESIRED_STATE.name());
    mygoalsList.add(mygoals); 

    mygoals = new Mygoals();
    mygoals.setQuestionKey("Current-soul-slider");
    mygoals.setAnswer("6");
    mygoals.setDateUpdated(new Date());
    mygoals.setInventoryType(MygoalsInventoryType.CURRENT_AND_DESIRED_STATE.name());
    mygoalsList.add(mygoals);
  
    mygoals = new Mygoals();
    mygoals.setQuestionKey("Desired-soul-slider");
    mygoals.setAnswer("7");
    mygoals.setDateUpdated(new Date());
    mygoals.setInventoryType(MygoalsInventoryType.CURRENT_AND_DESIRED_STATE.name());
    mygoalsList.add(mygoals); 

    mygoals = new Mygoals();
    mygoals.setQuestionKey("Current-surroundings-slider");
    mygoals.setAnswer("7");
    mygoals.setDateUpdated(new Date());
    mygoals.setInventoryType(MygoalsInventoryType.CURRENT_AND_DESIRED_STATE.name());
    mygoalsList.add(mygoals);
  
    mygoals = new Mygoals();
    mygoals.setQuestionKey("Desired-surroundings-slider");
    mygoals.setAnswer("8");
    mygoals.setDateUpdated(new Date());
    mygoals.setInventoryType(MygoalsInventoryType.CURRENT_AND_DESIRED_STATE.name());
    mygoalsList.add(mygoals); 

    mygoals = new Mygoals();
    mygoals.setQuestionKey("Current-mind-slider");
    mygoals.setAnswer("8");
    mygoals.setDateUpdated(new Date());
    mygoals.setInventoryType(MygoalsInventoryType.CURRENT_AND_DESIRED_STATE.name());
    mygoalsList.add(mygoals);
  
    mygoals = new Mygoals();
    mygoals.setQuestionKey("Desired-mind-slider");
    mygoals.setAnswer("9");
    mygoals.setDateUpdated(new Date());
    mygoals.setInventoryType(MygoalsInventoryType.CURRENT_AND_DESIRED_STATE.name());
    mygoalsList.add(mygoals); 

    mygoals = new Mygoals();
    mygoals.setQuestionKey("Current-prevention-slider");
    mygoals.setAnswer("9");
    mygoals.setDateUpdated(new Date());
    mygoals.setInventoryType(MygoalsInventoryType.CURRENT_AND_DESIRED_STATE.name());
    mygoalsList.add(mygoals);
  
    mygoals = new Mygoals();
    mygoals.setQuestionKey("Desired-prevention-slider");
    mygoals.setAnswer("10");
    mygoals.setDateUpdated(new Date());
    mygoals.setInventoryType(MygoalsInventoryType.CURRENT_AND_DESIRED_STATE.name());
    mygoalsList.add(mygoals); 
    
    mygoals = new Mygoals();
    mygoals.setQuestionKey("Current-intervention-slider");
    mygoals.setAnswer("10");
    mygoals.setDateUpdated(new Date());
    mygoals.setInventoryType(MygoalsInventoryType.CURRENT_AND_DESIRED_STATE.name());
    mygoalsList.add(mygoals);
  
    mygoals = new Mygoals();
    mygoals.setQuestionKey("Desired-intervention-slider");
    mygoals.setAnswer("10");
    mygoals.setDateUpdated(new Date());
    mygoals.setInventoryType(MygoalsInventoryType.CURRENT_AND_DESIRED_STATE.name());
    mygoalsList.add(mygoals); 
    
    mygoals = new Mygoals();
    mygoals.setQuestionKey("priority1-area");
    mygoals.setAnswer("Working My Body");
    mygoals.setDateUpdated(new Date());
    mygoals.setInventoryType(MygoalsInventoryType.PRIORITIES.name());
    mygoalsList.add(mygoals);

    mygoals = new Mygoals();
    mygoals.setQuestionKey("priority1-goal");
    mygoals.setAnswer("Ride my bike 30 minutes daily");
    mygoals.setDateUpdated(new Date());
    mygoals.setInventoryType(MygoalsInventoryType.PRIORITIES.name());
    mygoalsList.add(mygoals);

    mygoals = new Mygoals();
    mygoals.setQuestionKey("priority2-area");
    mygoals.setAnswer("My Personal Development");
    mygoals.setDateUpdated(new Date());
    mygoals.setInventoryType(MygoalsInventoryType.PRIORITIES.name());
    mygoalsList.add(mygoals);
  
    mygoals = new Mygoals();
    mygoals.setQuestionKey("priority2-goal");
    mygoals.setAnswer("I want to discuss with my healthcare team.");
    mygoals.setDateUpdated(new Date());
    mygoals.setInventoryType(MygoalsInventoryType.PRIORITIES.name());
    mygoalsList.add(mygoals); 
    
    return mygoalsList;
  } 
  public PdfGenerationContext buildContext() {
    PdfGenerationContext context = new PdfGenerationContext();

    context.setPatientIdentifier(createPatientIdentifier());
    context.setDateFilter(createDateFilter());
    if (messageSource == null) {
      throw new RuntimeException();
    }
    context.setMessageSource(messageSource);
    context.setLocale(locale);
    Patient patient = new Patient();
    patient.setFirstName("Usability");
    patient.setLastName("TestPatient");
    patient.setSsn("123456789");
    patient.setDateOfBirth(new DOBDate(DateHelper.parseDate("10/11/1961")));
    patient.setPatientIdentifier(context.getPatientIdentifier());
    context.setPatient(patient);
    context.setReportDate(new Date());
    
    return context;
  }
  
  protected PatientIdentifier createPatientIdentifier() {
    PatientIdentifier patientIdentifier = new PatientIdentifier();
    patientIdentifier.setAssigningAuthority("default");
    patientIdentifier.setUniqueId("D123401");
    return patientIdentifier;
  }
  
  protected DateFilter createDateFilter() {
    Calendar twoYearsAgo = Calendar.getInstance();
    twoYearsAgo.add(Calendar.YEAR, -2);
    Date startDate = twoYearsAgo.getTime();
    Date endDate = new Date();
    
    MutableDateFilter dateFilter = new DateFilterImpl();
    dateFilter.setStartDate(startDate);
    dateFilter.setEndDate(endDate);
    return dateFilter;
  }
  
}
