package com.agilex.healthcare.mobilehealthplatform.pdf;

import static org.junit.Assert.assertEquals;

import java.awt.Color;
import java.awt.GradientPaint;
import java.awt.Graphics2D;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.Set;

import org.junit.BeforeClass;
import org.junit.Test;

import com.agilex.healthcare.mbb.pdf.PdfRoundCorners;
import com.itextpdf.text.BadElementException;
import com.itextpdf.text.Chapter;
import com.itextpdf.text.Chunk;
import com.itextpdf.text.Document;
import com.itextpdf.text.DocumentException;
import com.itextpdf.text.Element;
import com.itextpdf.text.Font;
import com.itextpdf.text.FontFactory;
import com.itextpdf.text.Image;
import com.itextpdf.text.List;
import com.itextpdf.text.ListItem;
import com.itextpdf.text.PageSize;
import com.itextpdf.text.Paragraph;
import com.itextpdf.text.Phrase;
import com.itextpdf.text.Rectangle;
import com.itextpdf.text.Section;
import com.itextpdf.text.pdf.AcroFields;
import com.itextpdf.text.pdf.PdfContentByte;
import com.itextpdf.text.pdf.PdfPCell;
import com.itextpdf.text.pdf.PdfPTable;
import com.itextpdf.text.pdf.PdfReader;
import com.itextpdf.text.pdf.PdfStamper;
import com.itextpdf.text.pdf.PdfTemplate;
import com.itextpdf.text.pdf.PdfWriter;

/**
 * iText learning tests from http://www.roseindia.net/java/itext/
 * 
 * Other resources:
 * http://www.dikant.de/2007/10/15/layout-pdf-documents-in-java-with-itext/
 * http://www.itextpdf.com/book/examples.php
 */
public class PDFLearningTest {
  private static final String DOB_FIELD = "vha101349[0].#subform[0].DateTimeField1[0]";
  private static final String NAME_FIELD = "vha101349[0].#subform[0].TextField1[0]";
  private static final String DATE_FIELD = "vha101349[0].#subform[0].TextField2[0]";
  private static final String SSN_FIELD = "vha101349[0].#subform[0].SSN1[0]";
  private static final String BASE_LOCATION = "build/resources/test/pdf/";
  private static final String RESOURCES_LOCATION = "src/test/resources/com/agilex/healthcare/mobilehealthplatform/pdf/";
  
  @BeforeClass
  public static void setup() throws Exception {
    new File(BASE_LOCATION).mkdirs();
  }
  
  @Test
  public void testWritingDataToPDF() throws Exception{
    PdfReader reader = new PdfReader(RESOURCES_LOCATION + "ROA.pdf");
    ByteArrayOutputStream os = new ByteArrayOutputStream();
    PdfStamper stamper = new PdfStamper(reader, os);
    AcroFields acroFields = stamper.getAcroFields();
    
    @SuppressWarnings("unchecked")
    Set<String> keySet = acroFields.getFields().keySet();
    
    for (String key : keySet) {
      System.out.println(key + ":" + acroFields.getField(key));
    }
    
    acroFields.setField(SSN_FIELD, "");
    acroFields.setField(DATE_FIELD, "10/22/2012");
    acroFields.setField(NAME_FIELD, "Younger, Gallow");
    acroFields.setField(DOB_FIELD, "01/01/2000");
    stamper.setFormFlattening(true);
    stamper.close();

    assertEquals("", acroFields.getField(SSN_FIELD));
    assertEquals("10/22/2012", acroFields.getField(DATE_FIELD));
    assertEquals("Younger, Gallow", acroFields.getField(NAME_FIELD));

    
    
    os.flush();
    os.close();
  }

  @Test
  public void testCreateBasicPdf() throws DocumentException, FileNotFoundException {
    Document document = new Document();
    PdfWriter.getInstance(document, new FileOutputStream(BASE_LOCATION + "basic.pdf"));
    document.open();
    
    document.add(new Paragraph("test paragraph."));
    
    Chunk chunk=new Chunk("test chunk.");
    chunk.setUnderline(+1f,-2f);
    chunk.setBackground(Color.lightGray);
    document.add(new Paragraph(chunk));
    
    document.add(new Paragraph(new Chunk("This text is in Times Roman. ", new Font(Font.TIMES_ROMAN, 12))));
    
    document.close();
  }

  @Test
  public void testCreatePdfWithChaptersAndSections() throws DocumentException, FileNotFoundException {
    Document document = new Document();
    PdfWriter.getInstance(document, new FileOutputStream(BASE_LOCATION + "chaptersAndSections.pdf"));
    document.open();
    Chapter chapter=new Chapter(new Paragraph("Chapter1"),1);
    Section section1=chapter.addSection(new Paragraph("Section1"));
    section1.add(new Paragraph("paragraph in section 1"));
    Section section2=chapter.addSection(new Paragraph("section2"));
    section2.add(new Paragraph("paragarph in section 2"));
    document.add(chapter);
    document.close();
  }

  @Test
  public void testCreatePdfWithList() throws DocumentException, FileNotFoundException {
    Document document = new Document();
    PdfWriter.getInstance(document, new FileOutputStream(BASE_LOCATION + "list.pdf"));
    document.open();
    //Create a numbered List with  30-point field for the numbers. 
    List list = new List(true, 30);
    list.add(new ListItem("First List"));
    list.add(new ListItem("Second List"));
    list.add(new ListItem("Third List"));
    document.add(list);
    //Add a separator.
    document.add(Chunk.NEWLINE);
    //Create a symboled List with a 30-point field for the symbols.          
    list = new List(false, 30);
    list.add (new ListItem ("Orange"));
    list.add (new ListItem ("Apple"));
    list.add (new ListItem ("Cherry"));
    list.add (new ListItem ("Banana"));
    // Add the list to the document.
    document.add(list);
    document.close();
  }

  @Test
  public void testCreatePdfWithImages() throws DocumentException, FileNotFoundException, IOException {
    StringBuilder sb = new StringBuilder();
    sb.append("Lorem ipsum dolor sit amet, consectetur adipisicing elit, sed do eiusmod tempor incididunt ");
    sb.append("ut labore et dolore magna aliqua. Ut enim ad minim veniam, quis nostrud exercitation ullamco ");
    sb.append("laboris nisi ut aliquip ex ea commodo consequat. Duis aute irure dolor in reprehenderit in ");
    sb.append("voluptate velit esse cillum dolore eu fugiat nulla pariatur. Excepteur sint occaecat cupidatat");
    sb.append(" non proident, sunt in culpa qui officia deserunt mollit anim id est laborum.");
    sb.append("Lorem ipsum dolor sit amet, consectetur adipisicing elit, sed do eiusmod tempor incididunt ");
    sb.append("ut labore et dolore magna aliqua. Ut enim ad minim veniam, quis nostrud exercitation ullamco ");
    sb.append("laboris nisi ut aliquip ex ea commodo consequat. Duis aute irure dolor in reprehenderit in ");
    sb.append("voluptate velit esse cillum dolore eu fugiat nulla pariatur. Excepteur sint occaecat cupidatat");
    sb.append(" non proident, sunt in culpa qui officia deserunt mollit anim id est laborum.");
    sb.append("Lorem ipsum dolor sit amet, consectetur adipisicing elit, sed do eiusmod tempor incididunt ");
    sb.append("ut labore et dolore magna aliqua. Ut enim ad minim veniam, quis nostrud exercitation ullamco ");
    sb.append("laboris nisi ut aliquip ex ea commodo consequat. Duis aute irure dolor in reprehenderit in ");
    sb.append("voluptate velit esse cillum dolore eu fugiat nulla pariatur. Excepteur sint occaecat cupidatat");
    sb.append(" non proident, sunt in culpa qui officia deserunt mollit anim id est laborum.");
    
    Document document=new Document();
    PdfWriter.getInstance(document,new FileOutputStream(BASE_LOCATION + "images.pdf"));
    document.open();
    Image image = Image.getInstance(RESOURCES_LOCATION + "agilex.png");
    image.scalePercent(85.0f);
    image.setRotationDegrees(0.0f); 
    image.setAlt("customizable alt string");
    image.setCompressionLevel(0);
    image.setAbsolutePosition(50, 750);
    document.add(image);
    
    document.add(new Paragraph(sb.toString()));

    Image image2 = Image.getInstance(RESOURCES_LOCATION + "agilex.png");
    image2.scalePercent(100.0f);
    image2.setCompressionLevel(0);
    image2.setAlignment(Image.RIGHT | Image.TEXTWRAP);
    document.add(image2);

    document.add(new Paragraph(sb.toString()));
    
    Image image3 = Image.getInstance(RESOURCES_LOCATION + "agilex.png");
    image3.scalePercent(50.0f);
    image3.setCompressionLevel(0);
    document.add(image3);
    
    document.close();
  }

  @Test
  public void testCreatePdfWithTables() throws DocumentException, FileNotFoundException, IOException {
    Document document=new Document();
    PdfWriter.getInstance(document,new FileOutputStream(BASE_LOCATION + "tables.pdf"));
    document.open();
    int numColumns = 2;
    PdfPTable table=new PdfPTable(numColumns);
    PdfPCell cell = new PdfPCell (new Paragraph ("Table Name"));
    cell.setColspan (2);
    cell.setHorizontalAlignment (Element.ALIGN_CENTER);
    cell.setBackgroundColor (new Color (255, 0, 0));
    cell.setPadding (10.0f);
    table.addCell (cell);
    cell = new PdfPCell (new Paragraph ("Column 1 Name"));
    cell.setHorizontalAlignment (Element.ALIGN_CENTER);
    cell.setBackgroundColor (new Color (0, 255, 0));
    cell.setPadding (10.0f);
    table.addCell (cell);
    cell = new PdfPCell (new Paragraph ("Column 2 Name"));
    cell.setHorizontalAlignment (Element.ALIGN_CENTER);
    cell.setBackgroundColor (new Color (0, 255, 0));
    cell.setPadding (10.0f);
    table.addCell (cell);
    table.addCell("Key");
    table.addCell("Value");
    table.addCell("Key");
    table.addCell("Value");
    document.add(table);
    document.close();
  }
  
  @Test
  public void testCreatePdfWithCustomLayout() throws DocumentException, FileNotFoundException, IOException {
    Rectangle pageSize = new Rectangle(PageSize.A4);
    pageSize.setBackgroundColor(new Color(0xFF, 0xFF, 0xFF));
    
    Document document = new Document(pageSize);
    document.setMargins(45, 45, 90, 45);
    PdfWriter writer = PdfWriter.getInstance(document, new FileOutputStream(BASE_LOCATION + "custom.pdf"));
    
    document.open();
    PdfContentByte canvas = writer.getDirectContent();
    
    PdfTemplate template = createTemplate(canvas);
    writer.releaseTemplate(template);
    canvas.addTemplate(template, 0, 760);
    
    createGrid(canvas);
    
    int numColumns = 2;
    PdfPTable table = new PdfPTable(numColumns);
    table.setWidthPercentage(90);
    
    table.addCell(createHeaderCell("Contact Information"));

    table.addCell(createCell("Key"));
    table.addCell(createCell("Value"));
    
    table.addCell(createCell("Key"));
    table.addCell(createCell("Value"));
    
    table.addCell(createFooterCell());
    
    document.add(table);
    
    for (int i=0; i<200; i++) {
      document.add(new Paragraph("test" + i));
    }
    
    document.close();
  }
  
  private void createGrid(PdfContentByte canvas) {
    for (int i=0; i<9; i++) {
      canvas.moveTo(0, i*100);
      canvas.setLineWidth(1f); 
      canvas.lineTo(595, i*100);
      canvas.stroke();

      for (int j=0; j<10; j++) {
        canvas.moveTo(0, i*100+j*10);
        canvas.setLineWidth(0.5f); 
        canvas.lineTo(595, i*100+j*10);
        canvas.stroke();
      } 
    }
    
    for (int i=0; i<7; i++) {
      canvas.moveTo(i*100, 0);
      canvas.setLineWidth(1f); 
      canvas.lineTo(i*100, 843);
      canvas.stroke();

      for (int j=0; j<10; j++) {
        canvas.moveTo(i*100+j*10, 0);
        canvas.setLineWidth(0.5f); 
        canvas.lineTo(i*100+j*10, 843);
        canvas.stroke();
      } 
    }
  }
  
  private PdfTemplate createTemplate(PdfContentByte canvas) throws IOException, BadElementException, DocumentException {
        PdfTemplate template = canvas.createTemplate(595, 130);
        Graphics2D graphics = template.createGraphicsShapes(595, 100);
        GradientPaint gp1 = new GradientPaint(0, 0, new Color(17, 27, 66), 0, 100, new Color(60, 101, 209), true);
        graphics.setPaint(gp1);
        graphics.fillRect(0, 0, 595, 70);
        graphics.dispose();
    for (int i=0; i<200; i++) {
      Image ribbon = Image.getInstance(RESOURCES_LOCATION + "ribbon.png");
      ribbon.scalePercent(50.0f);
      ribbon.setCompressionLevel(0);
      ribbon.setAbsolutePosition(3*i, 30);
      template.addImage(ribbon);
    }
    Image image = Image.getInstance(RESOURCES_LOCATION + "dva.png");
    image.scalePercent(50.0f);
    image.setCompressionLevel(0);
    image.setAbsolutePosition(267, 0);
    template.addImage(image);
    
    return template;
  }
  
  private PdfPCell createCell(String value) {
    Font helveticaBlack = FontFactory.getFont(FontFactory.HELVETICA, 8, Font.NORMAL, Color.black);
    PdfPCell cell = new PdfPCell(new Phrase(new Chunk(value, helveticaBlack)));
    cell.setBorder(0);
    cell.setBackgroundColor(Color.WHITE);
    return cell;
  }
  
  private PdfPCell createHeaderCell(String value) {
    Font helveticaWhite = FontFactory.getFont(FontFactory.HELVETICA, 10, Font.NORMAL, Color.white);
    PdfPCell headerCell = new PdfPCell(new Phrase(new Chunk(value, helveticaWhite)));
    headerCell.setCellEvent(new PdfRoundCorners(new Color[]{new Color(48,125,201), new Color(7,31,88)}, true, true, false, false));
    headerCell.setColspan (2);
    headerCell.setBorder(0);
    headerCell.setHorizontalAlignment(Element.ALIGN_LEFT);
    headerCell.setPadding(5.0f);
    return headerCell;
  }
  
  private PdfPCell createFooterCell() {
    Font helveticaBlack = FontFactory.getFont(FontFactory.HELVETICA, 8, Font.NORMAL, Color.black);
    PdfPCell footerCell = new PdfPCell(new Phrase(new Chunk("", helveticaBlack)));
    footerCell.setBorder(0);
    footerCell.setColspan (2);
    footerCell.setCellEvent(new PdfRoundCorners(Color.WHITE, false, false, true, true));
    
    return footerCell;
  }
}
