package com.agilex.healthcare.mobilehealthplatform.pdf;

import static org.junit.Assert.assertTrue;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Calendar;
import java.util.Date;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.regex.Pattern;

import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.MessageSource;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;

import com.agilex.healthcare.mbb.datalayer.PdfDataLayer;
import com.agilex.healthcare.mbb.datalayer.PdfDataLayerItext;
import com.agilex.healthcare.mbb.datalayer.PdfGenerationContext;
import com.agilex.healthcare.mobilehealthplatform.domain.DOBDate;
import com.agilex.healthcare.mobilehealthplatform.domain.Patient;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilter;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilterImpl;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.MutableDateFilter;
import com.itextpdf.text.pdf.PRTokeniser;
import com.itextpdf.text.pdf.PdfReader;

@ContextConfiguration(locations = { "classpath:applicationContext.xml"})
@RunWith(SpringJUnit4ClassRunner.class)
public abstract class AbstractPDFReportTest {
  private static final org.apache.commons.logging.Log LOGGER = org.apache.commons.logging.LogFactory.getLog(AbstractPDFReportTest.class);
  
  @Autowired
  MessageSource messageSource;
  
  protected Locale locale = Locale.US;
  
  @Test
  public void testReport() throws IOException {
    PdfGenerationContext context = specifyContext(buildContext());
    ByteArrayOutputStream outputStream = generateReport(context);
    
    byte[] bytes = outputStream.toByteArray();
    outputStream.close();
    
    Iterator<String> iterator = getAssertionTextIterator();
    
    while (iterator.hasNext()) {
      String text = iterator.next();
      InputStream inputStream = new ByteArrayInputStream(bytes);
      
      List<PRTokeniser> tokenisers = parseReport(inputStream);
      
      boolean match = false;
      for (PRTokeniser tokeniser : tokenisers) {
        if (!match && containsText(tokeniser, text)) {
          match = true;
        }

        try {
          tokeniser.close();
        } catch (IOException e) {
          LOGGER.error("Error closing tokeniser", e);
        }
      }

      assertTrue(String.format("(%s) not found", text), match);
    }
  }
  
  public abstract PdfGenerationContext specifyContext(PdfGenerationContext context);
  
  public abstract Iterator<String> getAssertionTextIterator();
  
  protected boolean containsText(PRTokeniser tokeniser, String text) throws IOException {
    boolean match = false;
    while(tokeniser.nextToken()) {
      if (tokeniser.getTokenType() == PRTokeniser.TK_STRING
          && Pattern.compile(Pattern.quote(text), Pattern.CASE_INSENSITIVE).matcher(tokeniser.getStringValue()).find()) {
        match = true;
        break;
      }
    }
    
    return match;
  }
  
  private ByteArrayOutputStream generateReport(PdfGenerationContext context) {
    ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
    
    PdfDataLayer pdfDataLayer = new PdfDataLayerItext();
    pdfDataLayer.generateReport(outputStream, context);
    
    return outputStream;
  }
  
  private List<PRTokeniser> parseReport(InputStream inputStream) throws IOException {
    List<PRTokeniser> tokenisers = new LinkedList<PRTokeniser>();
    
    PdfReader reader = new PdfReader(inputStream);
        
    for (int i = 1; i <= reader.getNumberOfPages(); i++) {
      byte[] streamBytes = reader.getPageContent(i);
      PRTokeniser tokenizer = new PRTokeniser(streamBytes);
          tokenisers.add(tokenizer);
    }
        
    inputStream.close();
    reader.close();
        
    return tokenisers;
  }
  
  public PdfGenerationContext buildContext() {
    PdfGenerationContext context = new PdfGenerationContext();

    context.setPatientIdentifier(createPatientIdentifier());
    context.setDateFilter(createDateFilter());
    if (messageSource == null) {
      throw new RuntimeException();
    }
    context.setMessageSource(messageSource);
    context.setLocale(locale);
    Patient patient = new Patient();
    patient.setFirstName("testFirst");
    patient.setLastName("testLast");
    patient.setDateOfBirth(new DOBDate(new Date()));
    context.setPatient(patient);
    context.setReportDate(new Date());
    
    return context;
  }
  
  protected PatientIdentifier createPatientIdentifier() {
    PatientIdentifier patientIdentifier = new PatientIdentifier();
    patientIdentifier.setAssigningAuthority("default");
    patientIdentifier.setUniqueId("D123401");
    return patientIdentifier;
  }
  
  protected DateFilter createDateFilter() {
    Calendar twoYearsAgo = Calendar.getInstance();
    twoYearsAgo.add(Calendar.YEAR, -2);
    Date startDate = twoYearsAgo.getTime();
    Date endDate = new Date();
    
    MutableDateFilter dateFilter = new DateFilterImpl();
    dateFilter.setStartDate(startDate);
    dateFilter.setEndDate(endDate);
    return dateFilter;
  }
}
