/* JS Document */

/**
*
* @project APPOINTMENTS: Mobile Blue Button
* @version v0.9
* @package javascript/ jquery-mobile, backbone.js, underscore.js
* @author Agilex
* @jsdoc version 1
*
*/

(function($){
	/****** VIEWS ******/
	m.NotificationsView = Backbone.View.extend({
		tagName: 'ul',
		attributes: {"data-role": "listview"},
		
		initialize: function() {
			this.template = _.template($('#notifications-template').html());
		},

		render: function() {
			var template = this.template,
				container = this.options.viewContainer,
				appointmentRequests = this.collection,
				notifications = $(this.el);
			notifications.empty();

			appointmentRequests.each(function(appointmentRequest){
				var changedRequest = m.accessChanges(appointmentRequest);
				if (changedRequest !== false) {
					notifications.append(template(appointmentRequest.toJSON()));
				}
			});
			if ($("li:first-child", notifications).length > 0){
				$.mobile.changePage('#notifications');
				this.options.viewContainer
					.html(notifications)
					.trigger('create');
			}

			return this;
		}
	});

	m.AppointmentRequestStatusFilterView = Backbone.View.extend({
		initialize: function() {
			this.template = _.template($('#status-filter-template').html());
		},

		render: function() {
			this.options.viewContainer
				.html(this.template(this.model.toJSON()))
				.trigger('create');
			return this;
		}

	});

	m.AppointmentRequestsListView = Backbone.View.extend({
		tagName: 'ul',
		id: 'appointment-request-list',
		attributes: {"data-role": 'listview', "data-divider-theme": 'd' },
		
		initialize: function() {
			this.collection.bind('add', this.add, this);
			this.collection.bind('change', this.changeItem, this);
			this.collection.bind('reset', this.render, this);
			this.template = _.template($('#appointment-request-list-item-template').html());
		},
		
		render: function() {
			var appointmentRequests = this.collection,
				template = this.template,
				status = this.options.status,
				listView = $(this.el),
				changedRequests = $('#list').jqmData('changedAR');
			listView.empty(); 

			appointmentRequests.each(function(appointmentRequest){
				
				if (status == 'All' || appointmentRequest.get("status") == status) {
				
					listView.append(template($.extend({}, appointmentRequest.toJSON(), {cid: appointmentRequest.cid}))); //add cid to appointmentRequest.tojson() 

					if (_.indexOf(changedRequests, appointmentRequest.get('dataIdentifier').uniqueId) > -1) {
						$('#' + appointmentRequest.cid, listView).parent('li').addClass('ui-btn-up-e');
					}
				}
			});
			this.options.viewContainer
				.html(listView)
				.trigger('create');
			return this;
		},
		changeItem: function(item){
			
		},
		add: function(item){
			//this.collection.sort();
		}
	});

	m.AppointmentListView = Backbone.View.extend({
		tagName: 'ul',
		id: 'appointment-list',
		attributes: {"data-role": 'listview', "data-divider-theme": 'a' },
		
		initialize: function() {
			this.collection.bind('add', this.add, this);
			this.collection.bind('change', this.changeItem, this);
			this.collection.bind('reset', this.render, this);
			this.template = _.template($('#appointment-list-item-template').html());
			this.headerTemplate = _.template($('#list-view-header-template').html());
		},

		render: function() {
			var container = this.options.viewContainer,
				appointments = this.collection,
				template = this.template,
				headerTemplate = this.headerTemplate;
				listView = $(this.el);
			$(this.el).empty(); //I think el is the instance of a UL that is in context to the view
			
			listView.append(headerTemplate({heading: "Appointments within next 120 days"}));
			
			appointments.each(function(appointment){
				listView.append(template($.extend({}, appointment.toJSON(), {cid: appointment.cid}))); //add cid to appointmentRequest.tojson()
			});
			container.html($(this.el));
			container.trigger('create');
			return this;
		}
	});

	m.FooterView = Backbone.View.extend({
		tagName: 'div',
		attributes: {"data-role": 'navbar'},
		initialize: function() {
			this.template = _.template($('#footer-template').html());
			this.userTemplate = _.template($('#user-info-template').html());
		},
		render: function() {
			var navigation = this.collection,
				template = this.template,
				user = this.model,
				userTemplate = this.userTemplate,
				footerView = this.el;
			
			$(footerView).append('<ul>'); 
			navigation.each(function(NavigationLink){
				$('ul', footerView).append(template(NavigationLink.toJSON()));
			});

			this.options.viewContainer
				.empty()
				.append(footerView)
				.append(userTemplate(user.toJSON()))
				.trigger('create');

			return this;
		}
	});

	m.AboutView = Backbone.View.extend({
		initialize: function() {
			this.template = _.template($('#about-template').html());
		},
		render: function() {
			this.options.viewContainer
				.empty()
				.append(this.template())
				.trigger('create');
			return this;
		}
	});


	m.FeedbackView = Backbone.View.extend({
		initialize: function() {
			this.template = _.template($('#feedback-template').html());
		},
		render: function() {
			var firstQuesttionOptions = this.options.metadata.get("appointmentProcessedAsPerUserNeedOptions");

			this.options.viewContainer
			 	.empty()
			 	.append(
					this.template({
						appointmentProcessedAsPerUserNeedOptions: firstQuesttionOptions
					})
	 			)
				.trigger('create');
			return this;
		}
	});

	m.AppointmentRequestFormView = Backbone.View.extend({

		attributes: {"data-role": 'fieldcontain'},
		
		initialize: function() {
			this.template = _.template($('#request-form-template').html());
		},
		render: function() {
			var renderedContent = this.template($.extend({}, this.model.toJSON(), this.options.init.toJSON(),fetchNotificationPreferences(m.patient).toJSON()));

			this.options.viewContainer
				.html(renderedContent)
				.trigger('create');

			new m.NotificationPreferenceSettingsListView({
				model: fetchNotificationPreferences(m.patient),
				viewContainer: $('#request-form-notification-preferences')
			}).render();

			return this;
		}
	});

    m.NotificationPreferencesFormView = Backbone.View.extend({

        attributes: {"data-role": 'fieldcontain'},

        initialize: function() {
            this.template = _.template($('#notification-preferences-form-template').html());
        },
        render: function() {
			var renderedContent = this.template($.extend({}, this.model.toJSON()));

			this.options.viewContainer
				.html(renderedContent)
				.trigger('create');

			setNotificationPreferencesFields(this.model);

			return this;
        }
    });

    m.NotificationPreferenceSettingsListView = Backbone.View.extend({

        attributes: {"data-role": 'fieldcontain'},

        initialize: function() {
            this.template = _.template($('#notification-preferences-status-template').html());
		},
		render: function() {
			var renderedContent = this.template($.extend({}, this.model.toJSON()));

			this.options.viewContainer
				.html(renderedContent)
				.trigger('create');

			return this;
        }
    });


	m.AppointmentRequestDetailsView = Backbone.View.extend({
		initialize: function() {
			this.template = _.template($('#appointment-request-details-template').html());
		},
		render: function() {
			var $container = this.options.viewContainer,
				appointmentRequest = this.model,
				renderedContent;

			if (appointmentRequest.get('providerOption') === undefined || appointmentRequest.get('providerOption') === '') {
				appointmentRequest.set('providerOption', '');
				renderedContent = this.template($.extend({}, this.model.toJSON(), {hidden: 'display-none-important'}));
			}
			else {
				renderedContent = this.template($.extend({}, this.model.toJSON(), {hidden: ''}));
			}
			
			var statusTemplate = _.template($('#appointment-request-status-template').html());
			$container
				.html(statusTemplate($.extend({}, this.model.toJSON())));

			if (appointmentRequest.get('status') === 'Not Booked') {
				var secondRequestMessageTemplate = _.template($('#second-request-message-template').html()),
					secondRequestInputsTemplate = _.template($('#second-request-inputs-template').html());

				$container
					.prepend(secondRequestMessageTemplate($.extend({}, this.model.toJSON(), {hidden:''})))
					.append(secondRequestInputsTemplate($.extend({}, this.model.toJSON())));
			}

			$container
				.append(renderedContent);

			var	$messages = $container.find('.messages').empty(),
				$additionalRequestDetailsHeading = $container.find('h3:contains("Additional Request Details")'),
				$composeMessageFieldset = $container.find('.compose-message'),
				$newMessage = $composeMessageFieldset.find('#newMessage'),
				$sendMessageBtn = $composeMessageFieldset.find('#sendMessage');

			flattenLinks(appointmentRequest);
			
			if (typeof appointmentRequest.get('appointment-request-messages') !== 'undefined') {
				$('h3:contains("Request Messages"), .compose-message, .messages').show();
				var messagesUrl = appointmentRequest.get('appointment-request-messages').href,
					aptMessages = new m.AppointmentRequestMessage({url: messagesUrl });
				aptMessages.fetch({
					async: false,
					cache: false,
					url: messagesUrl
				});
				
				var messages = aptMessages.get('appointmentRequestMessage'),
					messageDateTime,
					lastSent = null;

				$additionalRequestDetailsHeading.add($composeMessageFieldset).add($messages).show();

				$newMessage.keyup(function(){
					$sendMessageBtn
						.button($(this).val().trim() !== '' ? 'enable' : 'disable');
				});

                //clear the read flag for this appointment request
                $.ajax(
                    {
                        type: "POST",
                        url: appointmentRequest.get('appointment-request-new-message-flag').href,
                        data: "{}",
                        contentType: "application/json; charset=utf-8",
                        dataType: "json",
                        success: function (data) {
                           //console.log("Cleared read flag for request " + appointmentRequest.get('dataIdentifier').uniqueId)
                        },
                        error: function (err) {
                            alert('Error clearing read messages flag for appointment request' + appointmentRequest.get('dataIdentifier').uniqueId + ". Error was " + err);
                        }
                    });



				_.each(messages, function(message){
					messageDateTime = message.messageDateTime.split(':').splice(0,2).join(':');

					$messages
						.append(
							'<span>' + [messageDateTime !== lastSent ? messageDateTime : ''] + '</span>' +
							'<div class="' + [message.senderId ==  m.user.get('id')
								? 'sent'
								: 'received'] + 
							'"><p>' + message.messageText + '</p></div>');

					lastSent = messageDateTime;
				});

				/*
					No messages allowed after:
					(a)	Appointment Date has passed
					(b)	Appointment has been cancelled
					(c) BusinessDays rule
				*/
/*
					No messages allowed after:
					(a)	Appointment Date has passed
					(b)	Appointment has been cancelled
					(c) BusinessDays rule
				*/
				var preventMessageScenarios = function() {
					var today = new Date(),
						appointmentDate = new Date(appointmentRequest.get('appointmentDate')),
						dayOfWeek = appointmentDate.getDay(),
						considerWeekend = dayOfWeek === 1 ? 2 : 0,
						oneBusinessDaysBefore = new Date(new Date(appointmentDate).setDate(appointmentDate.getDate()-considerWeekend)),
						twoBusinessDaysBefore = new Date(new Date(appointmentDate).setDate(appointmentDate.getDate()-considerWeekend-1)),
						intiatingMessage = $('.received').length <= $('.sent').length;

						today.setHours(0);
						today.setMinutes(0);
						today.setSeconds(0);
						today.setMilliseconds(0);

					//IMPORTANT .val('') needed to reset placeholder in IE9, IE10

					if (appointmentRequest.get('status') === 'Cancelled')
						return $newMessage.prop({
							placeholder: 'Appointment has been cancelled',
							disabled: true
						}).val('');

					if (today.getTime() > appointmentDate.getTime() ) 
						return $newMessage.prop({ 
							placeholder: 'Appointment Date has passed',
							disabled: true
						}).val('');

					if (intiatingMessage && today.getTime() > twoBusinessDaysBefore.getTime())
						return $newMessage.prop({ 
							placeholder: 'Please Call. Two business days required to initiate a message',
							disabled: true
						}).val('');

					if (!intiatingMessage && today.getTime() >= oneBusinessDaysBefore.getTime() && today.getTime() <= appointmentDate.getTime())
						return $newMessage.prop({ 
							placeholder: 'Please Call. One business day required to respond to a message',
							disabled: true
						}).val('');
				};

				if (appointmentRequest.get('appointmentDate') !== "")
					preventMessageScenarios();

				var sendMessage = function() {

					var	messageText = $newMessage.val().trim();

					$(this).button('disable');

					if (messageText.length>0) {
						var msgDetails = {
							messageText: messageText,
							AppointmentRequestId: appointmentRequest.get('dataIdentifier').uniqueId,
							senderId: m.user.get('id')
						};

						aptMessages.save(msgDetails, {
							url: messagesUrl,
							success: function(resp) {
								var messageDateTime = resp.get("messageDateTime").split(':').splice(0,2).join(':'),
									lastSent = $messages.find('span').eq(0).text();
								$newMessage.val('');
								$messages
									.prepend('<span>' + [messageDateTime !== lastSent ? messageDateTime : ''] + '</span><div class="sent"><p>' + msgDetails.messageText + '</p></div>');
							},

							error: function() {
								$newMessage.val('');
							},

							statusCode: {
								302 : function() { 
									loginRedirect();
								}
							}
						});
					} else {
						//Empty / Space(s) / Tab(s)
						$newMessage.val('');
					}
				};

				//will go in events: in new AMD structure
				$sendMessageBtn.on('tap', sendMessage );
			} else {
				$additionalRequestDetailsHeading.add($composeMessageFieldset).add($messages).hide();
			}

			var notificationPreferences = _.template($('#notification-preferences-status-template').html());
			$container
				.append("<h3>Notification Preferences</h3>")
				.append(notificationPreferences($.extend({},fetchNotificationPreferences(m.patient).toJSON())))
				.trigger('create');

			return this;
		}

	});

	m.ErrorView = Backbone.View.extend({
		initialize: function() {
			this.template = _.template($('#error-form-template').html());
		},
		render: function() {
			var renderedContent = this.template(this.model.toJSON());

			this.options.viewContainer
				.append(renderedContent)
				.trigger('create');

			return this;
		}
	});

	m.SuccessView = Backbone.View.extend({
		initialize: function() {
			this.template = _.template($('#success-form-template').html());
		},
		render: function() {
			var renderedContent = this.template({message: this.options.message});
				
			this.options.viewContainer
				.prepend(renderedContent)
				.trigger('create');

			return this;
		}
	});

	m.ConfirmCancellationContentView = Backbone.View.extend({
		initialize: function() {
			this.template = _.template($('#confirm-cancellation-content-template').html());
		},
		render: function() {
			var renderedContent = this.template(this.model.toJSON());

			this.options.viewContainer
				.html(renderedContent)
				.trigger('create');

			return this;
		}
	});

	m.ProviderSelectView = Backbone.View.extend({
		render: function() {
			var providers = this.model,
				optionList = "<option value=\"Select\">Select</option>";

			_.each(providers, function(provider) {
				var optionName = provider.providerName;
				if (provider.personClass !== '') {
					optionName += ' (' + provider.personClass + ')';
				}
				
				var option = "<option value=\"" + provider.providerId + "\">" + optionName + "</option>";
				optionList += option;
			});
			optionList += "<option value=\"0\">Provider that I want to see is not listed</option>";

			this.options.viewContainer
				.html(optionList)
				.selectmenu('refresh', true)
				.trigger('create');

			$('#providerContainer').removeClass('display-none');
			m.resetProviderOption();

			return this;
		}
	});

	/****** Routers ******/
	m.initData = function(){
		m.userFeeds = new m.UserFeeds();
		m.patientLinks = new m.PatientLinks();
		m.patientLinks.fetch({
			async: false,
			cache: false
		});
		m.patientIdentifiers = new m.PatientIdentifiers();
		m.patientIdentifiers.fetch({
			async: false,
			cache: false
		});
		m.requestMetaData = new m.RequestMetaData({
			async: false,
			cache: false
		});
		m.requestMetaData.fetch({
			async: false,
			cache: false
		});
		m.navigation = new m.Navigation();
		m.navigation.fetch({
			async: false,
			cache: false
		}); // CORRECT ME: Needs to async, should fetch in callback
		m.primaryCareProviders = new m.PrimaryCareProviders();
		m.mentalHealthProviders = new m.MentalHealthProviders();
		m.facilities = new m.Facilities();
		m.mostRecentFacilities = new m.Facilities();
		m.patient = new m.Patient();
	};
	
	/****** Functions ******/
	function containsDetailCode(ardcList, detailCode) {
		for (var i = 0; i < ardcList.length; i++) {
			if (ardcList[i].detailCode.code === detailCode)
				return true;
		}

		return false;
	}

	function convertFacilityToAutocompleteItem(facility, itemCategory) {
		var labelString = facility.get('name') + ' - ' + facility.get('city') + ' - ' + facility.get('state');

		return {
			label: labelString,
			value: facility.get('name'),
			category: itemCategory
		};
	}

	m.accessChanges = function(request) {

		var lastAccessDate = new Date(request.collection.lastAccessDate),
			status = request.get('status');

		if ((lastAccessDate < new Date(request.get('lastUpdatedDate'))) && (status !== 'Submitted')) {
			var ardcList = request.get('appointmentRequestDetailCode');
			if ((status !== 'Cancelled') || ((typeof ardcList !== 'undefined') && ardcList.length > 0 && !containsDetailCode(ardcList, "DETCODE8"))) {
				var changed = $('#list').jqmData('changedAR');

				changed = (changed === null || changed === undefined) ? [] : changed;
				changed.push(request.get('dataIdentifier').uniqueId);
				changed = _.uniq(changed);

				$('#list').jqmData('changedAR', changed);
				return true;
			}
		}
		return false;
	};

	m.doNotShowNotice = function() {
		if (Modernizr.localstorage){
			if (window.localStorage.getItem('do-not-show-notice') !== null)
				return window.localStorage.getItem('do-not-show-notice');
		} else {
			$('#do-not-show-notice')
				.siblings('label')
					.remove()
					.end()
				.remove();
		}
		return false;
	};

	m.setProviderOptionDisplay = function(providerId) {
		m.resetProviderOption();
		$('#providerOptionContainer').removeClass('display-none');
		if (providerId != '0') {
			$('#providerListedOptions').removeClass('display-none');
			$('#providerNotListedOptions').addClass('display-none');
		}
		else {
			$('#providerListedOptions').addClass('display-none');
			$('#providerNotListedOptions').removeClass('display-none');
		}
	};

	m.resetProviderOption = function() {
		$('#anyAvailableProvider')
			.attr('checked', false)
			.checkboxradio('refresh');
		
		$('#bookWithOther')
			.attr('checked', false)
			.checkboxradio('refresh');
		
		$('#callBeforeBooking')
			.attr('checked', false)
			.checkboxradio('refresh');

		$('#providerOptionContainer')
			.addClass('display-none');
	};

	m.secondRequestAllowed = function(appointmentRequest) {
		var lessThanSevenDays, 
			secondRequestSubmitted = appointmentRequest.get('secondRequestSubmitted'),
			secondRequest = appointmentRequest.get('secondRequest'),
			firstDay = new Date(appointmentRequest.get('lastUpdatedDate')),
			seventhDay = new Date(subtractFromDateUnformatted(0, -7, 0, 0, 0, 0, 0, firstDay)),
			today = new Date();

		seventhDay.setHours(23);
		seventhDay.setMinutes(59);
		seventhDay.setSeconds(59);
		today.setHours(0);
		today.setMinutes(0);
		today.setSeconds(0);

		lessThanSevenDays = today < seventhDay;

		return (lessThanSevenDays && !secondRequest && !secondRequestSubmitted);
	};

	m.convertFacilitiesToAutocompleteList = function(facilities) {
		var mostRecentListLength = m.mostRecentFacilities.length,
			listLength = facilities.length,
			autocompleteList = [],
			mostRecentFacilitiesCategoryText = "Most Recent Facilities",
			searchResultsText = "Search Results";

		for (var i = 0; i < mostRecentListLength; i++) {
			autocompleteList[i] = convertFacilityToAutocompleteItem(m.mostRecentFacilities.models[i], mostRecentFacilitiesCategoryText);
		}

		for (i = 0; i < listLength; i++) {
			autocompleteList[i + m.mostRecentFacilities.length] = convertFacilityToAutocompleteItem(facilities.models[i], searchResultsText);
		}

		return autocompleteList;
	};

	m.findFacilityByName = function(name) {
		var i = 0;
		
		for (i; i < m.mostRecentFacilities.length; i++) {
			if (name === m.mostRecentFacilities.models[i].get('name'))
				return m.mostRecentFacilities.models[i];
		}

		for (i = 0; i <  m.facilities.length; i++) {
			if (name === m.facilities.models[i].get('name')) {
				return m.facilities.models[i];
			}
		}

		return null;
	};

	m.secondRequestDatesValid = function(firstRequest, secondRequest) {
		var firstOption1 = formatDate(new Date(firstRequest.get('optionDate1'))),
			firstOption2 = formatDate(new Date(firstRequest.get('optionDate2'))),
			firstOption3 = formatDate(new Date(firstRequest.get('optionDate3'))),
			secondOption1 = formatDate(new Date(secondRequest.get('optionDate1'))),
			secondOption2 = formatDate(new Date(secondRequest.get('optionDate2'))),
			secondOption3 = formatDate(new Date(secondRequest.get('optionDate3')));
			
			firstOption1 = firstOption1 + firstRequest.get('optionTime1'),
			firstOption2 = firstOption2 + firstRequest.get('optionTime2'),
			firstOption3 = firstOption3 + firstRequest.get('optionTime3'),
			secondOption1 = secondOption1 + secondRequest.get('optionTime1'),
			secondOption2 = secondOption2 + secondRequest.get('optionTime2'),
			secondOption3 = secondOption3 + secondRequest.get('optionTime3');

		if (firstOption1 != secondOption1 || secondOption2 == secondOption3) {
			return false;
		}

		var firstOptions = [firstOption1, firstOption2, firstOption3],
			secondOptions = [secondOption2, secondOption3];

		for (var i=0; i < firstOptions.length; i++) {
			for (var j=0; j < secondOptions.length; j++) {
				if (firstOptions[i] == secondOptions[j]) {
					return false;
				}
			}
		}

		return true;
	};

	m.setDetailCodeForSave = function(appointmentRequest) {
		var isSecondAttempt = appointmentRequest.has('parentRequest');

		if (isSecondAttempt || appointmentRequest.get('appointmentRequestDetailCode').length === 0)
			appointmentRequest.unset('appointmentRequestDetailCode', {silent: true});
	};

	m.doesPatientIdentifiersContainDfnOrIcn = function(patientIdentifiers) {

		var patientIdentifierList = patientIdentifiers.get('patientIdentifier');
		
		for (var i=0; i < patientIdentifierList.length; i++) {
			if (m.isAssigningAuthorityIcnOrDfn(patientIdentifierList[i].assigningAuthority))
				return true;
		}

		return false;
	};

	m.isAssigningAuthorityIcnOrDfn = function(assigningAuthority) {
		return (assigningAuthority.indexOf('ICN') !== -1 || assigningAuthority.indexOf('DFN') !== -1);
	};
}(jQuery));


$.extend({
    getUrlVars: function(){
        var vars = [], hash;
        var hashes = window.location.href.slice(window.location.href.indexOf('?') + 1).split('&');
        for(var i = 0; i < hashes.length; i++)
        {
            hash = hashes[i].split('=');
            vars.push(hash[0]);
            vars[hash[0]] = hash[1];
        }
        return vars;
    },
    getUrlVar: function(name){
        return $.getUrlVars()[name];
    }
});

/****** CONTROLLERS ********/

var displayMaxLength = function(e) {
	var key = e.which,
		$me = $(e.currentTarget);
		$sendMessage = $('#sendMessage'),
		$submitNewRequest = $('#review-request'),
		max = $me.attr('data-maxlength'),
		len = $me.val().length,
		balance = max - len,
		overdrawn = balance < 0;

	$me
		.parent()
			.attr('data-balance', balance);

	if($sendMessage.length) {
		$sendMessage.button(
			overdrawn ? 'disable' : 'enable'
		);
	} else {
		//triggers form validation for new request
		$me.change()
	}
};

$('body')
	.on('keyup', '#initialRequestMessage', function(e){
		displayMaxLength(e);
	});
$('body').on('pageinit','#list', function(event){
	var roaPageDisplayed = false;

	globalInit(function(){
		if (!checkForROA()){
			roaPageDisplayed = true;
		}
	});

    if (typeof $.getUrlVars('token')["token"] === 'undefined') {
	if (!roaPageDisplayed){
		m.initData();
		$('body').on('pagebeforeshow','#list', function(event){
			var $this = $(this);
				collection = m.appointmentRequests,
				currentStatus = $('body').jqmData('statusFilter');

			if (currentStatus === undefined) {
				currentStatus = 'All';
			}


			m.appointmentRequests = new m.AppointmentRequests();
			m.appointmentRequests.fetch({
				cache: false,
				success: function(collection) {
					new m.NotificationsView({
						collection: collection,
						viewContainer: $('#notifications-list')
					}).render();

					new m.AppointmentRequestsListView({
						collection: collection,
						viewContainer: $this.find(":jqmData(role='content')"),
						status: currentStatus
					}).render();

					$('#status-filter-container select')
						.val(currentStatus)
						.selectmenu('refresh', true);
				},
				statusCode: {
					302 : function() { 
						loginRedirect();
					}
				}
			});

			new m.FooterView({
				collection: m.navigation,
				model: m.user,
				viewContainer: $this.find(':jqmData(role="footer")')
			}).render();

			new m.AppointmentRequestStatusFilterView({
				model: m.requestMetaData,
				viewContainer: $("#status-filter-container")
			}).render();
		});
	}

    }
});

$('#status-filter-container').on('change', 'select', function(event){
	var currentStatus = this.value;

	$('body').jqmData('statusFilter', currentStatus);

	new m.AppointmentRequestsListView({
		collection: m.appointmentRequests,
		viewContainer: $("#list").find(":jqmData(role='content')"),
		status: currentStatus
	}).render();
});

$('body').on('tap', '#new-request-btn', function(){
	if(m.doesPatientIdentifiersContainDfnOrIcn(m.patientIdentifiers)) {
		var appointmentRequest = new m.AppointmentRequest(),
			newRequest = true;
		$('body').on('pagebeforeshow','#new-request', function(){
			var minDays = m.requestMetaData.get('earliestTimeVeteranCanRequestAppointmentInDays'),
				maxDays = m.requestMetaData.get('farthestTimeVeteranCanRequestAppointmentInDays');
			if (newRequest) {
				new m.AppointmentRequestFormView({
					model: appointmentRequest,
					init: m.requestMetaData,
					viewContainer: $('#request-form-form')
				}).render();
				
				new m.FooterView({
					collection: m.navigation,
					model: m.user,
					viewContainer: $('#new-request-footer')
				}).render();
				
				m.mostRecentFacilities.fetch({
					cache: false,
					url: m.userFeeds.get("most-recent-facilities").get("href"),
					statusCode: {
						302 : function() { 
							loginRedirect();
						}
					}
				});

				$providerContainer = $('#providerContainer').addClass('display-none');

				$('#otherPurposeOfVisitContainer').addClass('display-none');

				$.widget("custom.catcomplete", $.ui.autocomplete, {
					_renderMenu: function( ul, items ) {
						var self = this,
							currentCategory = "";
						$.each( items, function( index, item ) {
							if ( item.category != currentCategory ) {
								ul.append( "<li class='ui-autocomplete-category'>" + item.category + "</li>" );
								currentCategory = item.category;
							}
							self._renderItem( ul, item );
						});
					}
				});

				$('#facilityName').catcomplete({
					delay: 100,
					minLength: 2,
					source: function( request, response ) {
						var generatedUrl = m.resources.get('facility-service').get('href');
						generatedUrl += "?term=" + request.term + "&app=VARCAR";

						m.facilities.fetch({ 
							cache: false,
							url: generatedUrl,
							success: function(collection) {
								response(m.convertFacilitiesToAutocompleteList(collection));
							},
							statusCode: {
								302 : function() {
									loginRedirect();
								}
							}
						});
					},
					open: function(event, ui) {
						var position = $(this).offset(),
							headerHeight = $('#new-request .ui-header').height();
						$(document).scrollTop(position.top - headerHeight - 10);
					},
					change: function(event, ui) {
						var selectedFacility = this.value,
							parentSiteCode = null,
							facilitySiteCode = null,
							familiar = false,
							isFacilitySet = false,
							appointmentType = $('input[name=appointmentType]:checked').val();

						if (this.value) {
							_.each(m.mostRecentFacilities.models, function(facility) {
								if (selectedFacility === facility.get('name')) {
									parentSiteCode = facility.get('parentSiteCode');
									facilitySiteCode = facility.get('facilityCode');
								}
							});
							
							if (parentSiteCode === null ) {
								_.each(m.facilities.models, function(facility) {
									if (selectedFacility === facility.get('name')) {
										parentSiteCode = facility.get('parentSiteCode');
										facilitySiteCode = facility.get('facilityCode');
									}
								});
							}
	
							if (parentSiteCode !== null) {
								var patientIdentifiersLength = m.patientIdentifiers.get('patientIdentifier').length;
								for (var i = 0; i < patientIdentifiersLength; i++) {
									if (m.patientIdentifiers.get('patientIdentifier')[i].assigningAuthority.indexOf(parentSiteCode) === 4) {
										familiar = true;
										break;
									}
								}

								if (typeof appointmentType !== 'undefined' && appointmentType === 'Mental Health' && facilitySiteCode)
									isFacilitySet = true;
							} else {
								familiar = true;
								this.value = '';
								$.mobile.changePage('#invalid-facility-dialog');
							}
						} else {
							$providerContainer.addClass('display-none');
							familiar = true;
						}

						if (isFacilitySet) {
							m.primaryCareProviders = new m.PrimaryCareProviders();
							m.primaryCareProviders.setFacilitySiteCode(facilitySiteCode);
							m.primaryCareProviders.fetch({
								async: false,
								cache: false
							});

							m.mentalHealthProviders = new m.MentalHealthProviders();
							m.mentalHealthProviders.setFacilitySiteCode(facilitySiteCode);
							m.mentalHealthProviders.fetch({
								async: false,
								cache: false
							});
							

							if (typeof appointmentType !== 'undefined') {
								var providers = m[appointmentType === 'Mental Health' 
													? 'mentalHealthProviders'
													: 'primaryCareProviders'].get('provider');
 
								new m.ProviderSelectView({
									model: providers,
									viewContainer: $('#providerId')
								}).render();
							}
						}
						
						if (!familiar) {
							this.value = '';
							$.mobile.changePage('#unfamiliar-facility-dialog');
						}
					}
				});

			}
			newRequest = false;
		});

		$.mobile.changePage(
			m.doNotShowNotice() !== false
				? '#new-request'
				: '#request-notice'
		);

	} else {
		var appointmentMetric = new m.AppointmentMetric();
		appointmentMetric.save({}, {
			url: m.resources.get('no-icn-or-dfn-metric').get('href'),
			success: function() { },
			error: function() { },
			statusCode: {
				302 : function() {
					loginRedirect();
				}
			}
		});
		$.mobile.changePage('#unrecognized-notice');
	}

});

$('body')
	.on('tap', '#dismiss-notice-btn', function(){
		var doNotShowAgain = $('#do-not-show-notice').is(':checked');
		if(Modernizr.localstorage){
			try {
				window.localStorage.setItem('do-not-show-notice', doNotShowAgain);
			} catch (err) {
				if ((err.name).toUpperCase() == 'QUOTA_EXCEEDED_ERR') {
					//ignore error? this means the browser does not allow local storage or has exceeded the storage limit
					//alert("You must have local storage enabled to remember this decision.");
				}
			}
		} 
	})
	.on('tap', '#cancel-request-from-unfamiliar-facility-btn', function(){
		var appointmentMetric = new m.AppointmentMetric();
		appointmentMetric.save({}, {
			url: m.resources.get('no-dfn-cancel-request-metric').get('href'),
			success: function() { },
			error: function() { },
			statusCode: {
				302 : function() { 
					loginRedirect();
				}
			}
		});
		$.mobile.changePage('#list');
	}).on('pagebeforeshow', '#schedule-list', function(){
		m.appointments = new m.Appointments();
		m.appointments.fetch({
			cache: false,
			success: function(collection) {
				new m.AppointmentListView({
					collection: collection,
					viewContainer: $('#schedule-list').find(":jqmData(role='content')")
				}).render();
			},
			statusCode: {
				302 : function() { 
					loginRedirect();
				}
			}
		});
		
		new m.FooterView({
			collection: m.navigation,
			model: m.user,
			viewContainer: $('#schedule-list-footer')
		}).render();
	});

$('#list').on('tap','.appointment-request-list-item', function(){
	var appointmentRequestId = this.id,
		$body =  $('body'),
		$list = $('#list');
	$('body')
		.jqmData('appReqCid', appointmentRequestId)
		.on('pagebeforeshow', '#appointment-request-details', function(){
			var appointmentRequestId = $body.jqmData('appReqCid'),
				appointmentRequestModel = m.appointmentRequests.getByCid(appointmentRequestId),
				status,
				isSecondRequest;

			status = appointmentRequestModel.get('status');
			isSecondRequest = appointmentRequestModel.get('secondRequest');
			var changed = _.without($list.jqmData('changedAR'), appointmentRequestModel.get('dataIdentifier').uniqueId);
			$list.jqmData('changedAR', changed);

			if (status == 'Submitted' || (status == 'Booked' && (new Date(appointmentRequestModel.get('appointmentDate')) > new Date() || isSecondRequest))) {
				$('#cancel-request-btn').removeClass('display-none-important');
				$('#second-request-btn').addClass('display-none-important');
			} else if (status == 'Not Booked') {
				$('#cancel-request-btn').addClass('display-none-important');
				if (m.secondRequestAllowed(appointmentRequestModel)) {
					$('#second-request-btn').removeClass('display-none-important');
				} else {
					$('#second-request-btn').addClass('display-none-important');
				}
			} else {
				$('#cancel-request-btn, #second-request-btn').addClass('display-none-important');
			}

			new m.AppointmentRequestDetailsView({
				model: appointmentRequestModel,
				viewContainer: $('#appointment-request-details').find(":jqmData(role='content')")
			}).render();

			new m.FooterView({
				collection: m.navigation,
				model: m.user,
				viewContainer: $('#appointment-request-details-footer')
			}).render();
		});
});

$('#appointment-request-details').on('change', 'input', function() {
	$('#secondRequestServerError, #secondRequestDateError')
		.addClass('display-none');

	var secondRequestDateOption1 = $('#second-request-date1').val(),
		secondRequestDateOption2 = $('#second-request-date2').val(),
		inputsIncomplete = (secondRequestDateOption1 === '' || secondRequestDateOption1 === undefined ||
		secondRequestDateOption2 === '' || secondRequestDateOption2 === undefined ||
		$('#second-request-dates input[type="radio"]:checked').length < 2);

	$('#submit-second-request-btn')
		.button(inputsIncomplete ? 'disable' : 'enable');
});

$('#appointment-request-details')
	.on('tap', '#submit-second-request-btn', function() {

		$("#cancel-second-request-btn, #submit-second-request-btn")
			.button('disable');

		var secondRequestServerError = $('#secondRequestServerError'),
			secondRequestDateError = $('#secondRequestDateError'),
			appointmentRequestId = $('body').jqmData('appReqCid'),
			originalRequest = m.appointmentRequests.getByCid(appointmentRequestId),
			secondRequest = originalRequest.clone(),

			secondRequestDate1 = $('#second-request-date1').val(),
			secondRequestTime1 = $('#second-request-time1 :checked').val(),
			secondRequestDate2 = $('#second-request-date2').val(),
			secondRequestTime2 = $('#second-request-time2 :checked').val(),

			originalDetailCodes = originalRequest.get('appointmentRequestDetailCode'),
			hasSecondAttemptDetailCode = { userId: m.user.get('id'), detailCode: {code: "DETCODE9"} };

		originalDetailCodes.unshift(hasSecondAttemptDetailCode);
		originalRequest.set('appointmentRequestDetailCode', originalDetailCodes);

		secondRequest.set('dataIdentifier', null);
		secondRequest.set('secondRequest', true);
		secondRequest.set('optionDate2', secondRequestDate1);
		secondRequest.set('optionTime2', secondRequestTime1);
		secondRequest.set('optionDate3', secondRequestDate2);
		secondRequest.set('optionTime3', secondRequestTime2);

		if (m.secondRequestDatesValid(originalRequest, secondRequest)) {
			originalRequest.set('secondRequestSubmitted', true);
			secondRequest.set('parentRequest', originalRequest.toJSON());
			
			m.setDetailCodeForSave(secondRequest);
			secondRequest.save({},
				{
					success: function(resp){
						m.appointmentRequests.add(resp);
						$('#secondRequestSuccess').removeClass('display-none');
						setTimeout(function() {
							$.mobile.changePage('#list');
						}, 3000);
					},

					error: function(resp, err){
						secondRequestServerError.removeClass('display-none');
					},

					url: m.userFeeds.get("appointments").get("href"),
					statusCode: {
						302 : function() { 
							loginRedirect();
						}
					}
				}
			);
		}
		else {
			$('#secondRequestDateError').removeClass('display-none');
		}
	}).on('focus', '#newMessage', function(e){
		//in case user just cancelled appointment
		if ($('#appointment-request-details-content div:first p:first').text().indexOf('Cancelled') != -1)
			return $(e.currentTarget).val('').prop({ placeholder: 'Appointment has been cancelled', disabled: true });

		//prevent Excessive Messages From Sender
		if ($('.sent').length >= 2)
			return $(e.currentTarget).prop({ placeholder: 'Can\'t send more than two messages...', disabled: true });

		//prevent Consecutive Messages From Sender
		if ($('.messages div:first-of-type').hasClass('sent'))
			return $(e.currentTarget).prop({ placeholder: 'Must wait for response before sending another message', disabled: true });

	}).on('keyup', '#newMessage', function(e){
		displayMaxLength(e);
	});

$('#appointment-request-details-template')
	.on('keyup', '#initialRequestMessage', function(e){
		displayMaxLength(e);
	});

var preventConsecutiveMessagesFromSender = function(){
		return $('.messages div:first-of-type').hasClass('sent');
	},
	preventExcessiveMessagesFromSender = function(){
		return $('.sent').length >= 2;
	};

$('#request-form-content').on('change', '#purposeOfVisit', function() {
	if (this.value == 'Medication concern') {
		var primaryCareSelected = $('input[name=appointmentType]:checked').val() === 'Primary Care';
		if (primaryCareSelected) {
			$('#visitTypes input[value = "Phone Call"]').attr('checked', 'checked');//.checkboxradio('refresh');
			$('#visitTypes input').checkboxradio('refresh');
		}
	}
});

$('#request-form-content').on('change', 'select, input, textarea', function(){
	var providers,
		appointmentType = $('input[name=appointmentType]:checked').val(),
		facilityName = $('#facilityName').val();
	
	if (appointmentType == 'Primary Care') {
		if (this.id === 'Primary Care') {
			var facilityCode = '';
			if (facilityName) {
				var facility = m.findFacilityByName(facilityName);

				m.primaryCareProviders = new m.PrimaryCareProviders();
				if (facility) {
					facilityCode = facility.get('facilityCode');
				}
			}
			m.primaryCareProviders.setFacilitySiteCode(facilityCode);
			m.primaryCareProviders.fetch({
				async: false,
				cache: false
			});
		}
		providers = m.primaryCareProviders.get('provider');
	} else if (appointmentType == 'Mental Health') {
		if (this.id === 'Mental Health') {
			var facilityCode = '';
			if (facilityName) {
				var facility = m.findFacilityByName(facilityName);
	
				m.mentalHealthProviders = new m.MentalHealthProviders();
				if (facility) { 
					facilityCode = facility.get('facilityCode');
				}
			}
			m.mentalHealthProviders.setFacilitySiteCode(facilityCode);
			m.mentalHealthProviders.fetch({
				async: false,
				cache: false
			});
		}
		providers = m.mentalHealthProviders.get('provider');
	}
	
	if (this.id === 'providerId') {
		m.setProviderOptionDisplay(this.value);
	} else if ((this.name === 'appointmentType' && facilityName) || (this.name === 'facilityName' && appointmentType)) {
		var providerSelectView = new m.ProviderSelectView({model: providers, viewContainer: $('#providerId')});
		providerSelectView.render();
	}

	var phoneCompleted = ($('#phoneNumber').val().length > 0),
		appointmentType = $('input[name=appointmentType]:checked').val(),
		appointmentTypeCompleted = (typeof appointmentType !== 'undefined' && appointmentType.length > 0),
		visitType = $('input[name=visitType]:checked').val(),
		visitTypeCompleted = (typeof visitType !== 'undefined' && visitType.length > 0),
		facilityNameCompleted = facilityName.length > 0 && facilityName !== 'Select',

		firstChoiceCompleted = $('#optionDate1').val().length > 0 && $('#optionTime1').val() !== 'Select',
		secondChoiceCompleted = $('#optionDate2').val().length > 0 && $('#optionTime2').val() !== 'Select',
		thirdChoiceCompleted = $('#optionDate3').val().length > 0 && $('#optionTime3').val() !== 'Select',

		optionTime1 = $('input[name=optionTime1]:checked').val(),
		optionTime1Completed = (typeof optionTime1 !== 'undefined' && optionTime1.length > 0),

		optionTime2 = $('input[name=optionTime2]:checked').val(),
		optionTime2Completed = (typeof optionTime2 !== 'undefined' && optionTime2.length > 0),

		optionTime3 = $('input[name=optionTime3]:checked').val(),
		optionTime3Completed = (typeof optionTime3 !== 'undefined' && optionTime3.length > 0),

		callCount = $('#bestTimetoCall input:checked').length,
		validCount = _.size($('#request-form-content .valid')),
		validChecked = $('#request-form-content .valid:checked'),

		providerCompleted = false,
		providerOptionCompleted = true;

	if ($('#providerId').val() !== 'Select') {
		var providerId = $('#providerId').val(),
			selectedProvider;
		providerCompleted = true;
		
		if (providerId == '0') {
			selectedProvider = { providerName: 'Provider that I want to see is not listed', personClass: ''};
			providerOptionCompleted = $('input[name=providerOption]:checked').length > 0;
		} else {
			_.each(providers, function(provider) {
				if (provider.providerId == providerId)
					selectedProvider = provider;
			});
		}
		
		if (typeof selectedProvider !== 'undefined') {
			$('#providerName').attr('value', selectedProvider.providerName);
			$('#providerPersonClass').attr('value', selectedProvider.personClass);
		}
	}

	var purposeOfVisitCompleted = false,
	// show hide other field for purpose of visit depending on selection
		purposeOfVisit = $('#purposeOfVisit').val();
	if (purposeOfVisit === "Other") {
		if ($('#otherPurposeOfVisit').val().length > 1)
			purposeOfVisitCompleted = true;
		$('#otherPurposeOfVisitContainer').removeClass('display-none');
	} else {
		$('#otherPurposeOfVisit').attr('value', '');
		$('#otherPurposeOfVisitContainer').addClass('display-none');
		
		if (purposeOfVisit.length > 0 && purposeOfVisit !== 'Select')
			purposeOfVisitCompleted = true;
	}

	// validation logic, all required fields must be completed
	$('#review-request').button(
		_.size(validChecked) === validCount && 
			callCount > 0 &&
			phoneCompleted && 
			facilityNameCompleted && 
			appointmentTypeCompleted && 
			providerCompleted && 
			providerOptionCompleted &&
			purposeOfVisitCompleted && 
			visitTypeCompleted && 
			firstChoiceCompleted && 
			secondChoiceCompleted && 
			thirdChoiceCompleted && 
			optionTime1Completed && 
			optionTime2Completed && 
			optionTime3Completed &&
			$('#initialRequestMessage').val().length <= $('#initialRequestMessage').data('maxlength')
		? 'enable'
		: 'disable'
	);

});

$('body').on('tap','#review-request', function(){

	$('#submit-request, #edit-request')
		.removeClass('ui-disabled');

	$('body').on('pagebeforeshow', '#verify-appointment-request', function(){

		var formJSON = $('#request-form-form').formParams(),
			appointmentRequest = new m.AppointmentRequest(formJSON),
			appointmentRequestReviewContainer = $('#review-request-content');

		var facility = m.findFacilityByName(appointmentRequest.get('facilityName'));
		appointmentRequest.set({'facility': facility.toJSON()});

		new m.AppointmentRequestDetailsView({
			model: appointmentRequest,
			viewContainer: appointmentRequestReviewContainer
		}).render();
		
	});
	$.mobile.changePage('#verify-appointment-request');
});

$('#notification-preferences-form-content').on('change', 'select, input, textarea', function(){

    if(this.id === 'authorizeTextMessaging'){
        $('#textMessagingPhoneNumber').parent().removeClass('display-none');
    }
    else if(this.id === 'prohibitTextMessaging'){
        $('#textMessagingPhoneNumber').parent().addClass('display-none');
    }
    if(this.id === 'authorizeEmailMessaging'){
        $('#messagingEmailAddress').parent().removeClass('display-none');
    }
    else if(this.id === 'prohibitEmailMessaging'){
        $('#messagingEmailAddress').parent().addClass('display-none');
    }

    // make email Messaging Phone Number input necessary for review/submission if it is visible
    var emailMessagingCompletedIfDesired = ($("#messagingEmailAddress").is(":hidden") || ($("#messagingEmailAddress").length>0 && $("#messagingEmailAddress").val().trim().length>0) );

    //make Text Messaging Phone Number input necessary for review/submission if it is visible
    var textMessagingCompletedIfDesired = ($('#textMessagingPhoneNumber').is(":hidden") || ($('#textMessagingPhoneNumber').length>0 && $('#textMessagingPhoneNumber').val().trim().length>0));


    // validation logic, all required fields must be completed
    $('#submit-notification-preferences').button(
        textMessagingCompletedIfDesired && emailMessagingCompletedIfDesired ? 'enable' : 'disable'
    );

});

$('body').on('tap','#submit-notification-preferences', function(){

    var notificationPreferences = fetchNotificationPreferences(m.patient),
    	newNotificationPreferences = {};

    newNotificationPreferences.textMsgAllowed = $('#authorizeTextMessaging').is(':checked');
    if(newNotificationPreferences.textMsgAllowed === true){
        newNotificationPreferences.textMsgPhNumber = $('#textMessagingPhoneNumber').val();
    }

    //we only attempt to send new email if button checked.
    newNotificationPreferences.emailAllowed = $('#authorizeEmailMessaging').is(':checked');
    if( newNotificationPreferences.emailAllowed === true){
        newNotificationPreferences.emailAddress = $('#messagingEmailAddress').val();
    }
    newNotificationPreferences.notificationFrequency = $('#notificationFrequency').val();

    notificationPreferences.save(newNotificationPreferences,{async: false,
        error: function (resp, err) {
            console.log(err);
            var $errorViewContainer = $('#notification-preferences-error');
            $('#notification-preferences-error').html('');

            var parsedJSON;
            if(err.responseText !== undefined && err.responseText !== null){
                try{
                    parsedJSON = $.parseJSON(err.responseText);
                }catch(e){//if we don't get JSON, construct an error message from the statusText
                    parsedJSON = {"errors": [{"errorMessage" : err.statusText,"fieldName" : ""}]};
                }
            }else if(err !== undefined && err !== null){
                parsedJSON = {"errors": [{"errorMessage" : err,"fieldName" : ""}]};
            }

            var errorMessages = new m.ErrorMsgs(parsedJSON.errors);

            errorMessages.each(function(error) {
                new m.ErrorView({
                    model: error, viewContainer: $errorViewContainer
                }).render();
            });
        },
        success: function (resp) {

            //refresh notification preferences on new request modal if present
            var notificationsPreferencesContainer = $('#request-form-notification-preferences');

            new m.NotificationPreferenceSettingsListView({
                model: fetchNotificationPreferences(m.patient),
                viewContainer: notificationsPreferencesContainer

            }).render();

            history.back(1);
        }

    });

});

$('body').on('tap','#submit-request', function(){

	$('#submit-request, #edit-request')
		.addClass('ui-disabled');

	var formJSON = $('#request-form-form').formParams(),
		facility,
		successViewContainer = $('#review-request-content'),
		appointmentRequest,
		errorView,
		successView,
		id;

	appointmentRequest = new m.AppointmentRequest(formJSON);

	facility = m.findFacilityByName(appointmentRequest.get('facilityName'));
	appointmentRequest.set({'cid':appointmentRequest.cid}); //create some identifier and bubble up event
	appointmentRequest.set({'facility': facility.toJSON()});
	appointmentRequest.unset("dataIdentifier", {silent: true});

	m.setDetailCodeForSave(appointmentRequest);

	appointmentRequest.save({}, 
	{
		success: function(resp){
			m.appointmentRequests.add(resp);
			successView = new m.SuccessView({viewContainer: successViewContainer, message: 'Request has been submitted successfully. After review by the clinic, updated information will be available.'});
			successView.render();
			setTimeout(function() {
				$.mobile.changePage('#list');
			}, 3000);
			if(appointmentRequest.get("initialRequestMessage") !== undefined && appointmentRequest.get("initialRequestMessage").trim() !== ''){
				saveAppointmentRequestMessage(resp, appointmentRequest.get("initialRequestMessage"));
			}
		},
		error: function(resp, err){
			$('#request-form-form .error:not(#errors-on-page)').remove();

			// TODO: add check for internal server error and display a user friendly message

			var parsedJSON = $.parseJSON(err.responseText),
				errorMsgs = new m.ErrorMsgs(parsedJSON.errors);
			history.back(1);

			var hasUnknownError = false;
			errorMsgs.each(function(error) {
				var field = error.get('fieldName'),
					message = error.get('errorMessage'),
					$errorViewContainer = $('#error');

				if (field === 'error'){
					hasUnknownError = true;
				}

				new m.ErrorView({
					model: error, viewContainer: $errorViewContainer
				}).render();
			});

			if (!hasUnknownError) {
				$('#errors-on-page')
					.removeClass('display-none')
					.addClass('error');
			}
			$('html, body').animate({scrollTop: 0}, 1000);
		},
		url: m.userFeeds.get("appointments").get("href"),
		statusCode: {
			302 : function() {
				loginRedirect();
			}
		}
	});


	function saveAppointmentRequestMessage(appointmentRequest,messageText){
		appointmentRequest = flattenLinks(appointmentRequest);
		var msgDetails = {
				messageText: messageText,
				AppointmentRequestId: appointmentRequest.get('dataIdentifier').uniqueId,
				senderId: m.user.get('id')
			},
			messagesUrl = appointmentRequest.get('appointment-request-messages').href,
			aptMessages = new m.AppointmentRequestMessage();

		aptMessages.save(msgDetails, {
			url: messagesUrl,
			success: function(resp) {
				console.log("Success saving initial appointment request");
			},

			error: function(resp, error) {
				console.log("Error saving initial appointment request")
				alert("Request message was not sent.")
			}
		});
	}
});

$('body').on('pagebeforeshow', '#confirm-cancellation-alert', function() {
	var $body = $('body'),
		appointmentRequestId = $body.jqmData('appReqCid'),
		appointmentRequest = m.appointmentRequests.getByCid(appointmentRequestId);

	$body.jqmData('appReq', appointmentRequest);
	
	new m.ConfirmCancellationContentView({
		model: appointmentRequest,
		viewContainer: $('#confirm-cancellation-content')
	}).render();	
});

$('body').on('tap','#confirm-cancellation-btn', function(){
	var appointmentRequest = $('body').jqmData('appReq'),
		appointmentRequestUri = appointmentRequest.get('self').href;
	
	if (appointmentRequest.get('status') === 'Booked') {
		var appointmentRequestDetailCode = {
				userId: m.user.get('id'),
				detailCode: {
					code: "DETCODE8"
				}
			},
			ardc = appointmentRequest.get('appointmentRequestDetailCode');
		
		if (ardc === null) {
			ardc = [];
			appointmentRequest.set('appointmentRequestDetailCode', ardc);
		}
		
		ardc.unshift(appointmentRequestDetailCode);
	}
	
	m.setDetailCodeForSave(appointmentRequest);
	appointmentRequest.save(
			{ 
				id: appointmentRequest.get('dataIdentifier').uniqueId, 
				status: 'Cancelled'
			}, 
			{
				url: appointmentRequestUri, 
				error: function(err, resp) {
						alert("Error canceling request.  Please try again later or contact technical support for assistance.");
						$('#confirm-cancellation-alert').dialog('close');
					}, 
				success: function() {
						$('#confirm-cancellation-alert').dialog('close');
						$('#cancel-request-btn').addClass('display-none-important');
					},
				statusCode: {
					302 : function() { 
						loginRedirect();
					}
				}
			});
	appointmentRequest.set('appointmentRequestDetailCode', []);
});

$('body').on('tap','#second-request-btn', function() {
	$('#second-request-dates').removeClass('display-none');
	$('#cancel-second-request-btn').button('enable');
	$('#submit-second-request-btn').button('disable');
	$('#second-request-btn').addClass('display-none-important');
});

$('body').on('tap','#cancel-second-request-btn', function() {
	$('#submit-second-request-btn, #cancel-second-request-btn').button('disable');
	$('#second-request-dates').addClass('display-none');
	$('#second-request-btn').removeClass('display-none-important');
});

$('body').on('pagebeforeshow', '#feedback', function() {
	$('#submit-feedback').button('disable');

	new m.FeedbackView({
		viewContainer: $(this).find(":jqmData(role='content')"),
		metadata: m.requestMetaData
	}).render();	
});

$('#feedback').on('change', 'input[type="radio"]', function() {
	$('#submit-feedback').button('enable');
});

$('#feedback').on('tap', '#submit-feedback', function() {
	$('#submit-feedback, #cancel-feedback')
		.button('disable');

	var $feedback = $('#feedback'),
		formJSON = $('#feedback-form').formParams(),
		feedbackModel = new Backbone.Model(formJSON);

	feedbackModel.save({}, {
		url: m.resources.get('feedback-appointment-request').get('href'),
		error: function() {},
		success: function(){
			
			new m.SuccessView({
				viewContainer: $feedback.find(":jqmData(role='content')"),
				message: 'Thank you!  Your feedback has been submitted.'
			}).render();
			setTimeout(function() {
				$feedback.dialog('close');
				$('#cancel-feedback').button('enable');
			}, 2000);
		},
		statusCode: {
			302 : function() { 
				loginRedirect();
			}
		}
	});
});

$('body').on('pagebeforeshow', '#notification-preferences', function() {

    new m.NotificationPreferencesFormView({
        model: fetchNotificationPreferences(m.patient),
        viewContainer: $(this).find(":jqmData(role='content')")
    }).render();

});

$('body').on('pagebeforeshow', '#about', function() {
    new m.AboutView({
        viewContainer: $(this).find(":jqmData(role='content')")
    }).render();
});


$('body').on('change', '#visitType input', function(event) {
	$("#videoConferenceCaveatText")
		[$('[id="Video Conference"]').is(":checked") ? "removeClass" : "addClass"]("display-none");
});


/***** FUNCTIONS ******/


function flattenLinks(object) {
    var links = object.attributes.link;

    if (typeof links !== 'undefined') {
        for (var i =0; i < links.length; i++) {
            if(links[i].title !== undefined)
                object.set(links[i].title,links[i]);

            if(links[i].rel === "self")
                object.set("self",links[i]);
        }
        object.unset('link');
    }

    return object;
};

function fetchNotificationPreferences(patientModel) {
    var notificationPreferences = new m.NotificationPreferences();

    patientModel.fetch({
    	async: false,
    	cache: false,
        success: function (resp) {
            var resp = flattenLinks(resp);
            notificationPreferences.url = resp.get("patient-preference").href;

            notificationPreferences.fetch({
                async: false,
                cache: false,
                error: function(err){
                    alert("Error loading notification preferences. Error: "+ err);
                }
            });
        },
        error: function(err){
            alert("Error loading patient data. Error: " + err);
        }
    });

    return notificationPreferences;
}

function setNotificationPreferencesFields(notificationPreferences){

    var textMessagingAllowed = notificationPreferences.get('textMsgAllowed'),
        textMessagingPhoneNumber = notificationPreferences.get('textMsgPhNumber'),
        emailMessagingAllowed = notificationPreferences.get('emailAllowed'),
        messagingEmailAddress = notificationPreferences.get('emailAddress');

    if (typeof textMessagingAllowed !== 'undefined' && textMessagingAllowed) {
        $('#authorizeTextMessaging')
            .trigger('click')
            .checkboxradio('refresh');
        $('#textMessagingPhoneNumber').parent().removeClass('display-none');
    } else {
        $('#prohibitTextMessaging')
            .trigger('click')
            .checkboxradio('refresh');
        $('#textMessagingPhoneNumber').parent().addClass('display-none');
    }

    if (typeof textMessagingPhoneNumber !== 'undefined' && textMessagingPhoneNumber) {
        $('#textMessagingPhoneNumber').val(textMessagingPhoneNumber);
    }

    if (typeof emailMessagingAllowed !== 'undefined' && emailMessagingAllowed) {
        $('#authorizeEmailMessaging')
            .trigger('click')
            .checkboxradio('refresh');
        $('#messagingEmailAddress').parent().removeClass('display-none');
    } else {
        $('#prohibitEmailMessaging')
            .trigger('click')
            .checkboxradio('refresh');
        $('#messagingEmailAddress').parent().addClass('display-none');
    }
    if (typeof messagingEmailAddress !== 'undefined' && messagingEmailAddress) {
        $('#messagingEmailAddress').val(messagingEmailAddress);
    }

    $('#notificationFrequency')
        .val(notificationPreferences.get('notificationFrequency'))
        .selectmenu("refresh", true);

}
