
define([
    'jquery',
    'Underscore',
    'veteran-appointment-requests',
    'sinon',
    'sinonQunit'
], function($, Underscore, veteranAppointmentRequests, sinon, sinonQunit) {

    var NOTIFICATION_PREFERENCES_ENDPOINT = "/notificationPreferencesEndpoint";
    var EMAIL_ERROR_MESSAGE = "Email must be between 3 and 50 characters in a valid email format";
    var PHONE_TEXT_ERROR_MESSAGE = "Phone must have a minimum of 10 digits and only contain digits and these characters: + ( ) - . x space";

    var notificationPreferences;
    var errorCallback;

    module('Tests for notification Preferences', {
        setup: function () {

            notificationPreferences = new m.NotificationPreferences();
            errorCallback = sinon.spy();
            notificationPreferences.on('error', errorCallback);


            this.server = sinon.fakeServer.create();
            this.server.respondWith(
                "POST",
                NOTIFICATION_PREFERENCES_ENDPOINT,
                [
                    200,
                    {"Content-Type": "application/json"},
                    '{"response":"a good response"}'
                ]
            );

        },

        teardown: function () {

            this.server = sinon.fakeServer.create();
        }
    });

    test('Notification Preferences default values', function() {
        equal(notificationPreferences.get("notificationFrequency"), "Never");
        equal(notificationPreferences.get("emailAllowed"), false);
        equal(notificationPreferences.get("emailAddress"), "");
        equal(notificationPreferences.get("textMsgAllowed"), false);
        equal(notificationPreferences.get("textMsgPhNumber"), "");
        expect(5);

    });

    test('If emailAllowed and textMsgllowed set to false, we expect no errors', function () {

        var newNotificationPreferences = {};
        newNotificationPreferences.emailAllowed =  false;
        newNotificationPreferences.textMsgAllowed = false;

        notificationPreferences.save(newNotificationPreferences, {async: false, url: NOTIFICATION_PREFERENCES_ENDPOINT});

        ok(errorCallback.notCalled, "Check that error was not raised.");
        expect(1);

    });

    test('If emailAllowed and valid emails set, we expect no errors', function () {

        var newNotificationPreferences = {};
        newNotificationPreferences.emailAllowed = true;
        newNotificationPreferences.emailAddress = "goodemail@goodemail.com";
        notificationPreferences.save(newNotificationPreferences, {async: false, url: NOTIFICATION_PREFERENCES_ENDPOINT});

        newNotificationPreferences.emailAddress = "goodemail@goodemail.net";
        notificationPreferences.save(newNotificationPreferences, {async: false, url: NOTIFICATION_PREFERENCES_ENDPOINT});

        newNotificationPreferences.emailAddress = "good.email@goodemail.net";
        notificationPreferences.save(newNotificationPreferences, {async: false, url: NOTIFICATION_PREFERENCES_ENDPOINT});

        newNotificationPreferences.emailAddress = "good.email123@goodemail.net";
        notificationPreferences.save(newNotificationPreferences, {async: false, url: NOTIFICATION_PREFERENCES_ENDPOINT});

        ok(errorCallback.notCalled, "Check that error was not raised for valid email.");
        expect(1);

    });

    test('If emailAllowed set true and email less than 3 characters set error', function () {

        var newNotificationPreferences = {};
        newNotificationPreferences.emailAllowed = true;
        newNotificationPreferences.emailAddress = "tt";
        notificationPreferences.save(newNotificationPreferences, {async: false, url: NOTIFICATION_PREFERENCES_ENDPOINT});

        ok(errorCallback.calledOnce, "Less than 3 characters marked as error");
        notEqual( errorCallback.getCall(0), undefined, "Checking error was triggered");
        equal( errorCallback.getCall(0).args[1], EMAIL_ERROR_MESSAGE, "Email address length error message shows for length less than 3");

        expect(3);

    });

    test('If emailAllowed set true and email more than 50 characters set error', function () {

        var newNotificationPreferences = {};
        newNotificationPreferences.emailAllowed = true;
        newNotificationPreferences.emailAddress = "ttttttttttt" + "ttttttttttt" + "ttttttttttt" + "ttttttttttt" + "ttttttttttt" + "t"; //51 characters

        notificationPreferences.save(newNotificationPreferences, {async: false, url: NOTIFICATION_PREFERENCES_ENDPOINT});

        ok(errorCallback.calledOnce, "Less than 3 characters marked as error");
        notEqual( errorCallback.getCall(0), undefined, "Checking error was triggered");
        equal( errorCallback.getCall(0).args[1], EMAIL_ERROR_MESSAGE, "Email address length error message shows for length more than 50");

        expect(3);

    });


    test('If emailAllowed set true and email all letters, fail', function () {

        var newNotificationPreferences = {};
        newNotificationPreferences.emailAllowed = true;
        newNotificationPreferences.emailAddress = "ttttttttttt";

        notificationPreferences.save(newNotificationPreferences, {async: false, url: NOTIFICATION_PREFERENCES_ENDPOINT});

        ok(errorCallback.calledOnce, "Less than 3 characters marked as error");
        notEqual( errorCallback.getCall(0), undefined, "Checking error was triggered");
        equal( errorCallback.getCall(0).args[1], EMAIL_ERROR_MESSAGE, "Email address error message for format raised.");

        expect(3);

    });

    test('If emailAllowed set true and email all numbers, fail', function () {

        var newNotificationPreferences = {};
        newNotificationPreferences.emailAllowed = true;
        newNotificationPreferences.emailAddress = "55555555555555";

        notificationPreferences.save(newNotificationPreferences, {async: false, url: NOTIFICATION_PREFERENCES_ENDPOINT});

        ok(errorCallback.calledOnce, "Less than 3 characters marked as error");
        notEqual( errorCallback.getCall(0), undefined, "Checking error was triggered");
        equal( errorCallback.getCall(0).args[1], EMAIL_ERROR_MESSAGE, "Email address error message for format raised.");

        expect(3);

    });

    test('If emailAllowed set true and email has no @, fail', function () {

        var newNotificationPreferences = {};
        newNotificationPreferences.emailAllowed = true;
        newNotificationPreferences.emailAddress = "email.test.com";

        notificationPreferences.save(newNotificationPreferences, {async: false, url: NOTIFICATION_PREFERENCES_ENDPOINT});

        ok(errorCallback.calledOnce, "Email has no @ ");
        notEqual( errorCallback.getCall(0), undefined, "Checking error was triggered");
        equal( errorCallback.getCall(0).args[1], EMAIL_ERROR_MESSAGE, "Email address error message for format raised.");

        expect(3);

    });

    test('If emailAllowed set true and email has no domain, fail', function () {

        var newNotificationPreferences = {};
        newNotificationPreferences.emailAllowed = true;
        newNotificationPreferences.emailAddress = "email@";

        notificationPreferences.save(newNotificationPreferences, {async: false, url: NOTIFICATION_PREFERENCES_ENDPOINT});

        ok(errorCallback.calledOnce, "Email has no domain as error");
        notEqual( errorCallback.getCall(0), undefined, "Checking error was triggered");
        equal( errorCallback.getCall(0).args[1], EMAIL_ERROR_MESSAGE, "Email address error message for format raised.");

        expect(3);

    });

    test('If emailAllowed set true and email has comma in address, fail', function () {

        var newNotificationPreferences = {};
        newNotificationPreferences.emailAllowed = true;
        newNotificationPreferences.emailAddress = "email@email,com";

        notificationPreferences.save(newNotificationPreferences, {async: false, url: NOTIFICATION_PREFERENCES_ENDPOINT});

        ok(errorCallback.calledOnce, "Email has comma");
        notEqual( errorCallback.getCall(0), undefined, "Checking error was triggered");
        equal( errorCallback.getCall(0).args[1], EMAIL_ERROR_MESSAGE, "Email address error message for format raised.");

        expect(3);

    });


    test('Test valid Phone number separators', function () {

        var newNotificationPreferences = {};
        newNotificationPreferences.textMsgAllowed = true;

        newNotificationPreferences.textMsgPhNumber = "333-333-3333";
        notificationPreferences.save(newNotificationPreferences, {async: false, url: NOTIFICATION_PREFERENCES_ENDPOINT});
        ok(errorCallback.notCalled, "No errors for - separator");


        newNotificationPreferences.textMsgPhNumber = "(333)-(333)-(3333)";
        notificationPreferences.save(newNotificationPreferences, {async: false, url: NOTIFICATION_PREFERENCES_ENDPOINT});
        ok(errorCallback.notCalled, "No errors for - and () separator");


        newNotificationPreferences.textMsgPhNumber = "333.333.3333";
        notificationPreferences.save(newNotificationPreferences, {async: false, url: NOTIFICATION_PREFERENCES_ENDPOINT});
        ok(errorCallback.notCalled, "No errors for . separator");

        newNotificationPreferences.textMsgPhNumber = "(333).(333).(3333)";
        notificationPreferences.save(newNotificationPreferences, {async: false, url: NOTIFICATION_PREFERENCES_ENDPOINT});
        ok(errorCallback.notCalled, "No errors for . and () separator");

        newNotificationPreferences.textMsgPhNumber = "333 333 3333";
        notificationPreferences.save(newNotificationPreferences, {async: false, url: NOTIFICATION_PREFERENCES_ENDPOINT});
        ok(errorCallback.notCalled, "No errors for space separator");

        expect(5);

    });

    test('Test valid Phone number with international code', function () {

        var newNotificationPreferences = {};
        newNotificationPreferences.textMsgAllowed = true;

        newNotificationPreferences.textMsgPhNumber = "011 + 44 + 333-333-3333";
        notificationPreferences.save(newNotificationPreferences, {async: false, url: NOTIFICATION_PREFERENCES_ENDPOINT});
        ok(errorCallback.notCalled, "No errors for international code");


        newNotificationPreferences.textMsgPhNumber = "+1 (333)-(333)-(3333)";
        notificationPreferences.save(newNotificationPreferences, {async: false, url: NOTIFICATION_PREFERENCES_ENDPOINT});
        ok(errorCallback.notCalled, "No errors for US Code");

        expect(2);

    });

    test('Test valid Phone number with extension code', function () {

        var newNotificationPreferences = {};
        newNotificationPreferences.textMsgAllowed = true;

        newNotificationPreferences.textMsgPhNumber = "1-(333)-(333)-(3333)x3589";
        notificationPreferences.save(newNotificationPreferences, {async: false, url: NOTIFICATION_PREFERENCES_ENDPOINT});
        ok(errorCallback.notCalled, "No errors for extension");

        expect(1);

    });


    test('If phone text set true and number less than 10 digits, fail', function () {

        var newNotificationPreferences = {};
        newNotificationPreferences.textMsgAllowed = true;
        newNotificationPreferences.textMsgPhNumber = "333-333-333";

        notificationPreferences.save(newNotificationPreferences, {async: false, url: NOTIFICATION_PREFERENCES_ENDPOINT});

        ok(errorCallback.calledOnce, "Less than 10 numbers marked as error");
        notEqual( errorCallback.getCall(0), undefined, "Checking error was triggered");
        equal( errorCallback.getCall(0).args[1], PHONE_TEXT_ERROR_MESSAGE, "Phone text error message for min length raised.");

        expect(3);

    });

    test('If phone text set true and number more than 50 characters set error', function () {

        var newNotificationPreferences = {};
        newNotificationPreferences.textMsgAllowed = true;
        newNotificationPreferences.textMsgPhNumber = "5555555555" + "5555555555" + "5555555555" + "5555555555" + "5555555555" + "5"; //51 characters

        notificationPreferences.save(newNotificationPreferences, {async: false, url: NOTIFICATION_PREFERENCES_ENDPOINT});

        ok(errorCallback.calledOnce, "More than 50 characters marked as error");
        notEqual( errorCallback.getCall(0), undefined, "Checking error was triggered");
        equal( errorCallback.getCall(0).args[1], PHONE_TEXT_ERROR_MESSAGE, "Phone text number length error message shows for length more than 50");

        expect(3);

    });

    test('If phone text set true and number contains alphanumeric', function () {

        var newNotificationPreferences = {};
        newNotificationPreferences.textMsgAllowed = true;
        newNotificationPreferences.textMsgPhNumber = "333-333-33fjklja33";

        notificationPreferences.save(newNotificationPreferences, {async: false, url: NOTIFICATION_PREFERENCES_ENDPOINT});

        ok(errorCallback.calledOnce, "Alphanumerics in phone number marked as error");
        notEqual( errorCallback.getCall(0), undefined, "Checking error was triggered");
        equal( errorCallback.getCall(0).args[1], PHONE_TEXT_ERROR_MESSAGE, "Phone text number contains alphanumerics");

        expect(3);

    });

    test('Test valid Phone number with only 10 numbers', function () {

        var newNotificationPreferences = {};
        newNotificationPreferences.textMsgAllowed = true;

        newNotificationPreferences.textMsgPhNumber = "1112223333";
        notificationPreferences.save(newNotificationPreferences, {async: false, url: NOTIFICATION_PREFERENCES_ENDPOINT});
        ok(errorCallback.notCalled, "No errors for ten consecutive digit phone number");

        expect(1);

    });

});