define([
    'Global',
    'headerView',
    'QuestionResponses',
    'QuestionResponse',
    'Inventory',
    'Confirm',
    'Text!../html/panes/details.html',
    'Text!../html/templates/common/disclaimer.html'
],
    function (Global, HeaderView, QuestionResponses, QuestionResponse, Inventory, Confirm, detailsTemplate, disclaimer) {
        'use strict';
        return Backbone.View.extend({
            id: 'form-question',
            tagName: 'div',
            className: 'alpha',
            categories: null,
            details: null,

            initialize: function (details) {
                this.formTemplate = _.template(detailsTemplate);
                this.disclaimer = _.template(disclaimer);
                this.startTimer = new Date();
                this.inventoryType = details.inventoryType;
                this.categories = details.categories;
                this.details = this.getBaseline(details);
            },

            render: function () {
                if ($('#' + this.el.id).length) {
                    return;
                }
                this.loadForm();

                $("#content")
                    .empty()
                    .append(this.$el)
                    .trigger('create');

                new HeaderView()
                    .render({
                        title: this.details.title,
                        className: 'go-back',
                        leftAction: this.details.leftAction
                    });
            },

            getBaseline: function (details) {
                var qR = new QuestionResponses();

                switch (details.inventoryType) {
                    case "REFLECTIONS":
                        details.questions.baseline = qR.getAnswer(details.questions.id);
                        break;
                    case "FEELINGS":
                        _.each(details.questions, function (q) {
                            q.baseline = qR.getAnswer(q.id);
                        });
                        break;
                    case "PRIORITIES":
                        _.each(details.questions, function (q) {
                            q.area = qR.getAnswer(q.id + '-area');
                            q.goal = qR.getAnswer(q.id + '-goal');
                        });
                        break;
                    default:
                        details.questions.states[0].rating = qR.getAnswer('Current-' + details.questions.id + '-slider');
                        details.questions.states[0].items = qR.getAnswer('Current-' + details.questions.id + '-items');
                        details.questions.states[1].rating = qR.getAnswer('Desired-' + details.questions.id + '-slider');
                        details.questions.states[1].items = qR.getAnswer('Desired-' + details.questions.id + '-items');
                        break;
                }

                return details;
            },

            events: {
                "keypress .reason-new": "returnAdd",
                "keypress .ui-slider-input": "oneToTen",
                "keypress input[maxlength], textarea[maxlength]": "enforceMaxLength",
                "change .ui-slider-input, textarea": "answerUpdated",
                "slidestop .ui-slider-input": "answerUpdated",
                "tap .reason-add": "reasonAdd",
                "tap .reason-omit": "reasonOmit",
                "tap #btn-save": "saveSubmit",
                "tap #btn-no": "dubiousDialog",
                "tap #btn-yes": "continueOn",
                "tap #btn-back": "cancel",
                "change .select-area": "selectedArea",
                "change .select-goal": "selectedGoal"
            },

            answerUpdated: function (e) {
                e.currentTarget.setAttribute('data-updated', true);
            },

            //restrict to number, backspace, delete, right, left arrow keys
            oneToTen: function (e) {
                var keyedValue = $(e.currentTarget).val() + String.fromCharCode(e.which),
                    pattern = new RegExp(/^([1-9]|10)$/);

                if (!pattern.test(keyedValue)) {
                    e.preventDefault();
                    return;
                }

                this.answerUpdated(e);
            },

            //will enfore maxlength attribute on elements that don't natively support.
            enforceMaxLength: function (e) {
                var key = e.which,
                    max = e.currentTarget.getAttribute('maxlength'),
                    len = e.currentTarget.value.length;

                if (key !== 8 && key !== 46) {
                    if (len >= max) e.preventDefault();
                }
                this.answerUpdated(e);
            },

            saveSubmit: function (e) {
                e.preventDefault();
                var elapsedTime = Math.round((new Date() - this.startTimer) / 1000),
                    $this = $(e.currentTarget),
                    $btn = $this.attr('disabled', 'disabled'),
                    $required = $('.ui-slider-input, textarea, select.priority, input.priority').not('.ignore'),
                    path = $this.data('path'),
                    $inputs = $('.ui-slider-input, textarea, .reasons, select.priority, input.priority').not('.ignore'),
                    inventory = Inventory.instance(),
                    inventoryType = this.inventoryType,
                    url = inventory.get('latest').href,
                    patientIdentifier = inventory.get('patientIdentifier'),
                    patientId = patientIdentifier.uniqueId,
                    answers = [],
                    addUnaddedReason = this.reasonAddItem,
                    qR = new QuestionResponses(),
                    answer,
                    question,
                    self = this,
                    loadForm = this.loadForm,
                    saveAnswers = function (answers) {
                        answer = answers.shift();
                        answer.save({}, {
                            url: url,
                            error: function () {
                            },
                            success: function () {
                                if (answers.length > 0) {
                                    qR.getDateUpdated(answer.questionKey);
                                    saveAnswers(answers);
                                } else {
                                    if (inventoryType === 'PRIORITIES') {
                                        loadForm(self);
                                    } else {
                                        if (path) {
                                            window.location = path;
                                        }
                                        else {
                                            window.location.hash = $('.go-back').eq(0).attr('href');
                                        }
                                    }
                                }
                            }
                        });
                    },
                    fails = function ($req) {
                        var failMsg = "";
                        $req.each(function () {
                            if ($(this).val().trim() === "") failMsg += "<p>" + $('label[for="' + this.id + '"]').text() + "</p>";
                        });

                        return failMsg;
                    },
                    failsRequired = fails($required);

                //auto add reason to list
                $('.reason-new').each(function () {
                    var $this = $(this);

                    if ($this.val().trim() !== "") {
                        addUnaddedReason($('#' + $this.parent().find('.reason-add').data('control')));
                    }
                });

                if (failsRequired) {
                    $.confirm({
                        'title': 'These questions require answers in order to save:',
                        'class': 'failedSaveAttempt',
                        'message': failsRequired,
                        'buttons': {
                            'OK': {
                                'class': 'ui-btn ui-shadow ui-btn-corner-all ui-btn-up-e'
                            }
                        }
                    });
                } else {
                    //build collection of answer objects
                    //equivalent to $.each minus overhead
                    for (var i = 0, l = $inputs.length; i < l; i++) {
                        question = $inputs[i];

                        answer = question.tagName === "UL"
                            ? question.innerHTML
                            : question.value;

                        answers.push(
                            new QuestionResponse({
                                questionKey: question.id.replace('-input', ''),
                                inventoryType: inventoryType,
                                patientId: patientId,
                                answer: answer,
                                elapsedTime: elapsedTime,
                                patientIdentifier: patientIdentifier
                            })
                        );
                    }
                    //recursive if need be...
                    saveAnswers(answers);
                }
                //Save button re-enabled: service error || failedValidation
                $btn.removeAttr('disabled');
            },

            //removes list-item and marks its parent list as updated
            reasonOmit: function (e) {
                e.preventDefault();
                var $li = $(e.currentTarget).parent(),
                    $ul = $li.parent();

                $li.remove();
                $ul.attr('data-updated', true);
            },

            //adds list-item to list, marks list as updated
            reasonAddItem: function ($control) {
                var $input = $control.find('.reason-new'),
                    $list = $control.find('.reasons'),
                // prevents html breakage, script injection, replaces quotes, removes leading/trailing spaces
                    value = $('<p>' + $input.val() + '</p>').text().replace(/"/g, '&quot;').replace(/'/g, '&#39;').trim();

                if (value !== '') {
                    $list
                        .append('<li><a href="#" title="remove this reason" class="reason-omit ui-icon ui-icon-delete ui-icon-shadow"></a>' + value + '</li>')
                        .attr('data-updated', true);
                }

                $input.val('');
            },

            reasonAdd: function (e) {
                this.reasonAddItem($('#' + $(e.currentTarget).data('control')));
            },

            //return key same as click add button
            returnAdd: function (e) {
                var $control = $('#' + $(e.currentTarget).data('control'));

                if (e.which === 13) {
                    this.reasonAddItem($control);
                }

                //mark data-updated regardless
                $control.find('.reasons').attr('data-updated', true);
            },

            //currently deprecated, added ability to edit list items
            updateLi: function (e) {
                var value = '', lis = '',
                    breaks = e.currentTarget.innerHTML.split(/<br[^>]*>/g);

                for (var i = -1, l = breaks.length; ++i < l;) {
                    // text() will remove html tags/ prevent script injection... html breakage
                    value = $('<p>' + breaks[i] + '</p>').text().replace(/"/g, '&quot;').replace(/'/g, '&#39;').trim();
                    if (value !== '') lis += '<li contenteditable="true">' + value + '</li>';
                }

                $(e.currentTarget).replaceWith(lis);
            },

            /* PRIORITIES SPECIFIC */
            selectedGoal: function (e) {
                //clear own goal input, set data-selected to selected option for CSS purposes
                var $this = $(e.currentTarget),
                    val = $this.val(),
                    $uiselect = $this.parent().parent(),
                    $ownGoal = $uiselect.parent().find('.goal-input');

                $ownGoal
                    .removeClass('ignore')
                    .val('');

                if (val === 'Enter My Own') {
                    $ownGoal.removeClass('ignore').val('');
                    $this.addClass('ignore');
                    $uiselect.addClass('ignore');
                } else {
                    $ownGoal.addClass('ignore')
                    $this.removeClass('ignore');
                    $uiselect.removeClass('ignore');
                }
            },

            selectedArea: function (e) {
                this.answerUpdated(e);
                var $this = $(e.currentTarget),
                    i = $('select.select-area').index(e.currentTarget),
                    $list = $('.goals').eq(i)
                        .css('visibility', 'visible')
                        .find('select'),
                    j = $this.children().index($this.find('option:selected')) - 1,
                    $fieldsets = $('fieldset'),
                    buildList = function ($list, data) {
                        var newOption = document.createElement('option');

                        newOption.value = '';
                        newOption.text = 'Select a goal';
                        $list
                            .children()
                            .remove()
                            .end()
                            .append(newOption);

                        //cannot user innerHTML to inject children options on select
                        for (var i = -1, l = data.length; ++i < l;) {
                            newOption = document.createElement('option');
                            newOption.value = data[i];
                            newOption.text = data[i];
                            $list.append(newOption);
                        }

                        newOption = document.createElement('option');
                        newOption.value = 'Talk to my VA Team';
                        newOption.text = 'Talk to my VA Team';
                        $list.append(newOption);

                        newOption = document.createElement('option');
                        newOption.value = 'Enter My Own';
                        newOption.text = 'Enter My Own';

                        $list
                            .append(newOption)
                            .children()
                            .eq(0).attr('selected', 'selected')
                            .end()
                            .end()
                            .prev('span').find('.ui-btn-text span').text('Select a goal');

                        $('.btnWrapper').show();

                    };

                if (j >= 0) {
                    $('.baseline').hide();
                    $fieldsets.eq(i)
                        .find('legend span')
                        .hide();
                    $fieldsets.not($fieldsets.eq(i))
                        .hide()
                        .addClass('ignore')
                        .find('.priority')
                        .addClass('ignore');

                    buildList($list, this.categories[j].goals);
                } else {
                    this.loadForm();
                }
            },

            continueOn: function () {
                $('.ready-check').remove();
            },

            dubiousDialog: function (e) {
                $.confirm({
                    'message': 'You are the expert on you! Thank you for reflecting on these areas of your life. Your VA Team is here to support you. Contact us if you have any questions.',
                    'buttons': {
                        'OK': {
                            'class': 'ui-btn ui-shadow ui-btn-corner-all ui-btn-up-e',
                            'action': function () {
                                window.location = "#home";
                            }
                        }
                    }
                });
            },

            cancel: function () {
                this.loadForm();
            },

            loadForm: function (self) {
                var scope = self !== undefined ? self : this;

                scope.details = scope.getBaseline(scope.details);

                scope.$el
                    .empty()
                    .append(scope.disclaimer())
                    .append(scope.formTemplate(scope.details))
                    .trigger('create');

                $(window).scrollTop(0);
            }
        });
    }
);

