package com.agilex.healthcare.pdf;

import java.util.Date;

import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilter;
import com.agilex.healthcare.utility.DateHelper;
import com.itextpdf.text.Anchor;
import com.itextpdf.text.BaseColor;
import com.itextpdf.text.Document;
import com.itextpdf.text.DocumentException;
import com.itextpdf.text.Phrase;
import com.itextpdf.text.pdf.PdfPCell;
import com.itextpdf.text.pdf.PdfPTable;
import com.itextpdf.text.pdf.PdfWriter;

public abstract class AbstractPage {
	protected Document document;
	protected PageEvent event;
	protected PdfWriter writer;
	protected PdfGenerationContext context;
	
	public AbstractPage(Document document, PageEvent event, PdfWriter writer, PdfGenerationContext context) {
		this.document = document;
		this.event = event;
		this.writer = writer;
		this.context = context;
	}

	public abstract void create();

    public void addHeading(String title, String anchorName) throws DocumentException {
        PdfPTable heading = new PdfPTable(1);
        heading.setWidthPercentage(100);
        heading.setSpacingBefore(20);
        heading.setSpacingAfter(0);

        Anchor anchorTarget = new Anchor(title, Stylesheet.getH2());
        anchorTarget.setName(anchorName);

        PdfPCell cell = new PdfPCell(anchorTarget);
        cell.setBorderWidth(0);
        cell.setPadding(0);
        cell.setHorizontalAlignment(PdfPTable.ALIGN_LEFT);
        heading.addCell(cell);

        document.add(heading);
    }

    public void addGeneratedFor() throws DocumentException{
        String firstName = context.getPatient().getFirstName();
        String lastName  = context.getPatient().getLastName();
        addPhrase(new Phrase("Generated exclusively for:  " + firstName + " " + lastName,
            Stylesheet.getH3()));
    }

    public void addGenerationTime() throws DocumentException {
        addPhrase(new Phrase("On:  " + DateHelper.formatDate(new Date()), Stylesheet.getH3()));
    }

    public void addDateRange(DateFilter dateFilter) throws DocumentException {
        if (dateFilter != null && dateFilter.getStartDate() != null && dateFilter.getEndDate() != null) {
            String startDate = DateHelper.formatDate(dateFilter.getStartDate());
            String endDate = DateHelper.formatDate(dateFilter.getEndDate());
            addPhrase(new Phrase("Date Range:  " + startDate + " to " + endDate, Stylesheet.getH3()),
                15, 10, PdfPCell.ALIGN_LEFT, 0);
        }
    }

    public void addNoDataFound() throws DocumentException {
        addPhrase(new Phrase("No data found within the specified date range.",
            Stylesheet.getBlueItalic()), 10, 10, PdfPTable.ALIGN_LEFT, 0);
    }

    public void addPhrase(Phrase phrase) throws DocumentException {
        addPhrase(phrase, 5, 0, PdfPTable.ALIGN_LEFT, 0);
    }

    public void addPhrase(Phrase phrase, float before, float after, int align, int bottomBorder) throws DocumentException {
        PdfPTable row = new PdfPTable(1);
        row.setWidthPercentage(80);
        row.setSpacingBefore(before);
        row.setSpacingAfter(after);
        row.setHorizontalAlignment(align);

        PdfPCell cell = new PdfPCell(phrase);
        cell.setBorderWidth(0);
        cell.setBorderWidthBottom(bottomBorder);
        cell.setPadding(0);
        cell.setHorizontalAlignment(align);
        row.addCell(cell);
        document.add(row);
    }

    protected PdfPTable createDataTable() throws DocumentException{
        int[] columnWidths = context.getColumnWidths();
        String[] columnHeaders = context.getColumnHeaders();

        PdfPTable table = new PdfPTable(columnWidths.length);
        table.setWidthPercentage(80);
        table.setWidths(columnWidths);
        table.setSpacingBefore(10);
        table.setSpacingAfter(10);
        table.setHorizontalAlignment(PdfPTable.ALIGN_LEFT);

        PdfPCell defaultCell = table.getDefaultCell();
        defaultCell.setPadding(5);
        defaultCell.setBorderWidth(0);
        defaultCell.setBorderWidthBottom(1);
        defaultCell.setHorizontalAlignment(PdfPTable.ALIGN_LEFT);
        defaultCell.setVerticalAlignment(PdfPTable.ALIGN_CENTER);

        for(int j=0; j<columnWidths.length; j++) {
            table.addCell(new Phrase(columnHeaders[j], Stylesheet.getBoldLarge()));
        }
        table.setHeaderRows(1);

        defaultCell.setBorderWidthBottom(0);

        java.util.List<java.util.List<String>> data = context.getReportDataTable();
        BaseColor lightYellow = new BaseColor(0xFF, 0xFF, 0xCC);
        for(int i=data.size() -1; i>=0; i--) {
        	BaseColor bgColor = (i % 2 == 0) ? null : lightYellow;
            defaultCell.setBackgroundColor(bgColor);

            for(int j=0; j<data.get(i).size(); j++) {
                table.addCell(data.get(i).get(j));
            }
        }

        return table;
    }
}
