package com.agilex.healthcare.mygoals.pdf;

import java.io.*;
import java.text.SimpleDateFormat;
import java.util.*;

import com.agilex.healthcare.mobilehealthplatform.domain.Mygoals;
import com.agilex.healthcare.pdf.PDFException;
import com.agilex.healthcare.pdf.PdfGenerationContext;
import com.agilex.healthcare.pdf.Stylesheet;
import com.agilex.healthcare.utility.NullChecker;
import com.agilex.healthcare.utility.StreamHelper;
import com.itextpdf.text.Element;
import com.itextpdf.text.Phrase;
import com.itextpdf.text.pdf.PdfPCell;
import com.itextpdf.text.pdf.PdfPTable;


public class MygoalsSummaryPage {
  
  private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(MygoalsSummaryPage.class);
  
  private Map<String, Mygoals> questionKeyMygoalsMap;
  private PdfGenerationContext context;
  

  private static final String PROPERTIES_PATH = "com/agilex/healthcare/mobilehealthplatform/restservice/mygoals/mygoals.properties";
  
  private static String REFLECTION_TEXT = "reflection";
  private static String FEELINGS_TEXT = "feelings";
  private static final String FEEL_MENTALLY = "feelings-mentally";
  private static final String CURRENT_STATE = "Current";
  private static final String DESIRED_STATE = "Desired";
  private static String PRIORITY1 = "priority1-area";
  private static String PRIORITY2 = "priority2-area";
  private static String AREA_QUESTION_TEXT = "area";
  private static String GOAL_QUESTION_TEXT = "goal";
  
  public MygoalsSummaryPage(PdfGenerationContext context, Map<String, Mygoals> qKeyMygoalsMap) {
    this.context = context;
    this.questionKeyMygoalsMap = qKeyMygoalsMap;
  }
  
  public void addSummaryDetails(PdfPTable reflectionsTable,
      PdfPTable feelingsTable, PdfPTable currentAndDesireStateTable,
      PdfPTable prioritiesTable) {
    
    List<String> sortedQuestionKeys =  loadAndSortQuestionKeys();
    
    for (String questionKey : sortedQuestionKeys) {
      if (questionKey.contains(REFLECTION_TEXT)) {
        Mygoals mygoals = getResponse(questionKey);
        addReflections(reflectionsTable, context, mygoals);
      } else if (questionKey.contains(FEELINGS_TEXT)) {
        Mygoals mygoals = getResponse(questionKey);
        addFeelings(feelingsTable, context, mygoals);
        if (mygoals.getQuestionKey().contains(FEEL_MENTALLY)) {
          setSupportLine(feelingsTable);
        }
      } else if (questionKey.contains(CURRENT_STATE)) {
        Mygoals mygoals = getResponse(questionKey);
        addCurrentAndDesireState(currentAndDesireStateTable, context, mygoals); 
      } else if (questionKey.contains(PRIORITY1) || questionKey.contains(PRIORITY2)) {
        Mygoals mygoals = getResponse(questionKey);
        addPriorities(prioritiesTable, context, mygoals);
      }
    }
  }

  private List<String> loadAndSortQuestionKeys()  {
    Properties questionProps = new Properties();
    InputStream inputStream = null;
    try {
        inputStream = MygoalsSummaryPage.class.getClassLoader().getResourceAsStream(PROPERTIES_PATH);   
      questionProps.load(inputStream);
      inputStream.close();
    } catch (IOException e) {
      throw new PDFException("Could not load properties");
    } finally {
      StreamHelper.safeClose(inputStream);
    }
      return sortQuestionKeys(questionProps);
  }

  private List<String> sortQuestionKeys(Properties questionProps) {
    
    Enumeration<Object> keysEnum = questionProps.keys();
      List<String> keyList = new ArrayList<String>();
      while(keysEnum.hasMoreElements()){
         keyList.add((String)keysEnum.nextElement());
      }
      Collections.sort(keyList);
       
    List<String> questionKeys = new ArrayList<String>();
    for (Object questionKey : keyList) {
      questionKeys.add((String)questionProps.get(questionKey));
    }
    return questionKeys;
  }

  private Mygoals getResponse(String questionKey) {
    Mygoals mygoals = questionKeyMygoalsMap.get(questionKey);
    if (mygoals == null) {
      mygoals = new Mygoals();
      mygoals.setQuestionKey(questionKey);
    }
    return mygoals;
  }

  private void addReflections(PdfPTable reflectionsTable, PdfGenerationContext context, Mygoals mygoals) {
    
    setReflectionQuestion(reflectionsTable, context, mygoals);    
    setAnswer(reflectionsTable, context, mygoals, Element.ALIGN_LEFT, 15.0F);
    
  }

  private void addFeelings(PdfPTable feelingsTable, PdfGenerationContext context, Mygoals mygoals) {
    
    setQuestion(feelingsTable, context, mygoals);
    setUpdatedDate(feelingsTable, mygoals);
    setAnswer(feelingsTable, context, mygoals, Element.ALIGN_CENTER, 0.0F);
  
  }

  private void addCurrentAndDesireState(PdfPTable currentAndDesireStateTable, PdfGenerationContext context, Mygoals mygoals) {
    Mygoals desireMygoals = new Mygoals();
    if (mygoals.getAnswer() != null) {
      desireMygoals = getDesireState(mygoals.getQuestionKey());
    }
    setQuestion(currentAndDesireStateTable, context, mygoals);
    setUpdatedDate(currentAndDesireStateTable, mygoals);
    setAnswer(currentAndDesireStateTable, context, mygoals, Element.ALIGN_CENTER, 0.0F);    
    setAnswer(currentAndDesireStateTable, context, desireMygoals, Element.ALIGN_CENTER, 0.0F);
  
  } 

  private void addPriorities(PdfPTable prioritiesTable, PdfGenerationContext context, Mygoals mygoals) {
    if (mygoals.getQuestionKey().contains(PRIORITY1)) {
      setPriority(prioritiesTable, context, "mygoals.report.priorities.p1");
    } else if (mygoals.getQuestionKey().contains(PRIORITY2)) {
      setPriority(prioritiesTable, context, "mygoals.report.priorities.p2");
    }
    setUpdatedDate(prioritiesTable, mygoals);
    Mygoals goalResp = getPriorityGoals(mygoals.getQuestionKey());
    setPriorityAnswer(prioritiesTable, context, mygoals, goalResp);
    
  }

  private void setReflectionQuestion(PdfPTable reflectionsTable, PdfGenerationContext context, Mygoals mygoals) {
    PdfPCell cell;
    if (mygoals.getDateUpdated() == null) {
      cell = new PdfPCell(new Phrase(context.getMessageSource().getMessage(mygoals.getQuestionKey(), null, context.getLocale()), Stylesheet.getNormal()));
    } else  {
      cell = new PdfPCell(new Phrase(context.getMessageSource().getMessage(mygoals.getQuestionKey(), null, context.getLocale()) + "   (" + 
          context.getMessageSource().getMessage("mygoals.report.udpated", null, context.getLocale()) + "  " +  formattedDate(mygoals.getDateUpdated())+")", Stylesheet.getNormal()));
    }
    cell.setPadding (10.0f);
    reflectionsTable.addCell(cell);
  }

  private void setQuestion(PdfPTable table, PdfGenerationContext context, Mygoals mygoals) {
    PdfPCell cell = new PdfPCell(new Phrase(context.getMessageSource().getMessage(mygoals.getQuestionKey(), null, context.getLocale())));
    cell.setPadding (8.0f);
    table.addCell(cell);
  }

  private void setUpdatedDate(PdfPTable table, Mygoals mygoals) {
    PdfPCell cell;
    if (mygoals.getDateUpdated() == null) {
      cell = new PdfPCell(new Phrase(""));
    } else {
      cell = new PdfPCell(new Phrase(formattedDate(mygoals.getDateUpdated()), Stylesheet.getNormal()));
    }
    cell.setPadding (8.0f);
    table.addCell(cell);
  }
  
  private void setAnswer(PdfPTable table, PdfGenerationContext context, Mygoals mygoals, int alignment, float indent) {
    PdfPCell cell;
    if (mygoals.getAnswer() != null) {
      cell = new PdfPCell(new Phrase(mygoals.getAnswer(), Stylesheet.getNormal()));
    } else {
      cell = new PdfPCell(new Phrase(context.getMessageSource().getMessage("mygoals.report.no.answer", null, context.getLocale())));
    }   
    cell.setHorizontalAlignment(alignment);
    cell.setIndent(indent);
    cell.setPadding(8.0F);
    table.addCell(cell);
  }

  private void setSupportLine(PdfPTable feelingsTable) {
    PdfPCell cell = new PdfPCell(new Phrase(context.getMessageSource().getMessage("mygoals.report.feelings.supportline", null, context.getLocale()), Stylesheet.getNormal()));
    cell.setPadding (10.0f);
    cell.setHorizontalAlignment(Element.ALIGN_LEFT);
    cell.setColspan(3);
    feelingsTable.addCell(cell);
  }

  private void setPriorityAnswer(PdfPTable prioritiesTable,
      PdfGenerationContext context, Mygoals areaResp,
      Mygoals goalResp) {
    
    PdfPCell cell;
    if (areaResp.getAnswer() != null && goalResp.getAnswer() != null) {
      cell = new PdfPCell(new Phrase(areaResp.getAnswer() + ": " + goalResp.getAnswer(), Stylesheet.getNormal()));
    } else {
      cell = new PdfPCell(new Phrase(context.getMessageSource().getMessage("mygoals.report.no.answer", null, context.getLocale())));
    }
    cell.setPadding (8.0f);
    prioritiesTable.addCell(cell);
  }

  private Mygoals getDesireState(String questionKey) {
    String desiredQuestionKey = questionKey.replaceAll(CURRENT_STATE, DESIRED_STATE);
    return questionKeyMygoalsMap.get(desiredQuestionKey);
  }

  private void setPriority(PdfPTable prioritiesTable, PdfGenerationContext context, String priority) {
    PdfPCell cell = new PdfPCell(new Phrase(context.getMessageSource().getMessage(priority, null, context.getLocale()), Stylesheet.getNormal()));
      cell.setPadding (8.0f);
      prioritiesTable.addCell(cell);
  } 
  
  private Mygoals getPriorityGoals(String questionKey) {
    String goalQuestionKey = questionKey.replaceAll(AREA_QUESTION_TEXT, GOAL_QUESTION_TEXT);
    return questionKeyMygoalsMap.get(goalQuestionKey);
  }

    private synchronized String formattedDate(Date date) {
        if(NullChecker.isNullish(date))
            return "";

        SimpleDateFormat DATE_FMT = new SimpleDateFormat("MM/dd/yyyy");
        return DATE_FMT.format(date);
    }

}
