package com.agilex.healthcare.mygoals.pdf;

import java.util.HashMap;
import java.util.Map;

import com.agilex.healthcare.mobilehealthplatform.domain.Mygoals;
import com.agilex.healthcare.mobilehealthplatform.domain.MygoalsList;
import com.agilex.healthcare.mobilehealthplatform.enumeration.MygoalsInventoryType;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.Domain;
import com.agilex.healthcare.pdf.AbstractPage;
import com.agilex.healthcare.pdf.PDFException;
import com.agilex.healthcare.pdf.PageEvent;
import com.agilex.healthcare.pdf.PdfGenerationContext;
import com.agilex.healthcare.pdf.Stylesheet;
import com.itextpdf.text.BaseColor;
import com.itextpdf.text.Chunk;
import com.itextpdf.text.Document;
import com.itextpdf.text.DocumentException;
import com.itextpdf.text.Element;
import com.itextpdf.text.Phrase;
import com.itextpdf.text.pdf.PdfPCell;
import com.itextpdf.text.pdf.PdfPTable;
import com.itextpdf.text.pdf.PdfWriter;


public class MygoalsPage extends AbstractPage {
  
  private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(MygoalsPage.class);
  
  private Map<String, MygoalsList> invTypeMygoalsListMap = new HashMap<String, MygoalsList>();
  private Map<String, Mygoals> questionKeyMygoalsMap = new HashMap<String, Mygoals>();
  
  public MygoalsPage(Document document, PageEvent event, PdfWriter writer, PdfGenerationContext context) {
    super(document, event, writer, context);
  }
  
  public void create() {
    
    try {
      if (!context.isProviderReport()) {
        addHeading(context.getMessageSource().getMessage("mygoals.report.title", null, context.getLocale()), Domain.mygoals+ "," + context.getPatient().getPatientIdentifier().getUniqueId());  
      }

      PdfPTable reflectionsTable = createReflectionsTable();
      PdfPTable feelingsTable = createFeelingsTable();
      PdfPTable currentAndDesireStateTable = createCurrentAndDesireStateTable();
      PdfPTable prioritiesTable = createPrioritiesTable();
    
      addMyStorySummaryHeadings(reflectionsTable, feelingsTable,
          currentAndDesireStateTable, prioritiesTable);

      MygoalsList mygoalsList = (MygoalsList)context.getInformation(Domain.mygoals+ "," + context.getPatient().getPatientIdentifier().getUniqueId());
    
      populateInvTypeMygoalsMap(mygoalsList);
      
      setQuestionResponseMap(MygoalsInventoryType.REFLECTIONS.name());
      setQuestionResponseMap(MygoalsInventoryType.FEELINGS.name());
      setQuestionResponseMap(MygoalsInventoryType.CURRENT_AND_DESIRED_STATE.name());
      setQuestionResponseMap(MygoalsInventoryType.PRIORITIES.name());
      
      MygoalsSummaryPage summaryDetailsPage = new MygoalsSummaryPage(context, questionKeyMygoalsMap);
      summaryDetailsPage.addSummaryDetails(reflectionsTable, feelingsTable, currentAndDesireStateTable, prioritiesTable);
      
      document.add(reflectionsTable);
      document.add(feelingsTable);
      document.add(currentAndDesireStateTable);
      document.add(prioritiesTable);
      
    } catch (DocumentException e) {
      logger.error("Error generating MyGoals PDF document ", e);
      throw new PDFException("Error creating PDF document ");
    }
  }

  private void addMyStorySummaryHeadings(PdfPTable reflectionsTable,
      PdfPTable feelingsTable, PdfPTable currentAndDesireStateTable,
      PdfPTable prioritiesTable) {
    
    addTableheading(reflectionsTable, "mygoals.report.reflections", 1);
    addTableheading(feelingsTable, "mygoals.report.feelings", 3);
    addSecondTableheading(feelingsTable, "mygoals.report.udpated", "mygoals.report.feelings.scale", null);
    addTableheading(currentAndDesireStateTable, "mygoals.report.currentanddesirestate", 4);
    addSecondTableheading(currentAndDesireStateTable, "mygoals.report.udpated", "mygoals.report.currentstate", "mygoals.report.desirestate");
    addTableheading(prioritiesTable, "mygoals.report.priorities", 3);
    addSecondTableheading(prioritiesTable, "mygoals.report.udpated", "", null);
  }

  private PdfPTable createReflectionsTable() {
    PdfPTable reflectionsTable = new PdfPTable(1);
    reflectionsTable.setWidthPercentage(100);
    reflectionsTable.setSpacingBefore(10);
    reflectionsTable.setSpacingAfter(10);
    
    return reflectionsTable;
  }

  private PdfPTable createFeelingsTable() throws DocumentException {
    PdfPTable feelingsTable = new PdfPTable(3);
    feelingsTable.setWidthPercentage(100);
    feelingsTable.setSpacingBefore(10);
    feelingsTable.setSpacingAfter(20);
    feelingsTable.setWidths(new int[] {36, 15, 39});  
    return feelingsTable;
  } 
  
  private PdfPTable createCurrentAndDesireStateTable() throws DocumentException {
    PdfPTable currentAndDesireStateTable = new PdfPTable(4);
    currentAndDesireStateTable.setWidthPercentage(100);
    currentAndDesireStateTable.setSpacingBefore(10);
    currentAndDesireStateTable.setSpacingAfter(10);
    currentAndDesireStateTable.setWidths(new int[] {35, 15, 20, 20}); 
    return currentAndDesireStateTable;
  } 

  private PdfPTable createPrioritiesTable() throws DocumentException {
    PdfPTable prioritiesTable = new PdfPTable(3);
    prioritiesTable.setWidthPercentage(100);
    prioritiesTable.setSpacingBefore(10);
    prioritiesTable.setSpacingAfter(10);
    prioritiesTable.setWidths(new int[] {40, 15, 45});  
    return prioritiesTable;
  } 
  
  private void addTableheading(PdfPTable table, String tableheading, int colspan) {
    PdfPCell cell = new PdfPCell(new Phrase(context.getMessageSource().getMessage(tableheading, null, context.getLocale()), Stylesheet.getBold()));
    cell.setHorizontalAlignment(Element.ALIGN_CENTER);
    cell.setBackgroundColor(BaseColor.LIGHT_GRAY);   
    cell.setColspan(colspan);
    table.addCell(cell);
  }
  
  private void addSecondTableheading(PdfPTable table, String dateHeading, String answer1Heading, String answer2Heading ) {
    PdfPCell cell = new PdfPCell(new Phrase("", Stylesheet.getNormal()));
    cell.setPadding (10.0f);
    table.addCell(cell);

    cell = new PdfPCell(new Phrase(context.getMessageSource().getMessage(dateHeading, null, context.getLocale()), Stylesheet.getBold()));
    cell.setPadding (10.0f);
    table.addCell(cell);

    if (answer2Heading == null) {
      if (!answer1Heading.isEmpty()) {
        cell = new PdfPCell(new Phrase(context.getMessageSource().getMessage(answer1Heading, null, context.getLocale()), Stylesheet.getBold()));
      } else {
        cell = new PdfPCell(new Phrase("", Stylesheet.getNormal()));
      }
      cell.setPadding (10.0f);
      table.addCell(cell);
    } else {
      cell = new PdfPCell(new Phrase(context.getMessageSource().getMessage(answer1Heading, null, context.getLocale()) + "\n" + "\n" +
          new Chunk(context.getMessageSource().getMessage("mygoals.report.currentanddesirestate.scale", null, context.getLocale()), Stylesheet.getItalic()), Stylesheet.getBold()));
      cell.setPadding (10.0f);
      table.addCell(cell);
      
      cell = new PdfPCell(new Phrase(context.getMessageSource().getMessage(answer2Heading, null, context.getLocale()) + "\n" + "\n" +
          new Chunk(context.getMessageSource().getMessage("mygoals.report.currentanddesirestate.scale", null, context.getLocale()), Stylesheet.getItalic()), Stylesheet.getBold()));
      cell.setPadding (10.0f);
      table.addCell(cell);  
      
    }
  }

  private void populateInvTypeMygoalsMap(MygoalsList mygoalsList) {
    MygoalsList feelings = new MygoalsList();
    MygoalsList reflections = new MygoalsList();
    MygoalsList currentAndDesiredState = new MygoalsList();
    MygoalsList priorities = new MygoalsList();
    for (Mygoals mygoals : mygoalsList) {
      if (MygoalsInventoryType.FEELINGS.name().equalsIgnoreCase(mygoals.getInventoryType())) {
        feelings.add(mygoals);
      } else if (MygoalsInventoryType.REFLECTIONS.name().equalsIgnoreCase(mygoals.getInventoryType())) {
        reflections.add(mygoals);
      } else if (MygoalsInventoryType.CURRENT_AND_DESIRED_STATE.name().equalsIgnoreCase(mygoals.getInventoryType())) {
        currentAndDesiredState.add(mygoals);
      } else if (MygoalsInventoryType.PRIORITIES.name().equalsIgnoreCase(mygoals.getInventoryType())) {
        priorities.add(mygoals);
      }
    }
    invTypeMygoalsListMap.put(MygoalsInventoryType.REFLECTIONS.name(), reflections);
    invTypeMygoalsListMap.put(MygoalsInventoryType.FEELINGS.name(), feelings);
    invTypeMygoalsListMap.put(MygoalsInventoryType.CURRENT_AND_DESIRED_STATE.name(), currentAndDesiredState);
    invTypeMygoalsListMap.put(MygoalsInventoryType.PRIORITIES.name(), priorities);
  }

  private void setQuestionResponseMap(String inventoryType) {
    if (!invTypeMygoalsListMap.isEmpty()) {
      for (Mygoals mygoals : invTypeMygoalsListMap.get(inventoryType)) {
        questionKeyMygoalsMap.put(mygoals.getQuestionKey(), mygoals);
      }
    }
  }

}
