package com.agilex.healthcare.mygoals.pdf;

import java.awt.Color;

import com.agilex.healthcare.pdf.AbstractPage;
import com.agilex.healthcare.pdf.PDFException;
import com.agilex.healthcare.pdf.PageEvent;
import com.agilex.healthcare.pdf.PdfGenerationContext;
import com.agilex.healthcare.pdf.PdfRoundCorners;
import com.agilex.healthcare.pdf.Stylesheet;
import com.agilex.healthcare.utility.DateHelper;
import com.itextpdf.text.BaseColor;
import com.itextpdf.text.Chunk;
import com.itextpdf.text.Document;
import com.itextpdf.text.DocumentException;
import com.itextpdf.text.Element;
import com.itextpdf.text.Paragraph;
import com.itextpdf.text.Phrase;
import com.itextpdf.text.pdf.PdfPCell;
import com.itextpdf.text.pdf.PdfPTable;
import com.itextpdf.text.pdf.PdfWriter;

public class MygoalsCoverPage extends AbstractPage {
  
  private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(MygoalsCoverPage.class);
  
  public MygoalsCoverPage(Document document, PageEvent event, PdfWriter writer, PdfGenerationContext context) {
    super(document, event, writer, context);
  }

  public void create() {
    try {
      if (!context.isProviderReport()) {
        addReportInformation(context);
        addPatientNote();
      } else {
        addProviderNote(context); 
      }
    } catch (DocumentException e) {
      logger.error("Error generating PDF Mygoals Cover Page ", e);
      throw new PDFException("Error generating PDF Mygoals Cover Page");
    }
    
    //addLogo();
  }

  private void addPatientNote() throws DocumentException {
    BaseColor noteColor = new BaseColor(0x54, 0x69, 0x93);

    PdfPTable table = new PdfPTable(1);
    table.setWidthPercentage(100);
    table.setSpacingBefore(20);
    table.setSpacingAfter(20);

    PdfPCell cell = new PdfPCell(new Phrase(context.getMessageSource().getMessage("mygoals.report.title", null, context.getLocale()), Stylesheet.getH1()));
    cell.setBorderColor(noteColor);
    cell.setBorderWidth(0);
    cell.setPaddingTop(5);
    cell.setPaddingBottom(5);
    cell.setPaddingLeft(15);
    cell.setPaddingRight(15);
    cell.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
    cell.setBorderWidthBottom(0);
    cell.setCellEvent(new PdfRoundCorners(noteColor, true, true, false, false));
    table.addCell(cell);

    Paragraph note = new Paragraph();
    note.add(new Paragraph(context.getMessageSource().getMessage("mygoals.report.cover.confidential", null, context.getLocale()), Stylesheet.getBold()));
    note.add(Chunk.NEWLINE);
    note.add(new Paragraph(context.getMessageSource().getMessage("mygoals.report.cover.summary", null, context.getLocale()), Stylesheet.getNormal()));
    note.add(Chunk.NEWLINE);

    cell = new PdfPCell(note);
    cell.setBorderWidth(2);
    cell.setBorderColor(noteColor);
    cell.setPadding(20);
    cell.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
    table.addCell(cell);

    document.add(table);
  }
  
  private void addProviderNote(PdfGenerationContext context) throws DocumentException {
    final String REPORT_DATE = DateHelper.formatDateTimeInReportFormat(context.getReportDate());

    PdfPTable table = new PdfPTable(2);
    table.setWidthPercentage(100);
    table.setSpacingBefore(20);
    table.setSpacingAfter(20);
  
    PdfPCell cell = new PdfPCell(new Phrase(context.getMessageSource().getMessage("mygoals.provider.report.title", null, context.getLocale()), Stylesheet.getBold()));
    cell.setBorderWidth(0);
    cell.setPadding (10.0f);
    cell.setHorizontalAlignment(Element.ALIGN_CENTER);
    cell.setBackgroundColor(BaseColor.LIGHT_GRAY);   
    cell.setColspan(2);
    table.addCell(cell);

    appendToMygoalsReport(context, table, "mygoals.report.patient.name", context.getPatient().getFirstName() + " " + context.getPatient().getLastName());
    
    appendToMygoalsReport(context, table, "mygoals.report.patient.dob", context.getPatient().getDateOfBirth() != null ? context.getPatient().getDateOfBirth().toString() : "");
    
    appendToMygoalsReport(context, table, "mygoals.report.patient.ssn", context.getPatient().getSsn() != null ?  context.getPatient().getSsn().substring(context.getPatient().getSsn().length() - 4, context.getPatient().getSsn().length()) : "");

    appendToMygoalsReport(context, table, "mygoals.report.patient.summary", REPORT_DATE);
    
    document.add(table);
  }

  private void appendToMygoalsReport(PdfGenerationContext context, PdfPTable table, String message, String content) {
    Phrase phrase = new Phrase();
    phrase.add(new Chunk(context.getMessageSource().getMessage(message, null, context.getLocale()), Stylesheet.getNormal()));
    phrase.add(new Chunk(" "));
    phrase.add(new Chunk(content, Stylesheet.getNormal()));
    PdfPCell cell = new PdfPCell(phrase);
    cell.setBorderWidth(0);
    cell.setPadding (8.0f);
    table.addCell(cell);
  }

  private void addReportInformation(PdfGenerationContext context) throws DocumentException {
    final String FIRST_NAME = context.getPatient().getFirstName();
    final String LAST_NAME = context.getPatient().getLastName();
    final String REPORT_DATE = DateHelper.formatDateTimeInReportDetailedFormat(context.getReportDate());

    PdfPTable reportTable = new PdfPTable(1);
    reportTable.setWidthPercentage(100);
    reportTable.setSpacingBefore(0);
    reportTable.setSpacingAfter(10);

    PdfPCell cell = new PdfPCell(new Phrase(context.getMessageSource().getMessage("mygoals.report.cover.generatedFor", null, context.getLocale()), Stylesheet.getBold()));
    cell.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
    cell.setVerticalAlignment(PdfPCell.ALIGN_TOP);
    cell.setBorderWidth(0);
    reportTable.addCell(cell);

    cell = new PdfPCell(new Phrase(String.format("%s %s", FIRST_NAME, LAST_NAME), Stylesheet.getBold()));
    cell.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
    cell.setVerticalAlignment(PdfPCell.ALIGN_TOP);
    cell.setBorderWidth(0);
    reportTable.addCell(cell);

    Phrase phrase = new Phrase();
    phrase.add(new Chunk(context.getMessageSource().getMessage("mygoals.report.cover.on", null, context.getLocale()), Stylesheet.getBold()));
    phrase.add(new Chunk(" "));
    phrase.add(new Chunk(REPORT_DATE, Stylesheet.getBold()));
    cell = new PdfPCell(phrase);
    cell.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
    cell.setVerticalAlignment(PdfPCell.ALIGN_TOP);
    cell.setBorderWidth(0);
    reportTable.addCell(cell);

    phrase = new Phrase();
    phrase.add(new Chunk(context.getMessageSource().getMessage("mygoals.report.cover.from", null, context.getLocale()), Stylesheet.getBold()));
    cell = new PdfPCell(phrase);
    cell.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
    cell.setVerticalAlignment(PdfPCell.ALIGN_TOP);
    cell.setPaddingTop(10);
    cell.setBorderWidth(0);
    reportTable.addCell(cell);
  
    document.add(reportTable);
  }

}
