package com.agilex.healthcare.mobilehealthplatform.web;

import java.io.IOException;
import java.net.URI;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

import javax.annotation.Resource;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import javax.ws.rs.WebApplicationException;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.UriBuilder;

import org.codehaus.jackson.map.ObjectMapper;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.NoSuchBeanDefinitionException;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.http.converter.HttpMessageConverter;
import org.springframework.http.converter.json.MappingJacksonHttpMessageConverter;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContext;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.oauth2.client.DefaultOAuth2ClientContext;
import org.springframework.security.oauth2.client.OAuth2RestTemplate;
import org.springframework.security.oauth2.client.token.AccessTokenProvider;
import org.springframework.security.oauth2.client.token.AccessTokenProviderChain;
import org.springframework.security.oauth2.client.token.AccessTokenRequest;
import org.springframework.security.oauth2.client.token.DefaultAccessTokenRequest;
import org.springframework.security.oauth2.client.token.grant.client.ClientCredentialsAccessTokenProvider;
import org.springframework.security.oauth2.client.token.grant.code.AuthorizationCodeAccessTokenProvider;
import org.springframework.security.oauth2.client.token.grant.code.AuthorizationCodeResourceDetails;
import org.springframework.security.oauth2.client.token.grant.implicit.ImplicitAccessTokenProvider;
import org.springframework.security.oauth2.client.token.grant.password.ResourceOwnerPasswordAccessTokenProvider;
import org.springframework.security.oauth2.common.AuthenticationScheme;
import org.springframework.security.oauth2.common.OAuth2AccessToken;
import org.springframework.security.web.authentication.rememberme.AbstractRememberMeServices;
import org.springframework.stereotype.Controller;
import org.springframework.ui.ModelMap;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.servlet.ModelAndView;

import com.agilex.healthcare.mobilehealthplatform.datalayer.patient.PatientIcnService;
import com.agilex.healthcare.mobilehealthplatform.domain.LinkTitles;
import com.agilex.healthcare.mobilehealthplatform.domain.MhpUser;
import com.agilex.healthcare.mobilehealthplatform.domain.ResourceDirectory;
import com.agilex.healthcare.mobilehealthplatform.security.HATokenValidation;
import com.agilex.healthcare.mobilehealthplatform.security.MhpUserFactory;
import com.agilex.healthcare.mobilehealthplatform.utils.PropertyHelper;
import com.agilex.healthcare.mobilehealthplatform.utils.uriformaters.ResourceDirectoryBuilder;
import com.agilex.healthcare.utility.ModeHelper;
import com.agilex.healthcare.utility.NullChecker;

@Controller
public class LoginLogoutController implements ApplicationContextAware{
	
	private static final String OAUTH_SSOE_TARGET_URL_KEY = "oauth.ssoe_target_url";

	private static final String OAUTH_SSOE_PROXY_KEY = "oauth.ssoe_proxy";

	private static final String ENCODING_FORMAT = "UTF-8";

	private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(LoginLogoutController.class);
	
	private static final String KEY_AUTH_CODE = "code";
	private static final String KEY_GRANT_TYPE = "grant_type";
	private static final String KEY_SCOPE = "scope";
	private static final String KEY_REDIRECT_URI = "redirect_uri";
	private static final String KEY_ORIGINAL_REDIRECT_URI = "original_redirect_uri";
	private static final String KEY_CLIENT_ID = "client_id";
	private static final String KEY_STATE = "state";
	private static final String KEY_RESPONSE_TYPE = "response_type";
	private static final String KEY_CLIENT_SECRET = "client_secret";

	private static final String CLIENT_SECRET = "oauth.client_secret";
	private static final String GRANT_TYPE = "oauth.grant_type";
	private static final String SCOPE = "oauth.scope";
	private static final String STATE = "oauth.state";
	private static final String CLIENT_ID = "oauth.client_id";
	private static final String KEY_SSOE_LOGOUT_URL = "oauth.ssoe_logout_url";

    private static final String KEY_ENABLE_REDIRECT_SECURITY = "oauth.enable_redirect_security";
    private static final String KEY_REDIRECT_URI_ROOTS = "oauth.permitted_redirect_uri_roots";
	
	@Resource(name="haMode")
	private String haMode;

	@Resource
	private HATokenValidation haTokenValidation;
	
	@Resource
	private Map<String, String> availableVistaLocations;
	
	@Resource
	private PropertyHelper propertyHelper;
	
	@Resource
	ResourceDirectoryBuilder resourceDirectoryBuilder;
	
	private ApplicationContext applicationContext;
	
	@Override
	public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
		this.applicationContext = applicationContext;
	}

	@RequestMapping(value = "/login", method = RequestMethod.GET)
	public ModelAndView displayLoginPage(@RequestParam(value = "error", required = false) boolean error,
			HttpServletRequest req, ModelMap model) {

		constructErrorMessageIfErrorExists(model, error);
		constructLoginPageContents(model);

		return new ModelAndView("userLogin", model);
	}

	@RequestMapping(value = "/oauthlogin", method = RequestMethod.GET)
	public void redirectToLoginPage(@RequestParam(value = "error", required = false) boolean error,
			@RequestParam(value = KEY_REDIRECT_URI, required = false) String originalRedirectUri,
			HttpServletRequest req, HttpServletResponse response) throws Exception{
		logger.debug("In Oauth login");

		String baseUri = buildBaseUriString(req);
		ResourceDirectory resourceDirectory = buildResourceDirectory(req);
		String mhpAuthorizeUrl = resourceDirectory.getLink().getLinkByTitle(LinkTitles.OAuthAuthorize).getHref().toString();
		
		UriBuilder authorizeUrlBuilder = UriBuilder.fromPath(mhpAuthorizeUrl);
		String redirectUri = baseUri + "/oauthtoken";
		if (NullChecker.isNotNullish(originalRedirectUri)) {
			StringBuilder sb = new StringBuilder(redirectUri);
			sb.append("?");
			sb.append(KEY_ORIGINAL_REDIRECT_URI);
			sb.append("=");
			sb.append(originalRedirectUri);
			redirectUri = sb.toString();
		}
		
		URI uri = authorizeUrlBuilder
				.queryParam(KEY_RESPONSE_TYPE, KEY_AUTH_CODE)
				.queryParam(KEY_STATE, propertyHelper.getProperty(STATE))
				.queryParam(KEY_CLIENT_ID, propertyHelper.getProperty(CLIENT_ID))
				.queryParam(KEY_REDIRECT_URI, redirectUri)
				.queryParam(KEY_SCOPE, propertyHelper.getProperty(SCOPE)).build();
			
		redirect(response, uri.toString());
	}

	@RequestMapping(value = "/ssoeoauthauthorize", method = RequestMethod.GET)
	public void redirectToSSOE(HttpServletRequest req, HttpServletResponse response) throws Exception{
		logger.debug("In SSOe Oauth Authorize proxy");
		
        String ssoe_proxy = propertyHelper.getProperty(OAUTH_SSOE_PROXY_KEY);
        String ssoe_target_url = propertyHelper.getProperty(OAUTH_SSOE_TARGET_URL_KEY);

        StringBuffer targetUrl = new StringBuffer();
        targetUrl.append(UriBuilder.fromPath(ssoe_target_url).build().toString())
                         .append("?")
                         .append(req.getQueryString());

        String encodedTargetUrl = URLEncoder.encode(targetUrl.toString(), ENCODING_FORMAT);
        String doubleEncodedTargetUrl = URLEncoder.encode(encodedTargetUrl, ENCODING_FORMAT);

        StringBuffer oauthAuthorizeUrl = new StringBuffer();
        oauthAuthorizeUrl.append(UriBuilder.fromUri(ssoe_proxy).build().toString())
                         .append("?TARGET=")
                         .append(doubleEncodedTargetUrl);

        //logger.debug("SSOE URL:" + oauthAuthorizeUrl.toString());
        redirect(response, oauthAuthorizeUrl.toString());
	}

	@RequestMapping(value = "/oauthtoken", method = RequestMethod.GET)
	public void getOauthToken(@RequestParam(value = KEY_AUTH_CODE, required = false) String code,
			@RequestParam(value = KEY_ORIGINAL_REDIRECT_URI, required = false) String originalRedirectUri,
			HttpServletRequest request, HttpServletResponse response) {
		
		String contextPath = NullChecker.isNotNullish(request.getContextPath()) ? request.getContextPath() : "";
		String redirectUri = NullChecker.isNotNullish(originalRedirectUri) ? originalRedirectUri : contextPath + "/launchpad/";
		
		if (NullChecker.isNotNullish(code)) {
			//logger.debug("In Oauth token for code: " + code);
			SecurityContext ctx = SecurityContextHolder.getContext();
			ctx.setAuthentication(null);
			String token = getAccessToken(request, code, originalRedirectUri);
			redirectUri = redirectUri + "?token=" + token;
		} else {
			// START
			// the code below is not necessary when oauth login is completely handled by separate authoriziation service
			Authentication currentUser = getCurrentUser();
//			mhpTokenStore.removeAllUSerTokens(currentUser);
			invalidateRememberMeTokens(request, response, currentUser);
			invalidateCookies(response, contextPath);
			HttpSession session = request.getSession(false);
			if (session != null) {
				session.invalidate();
			}
			// END
		}

		redirect(response, redirectUri);
	}

	
	private String getAccessToken(HttpServletRequest req, String code, String originalRedirectUri) {
		String baseUri = buildBaseUriString(req);
		
		ResourceDirectory resourceDirectory = buildResourceDirectory(req);
		
		String redirectUri = baseUri + "/oauthtoken";
		if (NullChecker.isNotNullish(originalRedirectUri)) {
			StringBuilder sb = new StringBuilder(redirectUri);
			sb.append("?");
			sb.append(KEY_ORIGINAL_REDIRECT_URI);
			sb.append("=");
			sb.append(originalRedirectUri);
			redirectUri = sb.toString();
		}
		
		MultiValueMap<String, String> formData = new LinkedMultiValueMap<String, String>();
		formData.add(KEY_GRANT_TYPE, propertyHelper.getProperty(GRANT_TYPE));
		formData.add(KEY_CLIENT_ID, propertyHelper.getProperty(CLIENT_ID));
		formData.add(KEY_SCOPE, propertyHelper.getProperty(SCOPE));
		formData.add(KEY_AUTH_CODE, code);
		formData.add(KEY_CLIENT_SECRET, propertyHelper.getProperty(CLIENT_SECRET));
		formData.add(KEY_REDIRECT_URI, redirectUri);
		AuthorizationCodeResourceDetails resource = new AuthorizationCodeResourceDetails();

		String mhpTokenUrl = resourceDirectory.getLink().getLinkByTitle(LinkTitles.OAuthToken).getHref().toString();
		resource.setAccessTokenUri(mhpTokenUrl);
		resource.setClientId(propertyHelper.getProperty(CLIENT_ID));
		resource.setScope(Arrays.asList(propertyHelper.getProperty(SCOPE)));
		resource.setClientSecret(propertyHelper.getProperty(CLIENT_SECRET));
		resource.setClientAuthenticationScheme(AuthenticationScheme.query);

		AccessTokenRequest request = new DefaultAccessTokenRequest();
		request.setAuthorizationCode(code);
		request.setPreservedState(new Object());
		request.setAll(formData.toSingleValueMap());
		
		OAuth2RestTemplate template = new OAuth2RestTemplate(resource, new DefaultOAuth2ClientContext(request));
		addJSONMessageConverter(template);
		
		OAuth2AccessToken accessToken = template.getAccessToken();

		updateICN(accessToken);
		
		return accessToken.getValue();
	}

	// This was added because of incompatability introduced by Yammer metrics library. Yammer metrics library has dependency on Jackson library and it was 
	// interfering with object mapper 
	private void addJSONMessageConverter(OAuth2RestTemplate template) {
		List<HttpMessageConverter<?>> messageConverters = new ArrayList<HttpMessageConverter<?>>();
		MappingJacksonHttpMessageConverter jsonMessageConverter = new MappingJacksonHttpMessageConverter();
		jsonMessageConverter.setObjectMapper(new ObjectMapper());
		messageConverters.add(jsonMessageConverter);

		AuthorizationCodeAccessTokenProvider authorizationCodeAccessTokenProvider = new AuthorizationCodeAccessTokenProvider();
		authorizationCodeAccessTokenProvider.setMessageConverters(messageConverters);
		
		AccessTokenProviderChain accessTokenProvider = new AccessTokenProviderChain(Arrays.<AccessTokenProvider> asList(
				authorizationCodeAccessTokenProvider, new ImplicitAccessTokenProvider(),
				new ResourceOwnerPasswordAccessTokenProvider(), new ClientCredentialsAccessTokenProvider()));
		
		template.setAccessTokenProvider(accessTokenProvider);
	}
	
	
	private ResourceDirectory buildResourceDirectory(HttpServletRequest req) {
		URI baseUri = UriBuilder.fromUri(buildBaseUriString(req)).build();
		URI restBaseUri = UriBuilder.fromUri(baseUri).path("rest").build();
		URI requestUri = UriBuilder.fromUri(restBaseUri).path("public").path("resource-directory").build();
		
		return resourceDirectoryBuilder.getResourceDirectory(restBaseUri, requestUri);
	}
	
	private String buildBaseUriString(HttpServletRequest req) {
		StringBuilder sb = new StringBuilder();
		sb.append(req.getScheme());
		sb.append("://");
		sb.append(req.getServerName());
		sb.append(":");
		sb.append(req.getServerPort());
		sb.append(req.getContextPath());
		
		return sb.toString();
	}
	
	private void updateICN(OAuth2AccessToken accessToken) {
		logger.debug("inside updateICN"); 
		Authentication  authentication = haTokenValidation.loadAuthentication(accessToken.getValue());
		if (authentication != null) {
            MhpUser user = MhpUserFactory.createFromAuthentication(authentication);
            ExecutorService executor = Executors.newFixedThreadPool(1);
            Callable<String> icnService = new PatientIcnService(user.getUserIdentifier().getUniqueId());
            Future<String> resultId = executor.submit(icnService);
            executor.shutdown();
            logger.debug("saved icn");
		}
	}
	
	@RequestMapping(value = "/denied", method = RequestMethod.GET)
	public ModelAndView getLoginPage(HttpServletRequest req, ModelMap model) {

		model.put("error", "You have entered an invalid username or password!");
		constructLoginPageContents(model);

		return new ModelAndView("userLogin", model);
	}

	@RequestMapping(value = "/logout", method = RequestMethod.GET)
	public void logout(HttpServletRequest request, HttpServletResponse response, ModelMap model) {
		String contextPath = NullChecker.isNotNullish(request.getContextPath()) ? request.getContextPath() : "";
		Authentication currentUser = getCurrentUser();
		invalidateRememberMeTokens(request, response, currentUser);
		invalidateCookies(response, contextPath);
		HttpSession session = request.getSession(false);
		if (session != null) {
			session.invalidate();
		}
		SecurityContextHolder.clearContext();
		
		String ssoeLogoutUrl = propertyHelper.getProperty(KEY_SSOE_LOGOUT_URL);

		String redirectUri = request.getParameter(KEY_REDIRECT_URI);
		redirectUri = determineRedirectUri(contextPath, ssoeLogoutUrl,
				redirectUri);
		redirect(response, redirectUri);
	}

	private String determineRedirectUri(String contextPath, String ssoeLogoutUrl, String redirectUri) {

		if(NullChecker.isNotNullish(ssoeLogoutUrl)){
			if(NullChecker.isNotNullish(redirectUri)){
				if(redirectUri.contains("mhplaunchPad")){
					redirectUri = ssoeLogoutUrl + "?filename=cih-logout.html";
				}else{
					redirectUri = ssoeLogoutUrl + "?filename=mbb-logout.html";
				}
			}else{
				if (ModeHelper.isProviderMode()){
					redirectUri = ssoeLogoutUrl + "?redirect_uri="+contextPath + "/launchpad/";
				}else{
					redirectUri = ssoeLogoutUrl + "?filename=mbb-logout.html";
				}
			}
		}else if (NullChecker.isNullish(redirectUri)){
			redirectUri = contextPath + "/launchpad/";
		}
		return redirectUri;
	}
	
	private void redirect(HttpServletResponse response, String redirectUri){

        if(redirectSecurityIsEnabled() && !redirectUriIsPermitted(redirectUri)) {
            logger.debug("Redirect URI path is not white-listed");
            throw new WebApplicationException(Response.Status.BAD_REQUEST);
        }

		try {
			response.sendRedirect(redirectUri);
		} catch (IOException e) {
			throw new RuntimeException("Exception while redirecting");
		}
	}
	
	private void invalidateRememberMeTokens(HttpServletRequest request, HttpServletResponse response, Authentication authentication){
		try {
			AbstractRememberMeServices rememberMeServices = (AbstractRememberMeServices) applicationContext
					.getBean("rememberMeServices");
			rememberMeServices.logout(request, response, authentication);
		} catch (NoSuchBeanDefinitionException exception) {
			// This exception is ignored as there is no need to clean up remember me services if it is not configured
		}
	}
	
	private void invalidateCookies(HttpServletResponse response, String contextPath){
		List<String> cookieNames = new LinkedList<String>();
		cookieNames.add("SPRING_SECURITY_REMEMBER_ME_COOKIE");
		cookieNames.add("JSESSIONID");
		for (String cookieName : cookieNames) {
			Cookie cookie = new Cookie(cookieName, null);
			cookie.setMaxAge(0);
			cookie.setPath(contextPath);
			response.addCookie(cookie);
		}
	}

	private Authentication getCurrentUser() {
		SecurityContext ctx = SecurityContextHolder.getContext();
		return ctx.getAuthentication();
	}

	private void constructErrorMessageIfErrorExists(ModelMap model, boolean error) {
		if (error == true) {
			model.put("error", "You have entered an invalid username or password!");
		} else {
			model.put("error", "");
		}
	}

	private void constructLoginPageContents(ModelMap model) {
		model.put("haMode", haMode);
		model.put("vistaLocation", "");
		model.put("availableVistaLocations", availableVistaLocations);
	}

    /*
     * This method ensures that the URI matches permitted base uris
     * using properties configured on the classpath.
     *
     * The property associated with KEY_REDIRECT_URI_ROOTS can be a comma
     * separated list of base URIs
     *
     * @param redirectUri
     */
    boolean redirectUriIsPermitted(String redirectUri) {
    	Locale.setDefault(Locale.ENGLISH);
    	
    	if(NullChecker.isNullish(redirectUri))
            return false;

        String redirectUriRoots = propertyHelper.getProperty(KEY_REDIRECT_URI_ROOTS);

        if(NullChecker.isNotNullish(redirectUriRoots)) {
            String[] roots = redirectUriRoots.split(",");

            String ignoreCaseRedirectUri = redirectUri.toLowerCase();
            for (String root : roots) {
                String ignoreCaseRoot = root.toLowerCase();
                if (ignoreCaseRedirectUri.startsWith(ignoreCaseRoot))
                    return true;
            }
        }

        return false;
    }

    boolean redirectSecurityIsEnabled() {
        String securityEnabled = propertyHelper.getProperty(KEY_ENABLE_REDIRECT_SECURITY);

        return "true".equalsIgnoreCase(securityEnabled);
    }


}
