package com.agilex.healthcare.mobilehealthplatform.restservice;

import java.io.IOException;
import java.io.OutputStream;
import java.net.URISyntaxException;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.Consumes;
import javax.ws.rs.DELETE;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.WebApplicationException;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.Status;
import javax.ws.rs.core.StreamingOutput;
import javax.ws.rs.core.UriInfo;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.MessageSource;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import com.agilex.healthcare.mbb.dataservice.PdfDataService;
import com.agilex.healthcare.mobilehealthplatform.datalayer.securemessage.SecureMessageDataService;
import com.agilex.healthcare.mobilehealthplatform.datalayer.securemessage.SecureMessageFilter;
import com.agilex.healthcare.mobilehealthplatform.domain.Patient;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.SecureMessage;
import com.agilex.healthcare.mobilehealthplatform.domain.SecureMessageCode;
import com.agilex.healthcare.mobilehealthplatform.domain.SecureMessageContacts;
import com.agilex.healthcare.mobilehealthplatform.domain.SecureMessageFolders;
import com.agilex.healthcare.mobilehealthplatform.domain.SecureMessageUser;
import com.agilex.healthcare.mobilehealthplatform.domain.SecureMessages;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.Domain;
import com.agilex.healthcare.mobilehealthplatform.utils.PropertyHelper;
import com.agilex.healthcare.mobilehealthplatform.utils.SessionStateHelper;
import com.agilex.healthcare.mobilehealthplatform.utils.uriformaters.linkbuilder.SecureMessageContactLinkBuilder;
import com.agilex.healthcare.mobilehealthplatform.utils.uriformaters.linkbuilder.SecureMessageLinkBuilder;

@Path("/securemessage-service/{assigning-authority}/{user-id}/")
@Component
@Scope("request")
public class SecureMessageResource {

	protected static final org.apache.commons.logging.Log LOGGER = org.apache.commons.logging.LogFactory.getLog(SecureMessageResource.class);
	
	@Autowired
	MessageSource messageSource;

	// JUSTIFICATION: Not using @Resource SessionStateHelper as oplock must be updated to read again.
	SessionStateHelper sessionStateHelper = new SessionStateHelper(true);
	
	@Resource
	private PropertyHelper propertyHelper;
	
	/**
	 * Generates a PDF for the specified {@link SecureMessage}.
	 * This is a stateful call, so the same message must have been read by the same HTTP session.
	 * 
	 * @param assigningAuthority
	 *            The authority associated with the logged in user
	 * @param userId
	 *            The userId tied to the assigningAuthority
	 * @param messageId
	 *            The messageId tied to the {@link SecureMessage}
	 */
	@GET
	@Produces("application/pdf")
	@Path("messages/id/{message-id}")
	public Response getReport(@PathParam("assigning-authority") String assigningAuthority, @PathParam("user-id") String userId, @PathParam("message-id") String messageId, @Context UriInfo uriInfo, @Context HttpHeaders headers, @Context HttpServletRequest req) throws URISyntaxException {
		String cacheKey = Domain.secureMessage + "," + messageId;
		Object cache = sessionStateHelper.checkCache(req.getSession(), cacheKey, 3);
		if (cache == null || !(cache instanceof SecureMessage)) {
			LOGGER.error("Cached instance not found");
			throw new WebApplicationException(Status.PRECONDITION_FAILED);
		}
		StreamingOutput stream = generatePdfToStreamingOutput((SecureMessage)cache);
		
		return Response.ok(stream).build();
	}
	
	private StreamingOutput generatePdfToStreamingOutput(final SecureMessage secureMessage) {
		StreamingOutput stream = new StreamingOutput() {
			public void write(OutputStream outputStream) throws IOException, WebApplicationException {
				try {
					PdfDataService dataService = new PdfDataService();
					dataService.generateSecureMessagePdf(outputStream, messageSource, secureMessage);
				} catch (Exception e) {
					LOGGER.error("Error generating secure message PDF", e);
					throw new WebApplicationException(Status.INTERNAL_SERVER_ERROR);
				}
			}
		};
		return stream;
	}
	
	/**
	 * Retrieves a user with all required information used to interact with MHV
	 * Secure Messaging. This call provides {@link SecureMessageFolders} and
	 * {@link SecureMessageContacts} associated with the user.
	 * 
	 * @param assigningAuthority
	 *            The authority associated with the logged in user.
	 * @param userId
	 *            The userId tied to the assigningAuthority
	 */
	@GET
	@Produces({ "application/xml", "application/json" })
	@Path("secure-message-user")
	public SecureMessageUser fetchSecureMessageUser(@PathParam("assigning-authority") String assigningAuthority, @PathParam("user-id") String userId, @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		PatientIdentifier userIdentifier = new PatientIdentifier(assigningAuthority, userId);

		SecureMessageDataService dataService = new SecureMessageDataService(propertyHelper);
		SecureMessageUser secureMessageUser = dataService.fetchSecureMessageUser(userIdentifier);

		SecureMessageUserLinkBuilder linkBuilder = new SecureMessageUserLinkBuilder(uriInfo.getBaseUri());
		linkBuilder.fillLinks(secureMessageUser, uriInfo.getRequestUri());

		return secureMessageUser;
	}

	/**
	 * Retrieves all valid recipients ({@link SecureMessageContacts}) associated
	 * with a {@link SecureMessageUser}
	 * 
	 * @param assigningAuthority
	 *            The authority associated with the logged in user.
	 * @param userId
	 *            The userId tied to the assigningAuthority
	 */
	@GET
	@Produces({ "application/xml", "application/json" })
	@Path("contacts")
	public SecureMessageContacts fetchContacts(@PathParam("assigning-authority") String assigningAuthority, @PathParam("user-id") String userId, @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		PatientIdentifier userIdentifier = new PatientIdentifier(assigningAuthority, userId);

		SecureMessageDataService dataService = new SecureMessageDataService(propertyHelper);
		SecureMessageContacts contacts = dataService.fetchContacts(userIdentifier);

		SecureMessageContactLinkBuilder linkBuilder = new SecureMessageContactLinkBuilder(uriInfo.getBaseUri());
		linkBuilder.fillLinks(contacts, uriInfo.getRequestUri());

		return contacts;
	}

	/**
	 * Retrieves all folders associated with the {@link SecureMessageUser}.
	 * 
	 * @param assigningAuthority
	 *            The authority associated with the logged in user.
	 * @param userId
	 *            The userId tied to the assigningAuthority
	 */
	@GET
	@Produces({ "application/xml", "application/json" })
	@Path("folders")
	public SecureMessageFolders fetchFolders(@PathParam("assigning-authority") String assigningAuthority, @PathParam("user-id") String userId, @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		PatientIdentifier userIdentifier = new PatientIdentifier(assigningAuthority, userId);

		SecureMessageDataService dataService = new SecureMessageDataService(propertyHelper);
		SecureMessageFolders folders = dataService.fetchFolders(userIdentifier);

		SecureMessageFolderLinkBuilder linkBuilder = new SecureMessageFolderLinkBuilder(uriInfo.getBaseUri());
		linkBuilder.fillLinks(folders, uriInfo.getRequestUri());

		return folders;
	}

	/**
	 * A resource that returns the {@link SecureMessages} that reside in the
	 * specified folder for a given {@link Patient}.
	 * 
	 * This method accepts integer values pageSize and pageNumber. If these
	 * values are not provided, they default to a pageSize of 25 and pageNumber
	 * of 0.
	 * 
	 * @param assigningAuthority
	 *            The authority associated with the logged in user.
	 * @param userId
	 *            The userId tied to the assigningAuthority
	 * @param folderId
	 *            The id associated with the folder to request messages for
	 */
	@GET
	@Produces({ "application/xml", "application/json" })
	@Path("folders/id/{folder-id}/messages")
	public SecureMessages fetchMessagesInFolder(@PathParam("assigning-authority") String assigningAuthority, @PathParam("user-id") String userId, @PathParam("folder-id") String folderId, @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		SecureMessageFilter filter = new SecureMessageFilter(uriInfo.getRequestUri(), folderId);

		PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, userId);
		SecureMessages secureMessages = fetchMessages(patientIdentifier, filter, uriInfo);

		generateMessageLinks(secureMessages, folderId, uriInfo);

		return secureMessages;
	}

	void generateMessageLinks(SecureMessages secureMessages, String folderId, UriInfo uriInfo) {
		if (SecureMessageCode.FOLDER_ID_DRAFT.toString().equals(folderId)) {
			SecureMessageDraftLinkBuilder linkBuilder = new SecureMessageDraftLinkBuilder(uriInfo.getBaseUri());
			linkBuilder.fillLinks(secureMessages, uriInfo.getRequestUri());
		} else {
			SecureMessageLinkBuilder linkBuilder = new SecureMessageLinkBuilder(uriInfo.getBaseUri());
			linkBuilder.fillLinks(secureMessages, uriInfo.getRequestUri());
		}
	}

	/**
	 * A resource that sends a {@link SecureMessage} for a given {@link Patient}
	 * 
	 * @param assigningAuthority
	 *            The authority associated with the logged in user.
	 * @param userId
	 *            The userId tied to the assigningAuthority
	 * @param secureMessage
	 */
	@POST
	@Path("messages")
	@Produces({ "application/xml", "application/json" })
	@Consumes({ "application/xml", "application/json" })
	public SecureMessage sendMessage(SecureMessage secureMessage, @PathParam("assigning-authority") String assigningAuthority, @PathParam("user-id") String userId, @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, userId);

		secureMessage.setPatientIdentifier(patientIdentifier);

		SecureMessageDataService dataService = new SecureMessageDataService(propertyHelper);
		SecureMessage newMessage = dataService.sendSecureMessage(patientIdentifier, secureMessage);

		SecureMessageLinkBuilder linkBuilder = new SecureMessageLinkBuilder(uriInfo.getBaseUri());
		linkBuilder.fillLinks(newMessage, uriInfo.getRequestUri());
		return newMessage;
	}

	/**
	 * Replies to an existing {@link SecureMessage}
	 * 
	 * @param assigningAuthority
	 *            The authority associated with the logged in user
	 * @param userId
	 *            The userId tied to the assigningAuthority
	 */
	@POST
	@Path("messages/id/{message-id}")
	@Produces({ "application/xml", "application/json" })
	@Consumes({ "application/xml", "application/json" })
	public SecureMessage replyMessage(SecureMessage message, @PathParam("assigning-authority") String assigningAuthority, @PathParam("user-id") String userId, @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, userId);

		SecureMessageDataService dataService = new SecureMessageDataService(propertyHelper);
		SecureMessage messageReply = dataService.replySecureMessage(patientIdentifier, message);

		SecureMessageLinkBuilder linkBuilder = new SecureMessageLinkBuilder(uriInfo.getBaseUri());
		linkBuilder.fillLinks(messageReply, uriInfo.getRequestUri());

		return messageReply;
	}

	/**
	 * A resource that retrieves a {@link SecureMessage} with the message body
	 * for a given user
	 * 
	 * @param assigningAuthority
	 *            The authority associated with the logged in user.
	 * @param userId
	 *            The userId tied to the assigningAuthority
	 * @param messageId
	 *            The id of the message to be retrieved
	 * @param addresseeId
	 *            The addresseeId of the message to be retrieved
	 * @param addresseeOplock
	 *            The addresseeOplock of the message to be retrieved
	 */
	@PUT
	@Produces({ "application/xml", "application/json" })
	@Consumes({ "application/xml", "application/json" })
	@Path("messages/id/{message-id}/addressee-id/{addressee-id}/addressee-oplock/{addressee-oplock}")
	public SecureMessage readMessage(SecureMessage secureMessage, @PathParam("assigning-authority") String assigningAuthority, @PathParam("user-id") String userId, @PathParam("message-id") String messageId, @PathParam("addressee-id") String addresseeId, @PathParam("addressee-oplock") String addresseeOplock, @Context UriInfo uriInfo, @Context HttpHeaders headers, @Context HttpServletRequest req) {
		PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, userId);

		SecureMessageDataService dataservice = new SecureMessageDataService(propertyHelper);
		SecureMessage readMessage = dataservice.readSecureMessage(patientIdentifier, secureMessage);
		readMessage = fixMessage(secureMessage, readMessage);
		
		if (SecureMessageCode.FOLDER_ID_DRAFT.toString().equals(readMessage.getFolderId())) {
			SecureMessageDraftLinkBuilder linkBuilder = new SecureMessageDraftLinkBuilder(uriInfo.getBaseUri());
			linkBuilder.fillLinks(readMessage, uriInfo.getRequestUri());
		} else {
			SecureMessageLinkBuilder linkBuilder = new SecureMessageLinkBuilder(uriInfo.getBaseUri());
			linkBuilder.fillLinks(readMessage, uriInfo.getRequestUri());
		}

		String cacheKey = Domain.secureMessage + "," + readMessage.getDataIdentifier().getUniqueId();
                sessionStateHelper.addCacheMapToSession(req.getSession(), cacheKey, readMessage);
        
		return readMessage;
	}

	

	/**
	 * A resource that deletes a {@link SecureMessage} for a given user. This is
	 * synonymous with moving a message to the Deleted folder.
	 * 
	 * @param assigningAuthority
	 *            The authority associated with the logged in user.
	 * @param userId
	 *            The userId tied to the assigningAuthority
	 * @param messageId
	 *            The id of the message to be removed.
	 * 
	 */
	@DELETE
	@Path("messages/id/{message-id}")
	@Consumes({ "application/xml", "application/json" })
	public void deleteMessage(@PathParam("assigning-authority") String assigningAuthority, @PathParam("user-id") String userId, @PathParam("message-id") String messageId, @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		PatientIdentifier userIdentifier = new PatientIdentifier(assigningAuthority, userId);
		SecureMessageDataService dataService = new SecureMessageDataService(propertyHelper);

		dataService.moveSecureMessage(userIdentifier, messageId, SecureMessageCode.FOLDER_ID_DELETED.toString());
	}

	/**
	 * Moves a message from one folder to another. The body of the POST must be
	 * a SecureMessage with the updated folderId associated with the folder that
	 * the message should be moved to.
	 * 
	 * @param secureMessage
	 *            The POST body representing a {@link SecureMessage} object
	 * @param assigningAuthority
	 *            The authority associated with the logged in user.
	 * @param userId
	 *            The userId tied to the assigningAuthority
	 * @param messageId
	 *            The messageId of the message being modified
	 */
	@POST
	@Path("messages/id/{message-id}/move")
	@Produces({ "application/xml", "application/json" })
	@Consumes({ "application/xml", "application/json" })
	public SecureMessage moveMessage(SecureMessage secureMessage, @PathParam("assigning-authority") String assigningAuthority, @PathParam("user-id") String userId, @PathParam("message-id") String messageId, @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		PatientIdentifier userIdentifier = new PatientIdentifier(assigningAuthority, userId);

		SecureMessageDataService dataService = new SecureMessageDataService(propertyHelper);
		SecureMessage result = dataService.moveSecureMessage(userIdentifier, messageId, secureMessage.getFolderId());
		return result;
	}

	/**
	 * Moves a message from one folder to another. The body of the POST must be
	 * a SecureMessage with the updated folderId associated with the folder that
	 * the message should be moved to.
	 * 
	 * @param secureMessage
	 *            The POST body representing a {@link SecureMessage} object
	 * @param assigningAuthority
	 *            The authority associated with the logged in user.
	 * @param userId
	 *            The userId tied to the assigningAuthority
	 * @param messageId
	 *            The messageId of the message being modified
	 */
	@PUT
	@Path("messages/id/{message-id}/move")
	@Produces({ "application/xml", "application/json" })
	@Consumes({ "application/xml", "application/json" })
	public SecureMessage updateMessageByMovingToFolders(SecureMessage secureMessage, @PathParam("assigning-authority") String assigningAuthority, @PathParam("user-id") String userId, @PathParam("message-id") String messageId, @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		PatientIdentifier userIdentifier = new PatientIdentifier(assigningAuthority, userId);

		SecureMessageDataService dataService = new SecureMessageDataService(propertyHelper);
		SecureMessage result =  dataService.moveSecureMessage(userIdentifier, messageId, secureMessage.getFolderId());
		return result;
	}

	/**
	 * GET method that retrieves all {@link SecureMessages} in the Drafts
	 * folder. This should be used for drafts instead of the GET
	 * /folders/id/{folder-id}/messages method.
	 * 
	 * @param assigningAuthority
	 *            The authority associated with the logged in user.
	 * @param userId
	 *            The userId tied to the assigningAuthority
	 */
	@GET
	@Path("drafts/messages")
	@Produces({ "application/xml", "application/json" })
	public SecureMessages fetchSecureMessageDrafts(@PathParam("assigning-authority") String assigningAuthority, @PathParam("user-id") String userId, @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		SecureMessageFilter draftFilter = new SecureMessageFilter(SecureMessageCode.FOLDER_ID_DRAFT.toString());
		SecureMessages draftMessages = fetchMessages(new PatientIdentifier(assigningAuthority, userId), draftFilter, uriInfo);

		SecureMessageDraftLinkBuilder linkBuilder = new SecureMessageDraftLinkBuilder(uriInfo.getBaseUri());
		linkBuilder.fillLinks(draftMessages, uriInfo.getRequestUri());

		return draftMessages;
	}

	
	/**
	 * Saves an existing draft {@link SecureMessage}
	 * 
	 * @param secureMessage
	 *            An existing {@link SecureMessage} to save. This will be associated with
	 *            the drafts folder
	 * @param assigningAuthority
	 *            The authority associated with the logged in user
	 * @param userId
	 *            The userId tied to the assigningAuthority
	 * 
	 */
	@PUT
	@Path("drafts/messages")
	@Produces({ "application/xml", "application/json" })
	@Consumes({ "application/xml", "application/json" })
	public SecureMessage saveExistingDraftMessage(SecureMessage secureMessage, @PathParam("assigning-authority") String assigningAuthority, @PathParam("user-id") String userId, @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		return saveDraft(secureMessage, assigningAuthority, userId, uriInfo, headers);
	}
	
	/**
	 * Saves a new draft {@link SecureMessage} and associates it with the Draft
	 * system folder.
	 * 
	 * @param secureMessage
	 *            A {@link SecureMessage} to save. This will be associated with
	 *            the drafts folder
	 * @param assigningAuthority
	 *            The authority associated with the logged in user
	 * @param userId
	 *            The userId tied to the assigningAuthority
	 * 
	 */
	@POST
	@Path("drafts/messages")
	@Produces({ "application/xml", "application/json" })
	@Consumes({ "application/xml", "application/json" })
	public SecureMessage saveDraftMessage(SecureMessage secureMessage, @PathParam("assigning-authority") String assigningAuthority, @PathParam("user-id") String userId, @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		return saveDraft(secureMessage, assigningAuthority, userId, uriInfo, headers);
	}
	
	public SecureMessage saveDraft(SecureMessage secureMessage, @PathParam("assigning-authority") String assigningAuthority, @PathParam("user-id") String userId, @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		PatientIdentifier userIdentifier = new PatientIdentifier(assigningAuthority, userId);

		SecureMessageDataService dataService = new SecureMessageDataService(propertyHelper);
		SecureMessage savedDraft = dataService.saveSecureMessageDraft(userIdentifier, secureMessage);

		SecureMessageDraftLinkBuilder linkBuilder = new SecureMessageDraftLinkBuilder(uriInfo.getBaseUri());
		linkBuilder.fillLinks(savedDraft, uriInfo.getRequestUri());

		return savedDraft;
	}
	

	/**
	 * Sends the draft message
	 * 
	 * @param assigningAuthority
	 *            The authority associated with the logged in user
	 * @param userId
	 *            The userId tied to the assigningAuthority
	 * @param messageId
	 *            The messageId associated with the draft to send
	 */
	@PUT
	@Path("drafts/messages/id/{message-id}")
	@Produces({ "application/xml", "application/json" })
	@Consumes({ "application/xml", "application/json" })
	public SecureMessage sendDraftMessage(SecureMessage secureMessage, @PathParam("assigning-authority") String assigningAuthority, @PathParam("user-id") String userId, @PathParam("message-id") String messageId, @Context UriInfo uriInfo, @Context HttpHeaders headers) {

		PatientIdentifier userIdentifier = new PatientIdentifier(assigningAuthority, userId);
		SecureMessageDataService dataService = new SecureMessageDataService(propertyHelper);
		SecureMessage sentMessage = dataService.sendSecureMessageDraft(userIdentifier, secureMessage);

		SecureMessageLinkBuilder linkBuilder = new SecureMessageLinkBuilder(uriInfo.getBaseUri());
		linkBuilder.fillLinks(sentMessage, uriInfo.getRequestUri());

		return sentMessage;
	}

	/**
	 * Deletes the draft message
	 * 
	 * @param assigningAuthority
	 *            The authority associated with the logged in user
	 * @param userId
	 *            The userId tied to the assigningAuthority
	 * @param messageId
	 *            The messageId associated with the draft to delete
	 */
	@DELETE
	@Path("drafts/messages/id/{message-id}")
	@Consumes({ "application/xml", "application/json" })
	public void deleteDraftMessage(@PathParam("assigning-authority") String assigningAuthority, @PathParam("user-id") String userId, @PathParam("message-id") String messageId, @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		PatientIdentifier userIdentifier = new PatientIdentifier(assigningAuthority, userId);

		SecureMessageDataService dataService = new SecureMessageDataService(propertyHelper);
		dataService.deleteSecureMessageDraft(userIdentifier, messageId);
	}

	private SecureMessages fetchMessages(PatientIdentifier patientIdentifier, SecureMessageFilter filter, UriInfo uriInfo) {
		SecureMessageDataService dataservice = new SecureMessageDataService(propertyHelper);
		SecureMessages messages = dataservice.fetchSecureMessages(patientIdentifier, filter);

		return messages;
	}
	
	/*
	 * The production read call does not return back all metadata, so we copy the original message and grab
	 * the fields that matter (oplock, message body) off of the response of the read call.
	 */
	private SecureMessage fixMessage(SecureMessage secureMessage, SecureMessage readMessage) {
		SecureMessage copy = new SecureMessage(secureMessage);
		copy.setAddresseeOplock(readMessage.getAddresseeOplock());
		copy.setMessageBody(readMessage.getMessageBody());
		
		return copy;
	}
	

}
