package com.agilex.healthcare.mobilehealthplatform.restservice;

import java.net.URI;

import javax.annotation.Resource;
import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.UriInfo;

import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import com.agilex.healthcare.mobilehealthplatform.domain.AtomLink;
import com.agilex.healthcare.mobilehealthplatform.domain.ResourceDirectory;
import com.agilex.healthcare.mobilehealthplatform.utils.QueryParameterHelper;
import com.agilex.healthcare.mobilehealthplatform.utils.uriformaters.ResourceDirectoryBuilder;

@Path("/public/resource-directory")
@Component
@Scope("request")
public class ResourceDirectoryResource {

	@Resource
	ResourceDirectoryBuilder resourceDirectoryBuilder;

	/**
	 * This resource returns a list of the "root" resources in the system. This
	 * resource should be the only thing known to client applications - every
	 * other resource should be discovered.
	 * 
	 * Support formats include:<br>
	 * application/atom+xml<br>
	 * application/xml<br>
	 * application/json
	 * 
	 * @return resource-directory containing each of the system resources
	 */
	@GET
	@Produces({ "application/atom+xml", "application/json", "application/xml" })
	public ResourceDirectory fetchResourceDirectory(@Context UriInfo u, @Context HttpHeaders headers) {
		ResourceDirectory resourceDirectory = resourceDirectoryBuilder.getResourceDirectory(u.getBaseUri(), u.getRequestUri());
		return resourceDirectory;
	}

	/**
	 * Use resource-directory (with "Accept: application/json") instead
	 * 
	 * @return json representation of the resource directory
	 */
	@Deprecated
	@GET
	@Path("json")
	@Produces({ "application/json" })
	public ResourceDirectory getResourcesAsJsonOnSubpath(@Context UriInfo u, @Context HttpHeaders headers) {
		ResourceDirectory resourceDirectory = resourceDirectoryBuilder.getResourceDirectory(u.getBaseUri(), u.getRequestUri());
		return resourceDirectory;
	}

	@Path("{link-title}")
	@GET
	@Produces({ "application/json", "application/xml" })
	public AtomLink fetchSingleResourceLink(@PathParam("link-title") String linkTitle, @QueryParam("redirect") String redirect, @Context UriInfo u, @Context HttpHeaders headers) {
		ResourceDirectory resourceDirectory = resourceDirectoryBuilder.getResourceDirectory(u.getBaseUri(), u.getRequestUri());
		AtomLink link = resourceDirectory.getLink().getLinkByTitle(linkTitle);
		if (link == null) {
			throw ExceptionFactory.generateNoDataFoundException();
		}
		if (shouldRedirect(redirect)) {
			throw ExceptionFactory.generateRedirect(link.getHref());
		}

		return link;
	}

	@Path("{link-title}/href")
	@GET
	@Produces({ "text/plain", "application/xml", "application/json" })
	public URI fetchSingleResourceUri(@PathParam("link-title") String linkTitle, @QueryParam("redirect") String redirect, @Context UriInfo u, @Context HttpHeaders headers) {
		AtomLink link = fetchSingleResourceLink(linkTitle, redirect, u, headers);
		return link.getHref();
	}

	private boolean shouldRedirect(String redirectQueryParamValue) {
		return QueryParameterHelper.parseQueryParamAsBoolean(redirectQueryParamValue, false, true);
	}
}
