package com.agilex.healthcare.mobilehealthplatform.restservice;

import javax.annotation.Resource;
import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.CacheControl;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.UriInfo;

import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import com.agilex.healthcare.mobilehealthplatform.datalayer.rightofaccess.RightOfAccessDataService;
import com.agilex.healthcare.mobilehealthplatform.domain.MhpUser;
import com.agilex.healthcare.mobilehealthplatform.domain.UserSession;
import com.agilex.healthcare.mobilehealthplatform.security.MhpUserFactory;
import com.agilex.healthcare.utility.NullChecker;

@Path("/public/user-session")
@Component
@Scope("request")
public class PublicUserSessionResource {

	@Resource(name = "sessionTimeoutInSeconds")
	Integer sessionTimeoutInSeconds;
	
	@GET
	@Produces({ "application/xml", "application/json" })
	public Response getUserSession(@QueryParam("clientTimeoutOverride") Integer clientTimeoutOverride, @Context UriInfo u, @Context HttpHeaders headers) {
		UserSession userSession = new UserSession();

		MhpUser mhpUser = MhpUserFactory.createFromSecurityContext();

		if (mhpUser != null) {
			mhpUser.setDisplayName(calculateDisplayName(mhpUser));
			mhpUser = mergeWithRightOfAccessInfo(mhpUser);
			userSession.setMhpUser(mhpUser);

			if (clientTimeoutOverride != null) {
				userSession.setClientTimeout(clientTimeoutOverride);
			} else {
				userSession.setClientTimeout(sessionTimeoutInSeconds/60);
			}
		}

		// IE 9 was caching this resource
		CacheControl cc = new CacheControl();
		cc.setMaxAge(-1);
		cc.setNoCache(true);
		  
		return Response.ok(userSession).cacheControl(cc).build();
	}

	public MhpUser mergeWithRightOfAccessInfo(MhpUser mhpUser) {
		if(MhpUserFactory.doesUserHasStaffRole()){
			mhpUser.setRightOfAccessAccepted(true);
		}else{
			RightOfAccessDataService dataService = new RightOfAccessDataService();
			mhpUser.setRightOfAccessAccepted(dataService.hasAcceptedRightOfAccess(mhpUser.getPatientId()));
		}
		
		return mhpUser;
	}
	
	private String calculateDisplayName(MhpUser user) {
		String displayName = null;

		if (NullChecker.isNotNullish(user.getLastName()) && NullChecker.isNotNullish(user.getFirstName())) {
			displayName = user.getLastName() + ", " + user.getFirstName();
		}

		return displayName;
	}

}
