package com.agilex.healthcare.mobilehealthplatform.restservice;

import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.UriInfo;

import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import com.agilex.healthcare.mobilehealthplatform.datalayer.assessment.AssessmentDataService;
import com.agilex.healthcare.mobilehealthplatform.domain.Assessment;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentQuestion;
import com.agilex.healthcare.mobilehealthplatform.domain.Assessments;
import com.agilex.healthcare.mobilehealthplatform.utils.uriformaters.linkbuilder.AssessmentLinkBuilder;

@Path("public/assessments")
@Component
@Scope("request")
public class PublicAssessmentResource {
	private AssessmentDataService dataService = new AssessmentDataService();

	/**
	 * A resource that either returns all active {@link Assessments}, or a specific {@link Assessment} as specified by the parameter
	 * @param searchString
	 * The search string for a specific assessment
	 * @return Returns all active {@link Assessments}
	 */
	@GET
	@Produces({ "application/xml", "application/json" })
	public Assessments fetchAssessments(@QueryParam("assessmentID") String searchString, @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		Assessments assessments = new Assessments();
		if (searchString != null) {
			assessments.add(dataService.fetchAssessment(searchString));
		} else {
			assessments = dataService.fetchActiveAssessments();
		}
		AssessmentLinkBuilder linkBuilder = new AssessmentLinkBuilder(uriInfo.getBaseUri());
		return linkBuilder.fillLinks(assessments, uriInfo.getRequestUri());
	}

	/**
	 * A resource that returns {@link Assessments} associated with the HealthAssessment application
	 * @return Returns all {@link Assessments} associated with the HealthAssessment application
	 */
	@GET
	@Produces({ "application/xml", "application/json" })
	@Path("health-assessment")
	public Assessments fetchAssessmentsForHealthAssessment(@Context UriInfo uriInfo, @Context HttpHeaders headers) {
		Assessments assessments = dataService.fetchAssessmentsForHealthAssessment();
		AssessmentLinkBuilder linkBuilder = new AssessmentLinkBuilder(uriInfo.getBaseUri());
		return linkBuilder.fillLinks(assessments, uriInfo.getRequestUri());
	}

	/**
	 * A resource that returns the indicated {@link Assessment}
	 * 
	 * @return Returns a given {@link Assessment}
	 */
	@GET
	@Produces({ "application/xml", "application/json" })
	@Path("id/{assessmentId}")
	public Assessment fetchAssessment(@PathParam("assessmentId") String assessmentId, @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		Assessment assessment = dataService.fetchAssessment(assessmentId);
		AssessmentLinkBuilder linkBuilder = new AssessmentLinkBuilder(uriInfo.getBaseUri());
		return linkBuilder.fillLinks(assessment, uriInfo.getRequestUri());
	}

	/**
	 * A resource that returns the indicated {@link AssessmentQuestion}
	 * 
	 * @return Returns a question for a given {@link Assessment}
	 */
	@GET
	@Produces({ "application/xml", "application/json" })
	@Path("id/{assessmentId}/questions/id/{questionId}")
	public AssessmentQuestion fetchQuestion(@PathParam("assessmentId") String assessmentId, @PathParam("questionId") String questionId,
			@Context UriInfo uriInfo, @Context HttpHeaders headers) {
		AssessmentQuestion assessmentQuestion = dataService.fetchQuestion(assessmentId, questionId);
		AssessmentLinkBuilder linkBuilder = new AssessmentLinkBuilder(uriInfo.getBaseUri());
		return linkBuilder.fillLinks(assessmentQuestion, uriInfo.getRequestUri());
	}
}