package com.agilex.healthcare.mobilehealthplatform.restservice;

import java.util.Calendar;
import java.util.Date;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.UriInfo;

import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import com.agilex.healthcare.mobilehealthplatform.datalayer.provider.ProviderDataService;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientProvider;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientProviders;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilter;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilterFactory;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.Domain;
import com.agilex.healthcare.mobilehealthplatform.utils.SessionStateHelper;

@Path("/provider-service")
@Component
@Scope("request")
public class ProviderResource {
	protected static final org.apache.commons.logging.Log LOGGER = org.apache.commons.logging.LogFactory.getLog(ProviderResource.class);

	@Resource SessionStateHelper sessionStateHelper;
	
	/**
	 * Returns all primary care {@link PatientProviders} assigned to the patient
	 * 
	 * @param assigningAuthority
	 * @param userId
	 * @param uriInfo
	 * @return
	 */
	@GET
	@Path("/patient/{assigning-authority}/{user-id}/providers/primarycare")
	@Produces({ "application/xml", "application/json" })
	public PatientProviders fetchPrimaryCareProviders(@Context HttpServletRequest req, @PathParam("assigning-authority") String assigningAuthority, 
			@PathParam("user-id") String userId, @QueryParam("facilitySiteCode") String facilitySiteCode, @Context UriInfo uriInfo) {
		String cacheKey = Domain.PrimaryCareProvider + facilitySiteCode;
		PatientProviders providers = null;
		
		Object cache = sessionStateHelper.checkCache(req.getSession(), cacheKey);
		if (cache != null && cache instanceof PatientProviders) {
			LOGGER.debug("Using cached Primary Care Providers");
			providers = (PatientProviders)cache;
		}
		
		if (providers == null) {
			LOGGER.debug("Fetching Primary Care Providers from dataservice");
			try {
				PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, userId);
				ProviderDataService providerDataService = new ProviderDataService();
				providers = providerDataService.fetchPrimaryCareProviders(patientIdentifier, facilitySiteCode);
		
				sessionStateHelper.addCacheMapToSession(req.getSession(), cacheKey, providers);
			} catch (Exception e) {
				LOGGER.error(e);
				throw new RuntimeException(e);
			}
		}

		return providers;
	}

    //TODO: user-id should be patient-id
	/**
	 * Returns Mental Health {@link PatientProviders} matching the filter criteria:
	 * 
	 *  <ul>
	 * <li>startDate/endDate: a range of dates (in MM/dd/yyyy format) to match
	 * against the {@link PatientProvider#getVisitDate()} field.
	 * </ul>
	 * 
	 * @param assigningAuthority
	 * @param userId
	 * @param uriInfo
	 * @return
	 */
	@GET
	@Path("/patient/{assigning-authority}/{user-id}/providers/mentalhealth")
	@Produces({ "application/xml", "application/json" })
	public PatientProviders fetchMentalHealthProviders(@Context HttpServletRequest req, @PathParam("assigning-authority") String assigningAuthority, 
			@PathParam("user-id") String userId, @QueryParam("facilitySiteCode") String facilitySiteCode, @Context UriInfo uriInfo) {

		String cacheKey = Domain.MentalHealthProvider + "=" + facilitySiteCode;

        PatientProviders providers = null;
		
		Calendar calendar = Calendar.getInstance();
		calendar.add(Calendar.MONTH, -13);
		Date thirteenMonthsBefore = calendar.getTime();
		DateFilter dateFilter = DateFilterFactory.createFilterFromDate(thirteenMonthsBefore, null);

		Object cache = sessionStateHelper.checkCache(req.getSession(), cacheKey);
		if (cache != null && cache instanceof PatientProviders) {
			LOGGER.debug("Using cached Mental Health Providers");
			providers = (PatientProviders)cache;
		}
		
		if (providers == null) {
			LOGGER.debug("Fetching Mental Health Providers from dataservice");
			try {
				PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, userId);

				ProviderDataService providerDataService = new ProviderDataService();
				providers = providerDataService.fetchMentalHealthProviders(patientIdentifier, dateFilter, facilitySiteCode);
		
				sessionStateHelper.addCacheMapToSession(req.getSession(), cacheKey, providers);
			} catch (Exception e) {
				LOGGER.error(e);
				throw new RuntimeException(e);
			}
		}
		
		return providers;
	}
}
