package com.agilex.healthcare.mobilehealthplatform.restservice;

import java.util.Map;
import java.util.Map.Entry;

import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.UriInfo;

import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;
import org.w3c.dom.Document;

import com.agilex.healthcare.mobilehealthplatform.datalayer.patient.PatientDataService;
import com.agilex.healthcare.mobilehealthplatform.datalayer.patient.PatientSearchCriteria;
import com.agilex.healthcare.mobilehealthplatform.datalayer.patient.PatientSearchPaging;
import com.agilex.healthcare.mobilehealthplatform.domain.MhpUser;
import com.agilex.healthcare.mobilehealthplatform.domain.Patient;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientSearchResults;
import com.agilex.healthcare.mobilehealthplatform.security.MhpUserFactory;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.ScopeFilter;
import com.agilex.healthcare.mobilehealthplatform.utils.QueryParameterHelper;
import com.agilex.healthcare.mobilehealthplatform.utils.uriformaters.linkbuilder.PatientLinkBuilder;
import com.agilex.healthcare.utility.NullChecker;
import com.agilex.healthcare.utility.atom.AtomUtility;
import com.sun.syndication.feed.synd.SyndFeed;

@Path("/patients")
@Component
@Scope("request")
public class PatientsResource {
	private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(PatientsResource.class);

	private static final String SEARCHCRITERIA_SSN = "ssn";
	private static final String SEARCHCRITERIA_SEARCHSTRING = "searchstring";
	private static final String SEARCHCRITERIA_FIRSTNAME = "firstname";
	private static final String SEARCHCRITERIA_LASTNAME = "lastname";
	private static final String SEARCHCRITERIA_PATIENTID = "patientid";
	private static final String PAGINGCRITERIA_MAXRECORDS = "maxrecords";
	private static final String PAGINGCRITERIA_PAGENUMBER = "page";

	/**
	 * This resource is a list of patients that can be constrained using query
	 * string parameters, such as: searchstring (to be handled by the source
	 * system), firstname, lastname, patientid, and ssn. Under development are
	 * paging options for maxrecords and page. This can be used as a patient
	 * search.
	 * 
	 * @param uriInfo
	 * @param headers
	 * @return The return value is a set of patients matching the search
	 *         criteria along with paging information.
	 */
	@GET
	@Produces({ "application/xml", "application/json" })
	public PatientSearchResults patientSearch(@Context UriInfo uriInfo, @Context HttpHeaders headers) {
		PatientSearchResults patientSearchResults = null;
		String siteCode = retrieveSiteCode();
		logger.info("received patient search request");
		patientSearchResults = performPatientSearch(uriInfo, siteCode);
		logger.info(patientSearchResults.size() + " patient(s) found");

		return patientSearchResults;
	}

	private String retrieveSiteCode() {
		MhpUser mhpUser = MhpUserFactory.createFromSecurityContext();
		String siteCode = mhpUser.getVistaLocation();
		if(siteCode == null || siteCode.equals("")){
			logger.error("User vista location is not available.");
			throw new RuntimeException("User vista location is not available.");
		}
		return siteCode;
	}

	/**
	 * This resource is a list of patients. This is a convenience resource to
	 * expose the patients resource as atom+xml
	 * 
	 * @param uriInfo
	 * @param headers
	 * @return The return value is an atom feed representing the patients
	 *         matching the search criteria
	 */
	@Deprecated
	@GET
	@Produces({ "application/atom+xml" })
	@Path("atom")
	public Document patientSearchAsAtom(@Context UriInfo uriInfo, @Context HttpHeaders headers) {
		String siteCode = retrieveSiteCode();
		PatientSearchResults patientSearchResults = performPatientSearch(uriInfo, siteCode);
		SyndFeed feed = AtomUtility.createAtomFeed(patientSearchResults, "patient-search-results");
		return AtomUtility.feedToXml(feed);
	}
	

	private PatientSearchResults performPatientSearch(UriInfo uriInfo, String siteCode) {
		ScopeFilter scopeFilter = ScopeFilter.getInstanceForOperationalScope();
		PatientSearchResults patientSearchResult = null;
		String query = null;
		if (uriInfo != null) {
			query = uriInfo.getRequestUri().getQuery();
		}

		PatientSearchCriteria patientSearchCriteria = extractPatientSearchCriteria(query, siteCode);
		PatientSearchPaging pagingInfo = extractPagingInfo(query);
		PatientDataService dataservice = new PatientDataService();

		try {
			if (NullChecker.isNotNullish(patientSearchCriteria.getSearchString())) {
				logger.debug("performing search by search string");
				patientSearchResult = dataservice.patientSearch(patientSearchCriteria.getSearchString(), siteCode, pagingInfo, scopeFilter);
			} else if (patientSearchCriteria.getPrototype() != null && NullChecker.isNotNullish(patientSearchCriteria.getPrototype().getPatientIdentifier().getUniqueId())) {
				logger.debug("performing search by patient id");
				Patient patient = dataservice.getPatient(patientSearchCriteria.getPrototype().getPatientIdentifier(), scopeFilter);

				patientSearchResult = new PatientSearchResults();
				patientSearchResult.add(patient);
			} else {
				logger.debug("performing search by prototype");
				patientSearchResult = dataservice.patientSearch(patientSearchCriteria.getPrototype(), pagingInfo, scopeFilter);
			}
		} catch (IllegalArgumentException e) {
			logger.error("IllegalArgumentException occurred, possibly due to insufficent criteria, returning empty collection ", e);
			patientSearchResult = new PatientSearchResults();
		}

		PatientLinkBuilder linkBuilder = new PatientLinkBuilder(uriInfo.getBaseUri());
		linkBuilder.fillLinks(patientSearchResult, uriInfo.getRequestUri());

		return patientSearchResult;
	}

	private PatientSearchCriteria extractPatientSearchCriteria(String query, String siteCode) {
		PatientSearchCriteria patientSearchCriteria = new PatientSearchCriteria();
		Patient prototype = new Patient();
		patientSearchCriteria.setPrototype(prototype);
		patientSearchCriteria.setVistaSiteCode(siteCode);
		
		if (NullChecker.isNotNullish(query)) {
			Map<String, String> queryParams = QueryParameterHelper.getQueryMap(query);
			for (Entry<String, String> queryParam : queryParams.entrySet()) {
				if (NullChecker.isNotNullish(queryParam.getKey()) && NullChecker.isNotNullish(queryParam.getValue())) {
					if (queryParam.getKey().toLowerCase().contentEquals(SEARCHCRITERIA_LASTNAME)) {
						prototype.setLastName(queryParam.getValue());
					}
					if (queryParam.getKey().toLowerCase().contentEquals(SEARCHCRITERIA_FIRSTNAME)) {
						prototype.setFirstName(queryParam.getValue());
					}
					if (queryParam.getKey().toLowerCase().contentEquals(SEARCHCRITERIA_SEARCHSTRING)) {
						patientSearchCriteria.setSearchString(queryParam.getValue());
					}
					if (queryParam.getKey().toLowerCase().contentEquals(SEARCHCRITERIA_SSN)) {
						prototype.setSsn(queryParam.getValue());
					}
					if (queryParam.getKey().toLowerCase().contentEquals(SEARCHCRITERIA_PATIENTID)) {
						logger.debug("got patient id value ");
						prototype.getPatientIdentifier().setUniqueId(queryParam.getValue());
					}
				}
			}
		}
		return patientSearchCriteria;
	}

	private PatientSearchPaging extractPagingInfo(String query) {
		PatientSearchPaging pagingInfo = new PatientSearchPaging();

		if (NullChecker.isNotNullish(query)) {
			Map<String, String> queryParams = QueryParameterHelper.getQueryMap(query);
			for (Entry<String, String> queryParam : queryParams.entrySet()) {
				if (NullChecker.isNotNullish(queryParam.getKey()) && NullChecker.isNotNullish(queryParam.getValue())) {
					if (queryParam.getKey().equalsIgnoreCase(PAGINGCRITERIA_MAXRECORDS)) {
						pagingInfo.setMaxRecords(Integer.parseInt(queryParam.getValue()));
					}
					if (queryParam.getKey().equalsIgnoreCase(PAGINGCRITERIA_PAGENUMBER)) {
						pagingInfo.setPageNumber(Integer.parseInt(queryParam.getValue()));
					}
				}
			}
		}
		return pagingInfo;
	}
}
