package com.agilex.healthcare.mobilehealthplatform.restservice;

import javax.ws.rs.*;
import javax.ws.rs.core.*;

import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import com.agilex.healthcare.mobilehealthplatform.datalayer.vital.*;
import com.agilex.healthcare.mobilehealthplatform.domain.*;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.ScopeFilter;
import com.agilex.healthcare.mobilehealthplatform.utils.QueryParameters;
import com.agilex.healthcare.mobilehealthplatform.utils.uriformaters.linkbuilder.VitalsLinkBuilder;
import com.agilex.healthcare.utility.NullChecker;
import com.agilex.healthcare.utility.objectmapper.VitalTableDataGenerator;

@Path("/patient/{assigning-authority}/{patient-id}/vitals")
@Component
@Scope("request")
public class PatientVitalResource {

	private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(PatientVitalResource.class);

	/**
	 * Retrieves all {@link VitalEntries} for the given Scope (operational, longitudinal)
	 * 
	 * @param assigningAuthority
	 * @param patientId
	 * @param scope
	 * @return
	 */
	@GET
	@Path("scope/{scope}")
	@Produces({ "application/xml", "application/json" })
	public VitalEntries fetchVitalEntriesByScope(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @PathParam("scope") String scope, @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);

		VitalEntries vitalEntries = fetchVitalEntries(uriInfo, patientIdentifier, ScopeFilter.getInstanceForScope(scope));

		VitalsLinkBuilder linkbuilder = new VitalsLinkBuilder(uriInfo.getBaseUri());
		linkbuilder.fillLinks(vitalEntries, uriInfo.getRequestUri());

		return vitalEntries;
	}

	/**
	 * Retrieves a specific {@link VitalEntry} using the system and id of the vital. This path endpoint represents a self-link for a {@link VitalEntry}.
	 * 
	 * @param assigningAuthority
	 * @param patientId
	 * @param vitalId
	 * @return {@link VitalEntry}
	 */
	@GET
	@Path("system/{system-id}/id/{vital-id}")
	@Produces({ "application/xml", "application/json" })
	public VitalEntry fetchVitalEntry(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @PathParam("system-id") String systemId, @PathParam("vital-id") String vitalId, @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		DataIdentifier vitalIdentifier = new DataIdentifier(systemId, vitalId);

		VitalDataService dataservice = new VitalDataService();
		VitalEntry vitalEntry = dataservice.fetchVitalEntry(patientIdentifier, vitalIdentifier, ScopeFilter.getInstanceForLongitudinalScope());

		if (vitalEntry != null) {
			VitalsLinkBuilder linkbuilder = new VitalsLinkBuilder(uriInfo.getBaseUri());
			linkbuilder.fillLinks(vitalEntry, uriInfo.getRequestUri());
		}

		return vitalEntry;
	}

	/**
	 * Retrieves all {@link VitalEntries} for the given scope (operational, longitudinal) and transforms it into a graph-able format ({@link GraphData}).
	 * 
	 * @param assigningAuthority
	 * @param patientId
	 * @param scope
	 * @return
	 */
	@GET
	@Path("scope/{scope}/graphdata")
	@Produces({ "application/xml", "application/json" })
	public GraphData getVitalEntriesGraphData(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @PathParam("scope") String scope, @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);

		VitalEntries vitals = fetchVitalEntries(uriInfo, patientIdentifier, ScopeFilter.getInstanceForScope(scope));
		ProviderVitalGraphGenerator graphGenerator = new ProviderVitalGraphGenerator();

		VitalEntryFilter filter = VitalsFilterFactory.createFilter(uriInfo.getRequestUri());

		QueryParameters p = new QueryParameters(uriInfo.getRequestUri().getQuery());
		String graphType = p.getValue("graphType");
		if (NullChecker.isNullish(graphType))
			graphType = filter.getSection();

		GraphData graphData = graphGenerator.createGraphDataBySection(vitals, graphType);

		VitalsLinkBuilder linkbuilder = new VitalsLinkBuilder(uriInfo.getBaseUri());
		linkbuilder.fillLinks(graphData, uriInfo.getRequestUri());

		return graphData;
	}

	/**
	 * Retrieves all {@link VitalEntries} for the given scope (operational, longitudinal) and transforms it into a format suitable for displaying a table ({@link VitalTableData}).
	 * 
	 * @param assigningAuthority
	 * @param patientId
	 * @param scope
	 * @return
	 */
	@GET
	@Path("scope/{scope}/tabledata")
	@Produces({ "application/xml", "application/json" })
	public VitalTableData getVitalEntriesAsTableData(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @PathParam("scope") String scope, @Context UriInfo uriInfo, @Context HttpHeaders headers) {

		PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);

		VitalEntries vitalEntries = fetchVitalEntries(uriInfo, patientIdentifier, ScopeFilter.getInstanceForScope(scope));

		VitalTableDataGenerator tableGenerator = new VitalTableDataGenerator();
		VitalTableData vitalTableData = tableGenerator.generateVitalTableData(vitalEntries);
		vitalTableData.sortDescending("entryTime");

		VitalsLinkBuilder linkbuilder = new VitalsLinkBuilder(uriInfo.getBaseUri());
		linkbuilder.fillLinks(vitalTableData, uriInfo.getRequestUri());

		return vitalTableData;
	}

	private VitalEntries fetchVitalEntries(UriInfo uriInfo, PatientIdentifier patientIdentifier, ScopeFilter scopeFilter) {
		logger.debug("Retrieving vitals for " + patientIdentifier);

		VitalEntryFilter filter = VitalsFilterFactory.createFilter(uriInfo.getRequestUri());
		VitalDataService dataservice = new VitalDataService();
		VitalEntries vitalEntries = dataservice.fetchVitalEntries(patientIdentifier, filter, scopeFilter);

		VitalsLinkBuilder linkbuilder = new VitalsLinkBuilder(uriInfo.getBaseUri());
		linkbuilder.fillLinks(vitalEntries, uriInfo.getRequestUri());

		return vitalEntries;
	}

}
