package com.agilex.healthcare.mobilehealthplatform.restservice;

import javax.ws.rs.Consumes;
import javax.ws.rs.DELETE;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.UriInfo;

import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import com.agilex.healthcare.mobilehealthplatform.domain.DataIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.ToolTrackingResult;
import com.agilex.healthcare.mobilehealthplatform.domain.ToolTrackingResults;
import com.agilex.healthcare.mobilehealthplatform.tooltrackingresult.ToolTrackingResultDataService;
import com.agilex.healthcare.mobilehealthplatform.utils.uriformaters.linkbuilder.ToolTrackingResultLinkBuilder;

@Path("/patient/{assigning-authority}/{patient-id}/toolTracking")
@Component
@Scope("request")
public class PatientToolTrackingResultResource {

    /**
     * Fetches the specific {@link ToolTrackingResult} specified by it's self link.
     *
     * @param assigningAuthority
     * @param patientId
     * @param toolId
     */
    @GET
    @Path("allResults")
    @Produces({ "application/xml", "application/json" })
    public ToolTrackingResults fetchAllToolResult(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @Context UriInfo uriInfo, @Context HttpHeaders headers) {
        PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);

        ToolTrackingResultDataService dataService = new ToolTrackingResultDataService();
        ToolTrackingResults results = dataService.fetchResults(patientIdentifier);

        return results;
    }

	/**
	 * This method does not return any {@link ToolTrackingResults}. Instead, it provides a mechanism to discover links to the most recent results for individual tools.
	 * 
	 * @param assigningAuthority
	 * @param patientId
	 * 
	 */
	@GET
	@Produces({ "application/xml", "application/json" })
	public ToolTrackingResults fetchToolTypes(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		ToolTrackingResults results = new ToolTrackingResults();
		
//		ToolTrackingResultDataService dataService = new ToolTrackingResultDataService();
//		dataService.fetchResults(patientIdentifier);
		
		ToolTrackingResultLinkBuilder linkBuilder = new ToolTrackingResultLinkBuilder(uriInfo.getBaseUri());
		linkBuilder.fillToolRelatedLinks(results, patientIdentifier);
		
		return results;
	}
	
	/**
	 * Fetches the specific {@link ToolTrackingResult} specified by it's self link.
	 * 
	 * @param assigningAuthority
	 * @param patientId
	 * @param toolId
	 */
	@GET
	@Path("system/{system-id}/tool/id/{id}")
	@Produces({ "application/xml", "application/json" })
	public ToolTrackingResult fetchToolResult(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @PathParam("id") String toolId, @PathParam("system-id") String systemId, @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		DataIdentifier dataIdentifier = new DataIdentifier(systemId, toolId);
		
		ToolTrackingResultDataService dataService = new ToolTrackingResultDataService();
		ToolTrackingResult result = dataService.fetchResult(patientIdentifier, dataIdentifier);
		
		ToolTrackingResultLinkBuilder linkBuilder = new ToolTrackingResultLinkBuilder(uriInfo.getBaseUri());
		linkBuilder.fillLinks(result, uriInfo.getRequestUri());
		
		return result;
	}
	
	/**
	 * Fetches the most recent {@link ToolTrackingResult} available to the user.
	 * 
	 * @param assigningAuthority
	 * @param patientId
	 * @param tool
	 */
	@GET
	@Path("recent/tool/{tool}")
	@Produces({ "application/xml", "application/json" })
	public ToolTrackingResult fetchMostRecentToolResult(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @PathParam("tool") String tool, @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		
		ToolTrackingResultDataService dataService = new ToolTrackingResultDataService();
		ToolTrackingResult result = dataService.fetchMostRecentResult(patientIdentifier, tool);
		
		ToolTrackingResultLinkBuilder linkBuilder = new ToolTrackingResultLinkBuilder(uriInfo.getBaseUri());
		linkBuilder.fillLinks(result, uriInfo.getRequestUri());
		
		return result;
	}

	/**
	 * Provides the mechanism to persist a {@link ToolTrackingResult} to a backing storage. The saved object will be returned along with a self link.
	 *   
	 * @param result
	 * @param assigningAuthority
	 * @param patientId
	 */
	@POST
	@Produces({ "application/xml", "application/json" })
	@Consumes({ "application/xml", "application/json" })
	public ToolTrackingResult save(ToolTrackingResult result, @PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		result.setPatientIdentifier(patientIdentifier);
		
		ToolTrackingResultDataService dataService = new ToolTrackingResultDataService();
		ToolTrackingResult updatedResult = dataService.save(result);
		
		ToolTrackingResultLinkBuilder linkBuilder = new ToolTrackingResultLinkBuilder(uriInfo.getBaseUri());
		linkBuilder.fillLinks(updatedResult, uriInfo.getRequestUri());
		
		return updatedResult;
	}
	
	
	/**
	 * Persists changes to an existing {@link ToolTrackingResult} with the backing storage. The updated object will be returned along with the self link.
	 *  
	 * @param result
	 * @param assigningAuthority
	 * @param patientId
	 */
	@PUT
	@Path("system/{system-id}/tool/id/{id}")
	@Produces({ "application/xml", "application/json" })
	@Consumes({ "application/xml", "application/json" })
	public ToolTrackingResult update(ToolTrackingResult result, @PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @PathParam("id") String id, @PathParam("system-id") String systemId,@Context UriInfo uriInfo, @Context HttpHeaders headers) {
		PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		result.setPatientIdentifier(patientIdentifier);
		
		ToolTrackingResultDataService dataService = new ToolTrackingResultDataService();
		ToolTrackingResult updatedResult = dataService.update(result);
		 
		ToolTrackingResultLinkBuilder linkBuilder = new ToolTrackingResultLinkBuilder(uriInfo.getBaseUri());
		linkBuilder.fillLinks(updatedResult, uriInfo.getRequestUri());
		
		return updatedResult;
	}
	
	/**
	 * Deletes an existing {@link ToolTrackingResult}.
	 * 
	 * @param result
	 * @param assigningAuthority
	 * @param patientId
	 */
	@DELETE
	@Path("system/{system-id}/tool/id/{id}")
	@Consumes({ "application/xml", "application/json" })
	public void delete(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @PathParam("id") String id, @PathParam("system-id") String systemId, @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		DataIdentifier dataIdentifier = new DataIdentifier(systemId, id);
		PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		
		ToolTrackingResult result = new ToolTrackingResult();
		result.setDataIdentifier(dataIdentifier);
		result.setPatientIdentifier(patientIdentifier);

		ToolTrackingResultDataService dataService = new ToolTrackingResultDataService();
		dataService.delete(result);
	}
	
	
}
