package com.agilex.healthcare.mobilehealthplatform.restservice;

import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.UriInfo;

import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import com.agilex.healthcare.mobilehealthplatform.datalayer.surgery.SurgeryDataService;
import com.agilex.healthcare.mobilehealthplatform.domain.DataIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.Surgeries;
import com.agilex.healthcare.mobilehealthplatform.domain.Surgery;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.*;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.ScopeFilter;
import com.agilex.healthcare.mobilehealthplatform.utils.uriformaters.linkbuilder.SurgeryLinkBuilder;

@Path("/patient/{assigning-authority}/{patient-id}/surgeries")
@Component
@Scope("request")
public class PatientSurgeryResource {

	/**
	 * Deprecated.
	 */
	@GET
	@Produces({ "application/xml", "application/json" })
	public Surgeries fetchSurgeries(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @PathParam("scope") String scope, @Context UriInfo u, @Context HttpHeaders headers) {
		final PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		return fetchSurgeries(patientIdentifier, ScopeFilter.getInstanceForLongitudinalScope(), u);
	}

	/**
	 * Retrieves Surgeries according to the defined scope {longitudinal, operational}. 
	 * 
	 * This method accepts date filter query parameters
	 *  
	 * @param assigningAuthority
	 * @param patientId
	 * @param scope
	 * @return {@link Surgeries}
	 */
	@GET
	@Produces({ "application/xml", "application/json" })
	@Path("scope/{scope}")
	public Surgeries fetchSurgeriesByScope(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @PathParam("scope") String scope, @Context UriInfo u, @Context HttpHeaders headers) {
		final PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		return fetchSurgeries(patientIdentifier, ScopeFilter.getInstanceForScope(scope), u);
	}

	/**
	 * Retrieves Surgeries by system identifier. 
	 * 
	 * This method access date filter query parameters
	 * 
	 * @param assigningAuthority
	 * @param patientId
	 * @param systemIdentifier
	 * @return {@link Surgeries}
	 */
	@GET
	@Produces({ "application/xml", "application/json" })
	@Path("system/{system-id}")
	public Surgeries fetchSurgeriesBySystem(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @PathParam("system-id") String systemIdentifier, @Context UriInfo u, @Context HttpHeaders headers) {
		final PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		return fetchSurgeries(patientIdentifier, ScopeFilter.getInstanceForSystem(systemIdentifier), u);
	}

	/**
	 * Retrieves a single surgery value. This method represents the self link of a {@link Surgery} object.
	 * 
	 * @param assigningAuthority
	 * @param patientId
	 * @param systemId
	 * @return {@link Surgery}
	 */
	@GET
	@Produces({ "application/xml", "application/json" })
	@Path("system/{system-id}/id/{id}")
	public Surgery fetchSurgery(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @PathParam("system-id") String systemId, @PathParam("id") String id, @Context UriInfo u,
			@Context HttpHeaders headers) {
		final PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		final DataIdentifier dataIdentifier = new DataIdentifier(systemId, id);
		SurgeryDataService dataService = new SurgeryDataService();
		Surgery Surgery = dataService.fetchSurgery(patientIdentifier, dataIdentifier);

		SurgeryLinkBuilder linkbuilder = new SurgeryLinkBuilder(u.getBaseUri());
		linkbuilder.fillLinks(Surgery, u.getRequestUri());

		if (Surgery == null)
			throw ExceptionFactory.generateNoDataFoundException();

		return Surgery;
	}

	private Surgeries fetchSurgeries(final PatientIdentifier patientIdentifier, ScopeFilter scopeFilter, UriInfo uriInfo) {
		SurgeryDataService dataService = new SurgeryDataService();
		DateFilter dateFilter = DateFilterFactory.createFilterFromUri(uriInfo);
		Surgeries Surgeries = dataService.getPatientSurgeries(patientIdentifier, dateFilter, scopeFilter);

		SurgeryLinkBuilder linkbuilder = new SurgeryLinkBuilder(uriInfo.getBaseUri());
		linkbuilder.fillLinks(Surgeries, uriInfo.getRequestUri());

		return Surgeries;
	}
}
