package com.agilex.healthcare.mobilehealthplatform.restservice;

import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.UriInfo;

import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import com.agilex.healthcare.mobilehealthplatform.datalayer.patient.PatientDataService;
import com.agilex.healthcare.mobilehealthplatform.datalayer.patient.PatientInternalDataService;
import com.agilex.healthcare.mobilehealthplatform.domain.MhpUser;
import com.agilex.healthcare.mobilehealthplatform.domain.Patient;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientDemographics;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifiers;
import com.agilex.healthcare.mobilehealthplatform.security.MhpUserFactory;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.ScopeFilter;
import com.agilex.healthcare.mobilehealthplatform.utils.uriformaters.linkbuilder.DemographicsLinkBuilder;
import com.agilex.healthcare.mobilehealthplatform.utils.uriformaters.linkbuilder.PatientLinkBuilder;
import com.agilex.healthcare.utility.ModeHelper;
import com.agilex.healthcare.utility.NullChecker;
import com.sun.jersey.api.NotFoundException;

@Path("/patient")
@Component
@Scope("request")
public class PatientResource extends AbstractUserResource {
	private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(PatientResource.class);

	/**
	 * This resource represents a single patient.
	 * 
	 * @param assigningAuthority
	 *            the qualifier for the patient identifier
	 * @param patientId
	 *            the patient identifier
	 * @param uriInfo
	 * @param headers
	 * @return Returns a single patient including links to other patient
	 *         resources.
	 */

	@GET
	@Path("{assigning-authority}/{patient-id}")
	@Produces({ "application/xml", "application/json" })
	public Patient getPatient(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @Context UriInfo uriInfo, @Context HttpHeaders headers) {

		final PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		Patient patient = fetchPatientBasedOnSystemMode(patientIdentifier);
		buildPatientAtomLinks(uriInfo, patient);

		return patient;
	}

	private void buildPatientAtomLinks(UriInfo uriInfo, Patient patient) {
		if (uriInfo != null) {
			PatientLinkBuilder linkBuilder = new PatientLinkBuilder(uriInfo.getBaseUri());
			linkBuilder.fillLinks(patient, uriInfo.getRequestUri());
		}
	}

	private Patient fetchPatientBasedOnSystemMode(final PatientIdentifier patientIdentifier) {
		if(ModeHelper.isVeteranMode()){
			return fetchPatientFromMhpUserObject();
		}else{
			return fetchPatientFromVista(patientIdentifier);
		}
	}

	private Patient fetchPatientFromMhpUserObject() {
		MhpUser currentUser = MhpUserFactory.createFromSecurityContext();
		return currentUser.getPatient();
	}

	private Patient fetchPatientFromVista(final PatientIdentifier patientIdentifier) {
		PatientDataService dataservice = new PatientDataService();
		Patient patient = dataservice.getPatient(patientIdentifier, ScopeFilter.getInstanceForLongitudinalScope());
		return patient;
	}
	
	@GET
	@Path("{assigning-authority}/{patient-id}/identifiers")
	@Produces({ "application/xml", "application/json" })
	public PatientIdentifiers fetchPatientIdentifiers(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		final PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		
		PatientDataService dataService = new PatientDataService();
		PatientIdentifiers patientIdentifiers = dataService.fetchPatientIdentifiers(patientIdentifier);
		
		return patientIdentifiers;
	}

	/**
	 * This resource represents a single patient's demographics.
	 * 
	 * @param assigningAuthority
	 *            the qualifier for the patient identifier
	 * @param patientId
	 *            the patient identifier
	 * @param uriInfo
	 * @param headers
	 * @return Returns a demographics object for the patient.
	 */
	@GET
	@Path("{assigning-authority}/{patient-id}/demographics")
	@Produces({ "application/xml", "application/json" })
	public PatientDemographics getDemographics(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		logger.debug("received request for demographics for patient");

		final PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		PatientDemographics demographics = null;

		if (NullChecker.isNullish(patientId)) {
			logger.debug("no patient id, returning null");
			return null;
		}

		logger.debug("checking to see patient exists");
		Patient patient = fetchPatientBasedOnSystemMode(patientIdentifier);

		if (patient == null) {
			logger.warn("patient not found, throwing 404 exception");
			throw new NotFoundException();
		}

		logger.debug("retrieving demographics");
		PatientDataService dataservice = new PatientDataService();
		demographics = dataservice.getPatientDemographics(patientIdentifier, ScopeFilter.getInstanceForLongitudinalScope());

		if (demographics == null) {
			demographics = new PatientDemographics();
			demographics.setPatientIdentifier(patientIdentifier);
		}

		if (uriInfo != null) {
			DemographicsLinkBuilder linkBuilder = new DemographicsLinkBuilder(uriInfo.getBaseUri());
			linkBuilder.fillLinks(demographics, uriInfo.getRequestUri());
		}

		logger.debug("returning demographics: " + demographics.toString());
		return demographics;
	}
	
	@GET
	@Path("{assigning-authority}/{patient-id}/demographics/scope/{scope}")
	@Produces({ "application/xml", "application/json" })
	public PatientDemographics getDemographicsByScope(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @PathParam("scope") String scope,  @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		
		PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		ScopeFilter scopeFilter = ScopeFilter.getInstanceForScope(scope);
		
		PatientDataService dataService = new PatientDataService();
		PatientDemographics demographics = dataService.getPatientDemographics(patientIdentifier, scopeFilter);
		
		DemographicsLinkBuilder linkBuilder = new DemographicsLinkBuilder(uriInfo.getBaseUri());
		linkBuilder.fillLinks(demographics, uriInfo.getRequestUri());
		
		return demographics;
	}

	/**
	 * This resource represents the currently logged on patient.
	 * 
	 * @param uriInfo
	 * @param headers
	 * @return Returns the patient resource associated with the currently logged
	 *         on patient user.
	 */
	@GET
	@Path("me")
	@Produces({ "application/xml", "application/json" })
	@Deprecated
	public Patient getCurrentPatient(@Context UriInfo uriInfo, @Context HttpHeaders headers) {

		logger.debug("Retrieving current patient");

		PatientDataService dataservice = new PatientDataService();
		Patient patient = dataservice.getPatient(getPatientIdentifier(), ScopeFilter.getInstanceForLongitudinalScope());

		buildPatientAtomLinks(uriInfo, patient);

		return patient;
	}
	

	/**
	 * This resource returns the patient from the internal patient data store. It uses the EDIPI id and it retrieves data from
	 * health adapter repository or appointments database
	 * 
	 * @param patientId
	 * @param uriInfo
	 * @param headers
	 * @return
	 */
	@GET
	@Path("/id/{patient-id}")
	@Produces({ "application/xml", "application/json" })
	public Patient getPatient(@PathParam("patient-id") String patientId, @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		
		PatientInternalDataService dataservice = new PatientInternalDataService();
		Patient patient = dataservice.fetchPatientById(patientId);

		return patient;
	}
}
