package com.agilex.healthcare.mobilehealthplatform.restservice;

import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.UriInfo;

import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import com.agilex.healthcare.mobilehealthplatform.datalayer.problem.ProblemDataService;
import com.agilex.healthcare.mobilehealthplatform.domain.DataIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.Problem;
import com.agilex.healthcare.mobilehealthplatform.domain.Problems;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.*;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.ScopeFilter;
import com.agilex.healthcare.mobilehealthplatform.utils.uriformaters.linkbuilder.ProblemLinkBuilder;

@Path("/patient/{assigning-authority}/{patient-id}/problems")
@Component
@Scope("request")
public class PatientProblemsResource {

	/**
	 * Do not use this method.
	 * 
	 * @deprecated
	 * @param assigningAuthority
	 * @param patientId
	 * @return
	 */
	@GET
	@Produces({ "application/xml", "application/json" })
	@Deprecated
	public Problems fetchProblems(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @Context UriInfo u, @Context HttpHeaders headers) {
		final PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		return fetchProblems(patientIdentifier, ScopeFilter.getInstanceForLongitudinalScope(), u);
	}

	/**
	 * Retrieves all problems for patients based upon the scope provided.
	 * 
	 * This method also takes {startDate, endDate, entryDate} query parameters
	 * 
	 * @param assigningAuthority Assigning Authority associated with the patient's login
	 * @param patientId Identifier of the Patient
	 * @param scope {operational, longitudinal}
	 * @return
	 */
	@GET
	@Produces({ "application/xml", "application/json" })
	@Path("scope/{scope}")
	public Problems fetchProblemsByScope(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @PathParam("scope") String scope, @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		final PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		return fetchProblems(patientIdentifier, ScopeFilter.getInstanceForScope(scope), uriInfo);
	}

	/**
	 * Retrieves all problems for patients based upon the system provided
	 * 
	 * @param assigningAuthority Assigning Authority associated with the patient's login
	 * @param patientId Identifier of the Patient
	 * @param systemId System to request {@link Problems} from - {vista, cdw}
	 * @return
	 */
	@GET
	@Produces({ "application/xml", "application/json" })
	@Path("system/{system-id}")
	public Problems fetchProblemsBySystem(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @PathParam("system-id") String systemId, @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		final PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		return fetchProblems(patientIdentifier, ScopeFilter.getInstanceForSystem(systemId), uriInfo);
	}

	private Problems fetchProblems(final PatientIdentifier patientIdentifier, ScopeFilter scopeFilter, UriInfo uriInfo) {
		ProblemDataService dataService = new ProblemDataService();
		
		DateFilter dateFilter = DateFilterFactory.createFilterFromUri(uriInfo.getRequestUri());
		Problems problems = dataService.fetchProblems(patientIdentifier, dateFilter, scopeFilter);

		ProblemLinkBuilder linkbuilder = new ProblemLinkBuilder(uriInfo.getBaseUri());
		linkbuilder.fillLinks(problems, uriInfo.getRequestUri());

		return problems;
	}

	/**
	 * Retrieves a single problem given a system and it's specific id. This link will match the self link of a problem object.
	 * 
	 * @param assigningAuthority Assigning Authority associated with the patient's login
	 * @param patientId Identifier of the Patient
	 * @param systemId System that problems originate from {vista, cdw}
	 * @param id Unique Identifier of the Problem object
	 * @return
	 */
	@GET
	@Produces({ "application/xml", "application/json" })
	@Path("system/{system-id}/id/{id}")
	public Problem fetchProblem(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @PathParam("system-id") String systemId, @PathParam("id") String id, @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		final PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		final DataIdentifier dataIdentifier = new DataIdentifier(systemId, id);

		ProblemDataService dataService = new ProblemDataService();
		Problem Problem = dataService.fetchProblem(patientIdentifier, dataIdentifier);

		ProblemLinkBuilder linkbuilder = new ProblemLinkBuilder(uriInfo.getBaseUri());
		linkbuilder.fillLinks(Problem, uriInfo.getRequestUri());

		if (Problem == null)
			throw ExceptionFactory.generateNoDataFoundException();

		return Problem;
	}

}
