package com.agilex.healthcare.mobilehealthplatform.restservice;

import java.io.IOException;
import java.io.OutputStream;
import java.net.URISyntaxException;
import java.util.Date;

import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.Consumes;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.WebApplicationException;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.Status;
import javax.ws.rs.core.StreamingOutput;
import javax.ws.rs.core.UriInfo;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.MessageSource;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import com.agilex.healthcare.mobilehealthplatform.datalayer.mygoals.MygoalsDataService;
import com.agilex.healthcare.mobilehealthplatform.domain.DataIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.Mygoals;
import com.agilex.healthcare.mobilehealthplatform.domain.MygoalsList;
import com.agilex.healthcare.mobilehealthplatform.domain.MygoalsURI;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilter;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilterFactory;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.Domain;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.ScopeFilter;
import com.agilex.healthcare.mobilehealthplatform.utils.uriformaters.linkbuilder.PatientMygoalsLinkBuilder;
import com.agilex.healthcare.mygoals.dataservice.PdfDataService;
import com.agilex.healthcare.pdf.PdfGenerationContext;

@Component
@Scope("request")
@Path("patient/{assigning-authority}/{patient-id}/mygoals")
public class PatientMygoalsResource {
	private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(PatientMygoalsResource.class);
	private final MygoalsDataService dataService = new MygoalsDataService();
	private static final ScopeFilter scope = ScopeFilter.getInstanceForLongitudinalScope();

	@Autowired
	MessageSource messageSource;
	
	/**
	 * Retrieves the latest {@link Mygoals}
	 * 
	 * @param assigningAuthority
	 * @param patientId
	 * @param inventoryType
	 */
	@GET
	@Path("/latest/{inventory-type}")
	@Produces({ "application/xml", "application/json" })
	public MygoalsList fetchLatestMygoalsByType(@PathParam("patient-id") String patientId, @PathParam("assigning-authority") String assigningAuthority,
			@PathParam("inventory-type") String inventoryType, @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		
		logger.debug("PatientMygoalsResource.fetchLatestMygoalsByType() called");
		PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		MygoalsList mygoalsList = dataService.fetchLatestMygoalsByType(inventoryType, patientIdentifier, scope);
		if (mygoalsList != null && !mygoalsList.isEmpty()) {
			PatientMygoalsLinkBuilder linkBuilder = new PatientMygoalsLinkBuilder(uriInfo.getBaseUri());
			linkBuilder.fillFetchLatestLinks(mygoalsList.get(0), uriInfo.getRequestUri());
		}
		return mygoalsList;
	}

	/**
	 * Retrieves the latest {@link MygoalsList}
	 * 
	 * @param assigningAuthority
	 * @param patientId
	 */
	@GET
	@Path("/latest")
	@Produces({ "application/xml", "application/json" })
	public MygoalsList fetchLatestMygoals(@PathParam("patient-id") String patientId, @PathParam("assigning-authority") String assigningAuthority,
		    @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		
		logger.debug("PatientMygoalsResource.fetchLatestMygoals() called");
		PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		MygoalsList mygoalsList = dataService.fetchLatestMygoals(patientIdentifier, scope);
		if (mygoalsList!= null && !mygoalsList.isEmpty()) {
			PatientMygoalsLinkBuilder linkBuilder = new PatientMygoalsLinkBuilder(uriInfo.getBaseUri());
			linkBuilder.fillLatestLinks(mygoalsList, uriInfo.getRequestUri());
		}
		return mygoalsList;
	}
	
	/**
	 * Saves the {@link Mygoals}
	 * 
	 * @param assigningAuthority
	 * @param patientId
	 * @param mygoals
	 */
	@POST
	@Path("/latest")
	@Produces({ "application/xml", "application/json" })
	@Consumes({ "application/xml", "application/json" })
	public Mygoals saveMygoals(Mygoals mygoals, @PathParam("patient-id") String patientId, @PathParam("assigning-authority") String assigningAuthority,
			 @Context UriInfo uriInfo, @Context HttpHeaders headers) {

		final PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		mygoals.setPatientIdentifier(patientIdentifier);
		mygoals.setDateUpdated(new Date());
		Mygoals resultMygoals = dataService.saveMygoals(mygoals, scope);
		if (resultMygoals != null) {
			PatientMygoalsLinkBuilder linkbuilder = new PatientMygoalsLinkBuilder(uriInfo.getBaseUri());
			linkbuilder.fillSaveLinks(resultMygoals, uriInfo.getRequestUri());

//			logger.debug(String.format("saved inventory response [id=%s][patient=%s][inventory type=%s]", resultMygoals.getId(),
//					resultMygoals.getPatientId(), resultMygoals.getInventoryType()));
		}
		return resultMygoals;
	}
	
	/**
	 * Retrieves the {@link Mygoals}
	 * 
	 * @param assigningAuthority
	 * @param patientId
	 * @param responseId
	 */
	@GET
	@Path("/id/{response-id}")
	@Produces({ "application/xml", "application/json" })
	public Mygoals fetchInventoryResponse(@PathParam("patient-id") String patientId, @PathParam("assigning-authority") String assigningAuthority,
			@PathParam("response-id") String responseId, @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		
		PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		Mygoals mygoals = dataService.fetchMygoalsById(patientIdentifier, responseId);
		if (mygoals != null) {
			PatientMygoalsLinkBuilder linkBuilder = new PatientMygoalsLinkBuilder(uriInfo.getBaseUri());
			linkBuilder.fillLinks(mygoals, uriInfo.getRequestUri());
//			logger.debug(String.format("inventory response [id=%s][patient=%s][inventory type=%s]", mygoals.getUniqueId(),
//					mygoals.getPatientId(), mygoals.getInventoryType()));
		}
		return mygoals;
	}
	
	/**
	 * Retrieves the {@link MygoalsList}
	 * 
	 * @param assigningAuthority
	 * @param patientId
	 */
	@GET
	@Path("/all")
	@Produces({ "application/xml", "application/json" })
	public MygoalsList fetchMygoals(@PathParam("patient-id") String patientId, @PathParam("assigning-authority") String assigningAuthority,
			@Context UriInfo uriInfo, @Context HttpHeaders headers) {
		
		logger.debug("PatientMygoalsResource.fetchMygoals() called");
		PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		MygoalsList mygoalsList = dataService.fetchMygoalsByPatient(patientIdentifier, scope);
		if (mygoalsList != null && !mygoalsList.isEmpty()) {
			PatientMygoalsLinkBuilder linkBuilder = new PatientMygoalsLinkBuilder(uriInfo.getBaseUri());
			linkBuilder.fillLinks(mygoalsList, uriInfo.getRequestUri());
		}
		return mygoalsList;
	}

	/**
	 * Retrieves the report of {@link MygoalsList}
	 * 
	 * @param assigningAuthority
	 * @param patientId
	 */
	@GET
	@Path("/latest1")
	@Produces({ "application/pdf" })
	public Response fetchLatestMygoals(@PathParam("patient-id") String patientId, @PathParam("assigning-authority") String assigningAuthority,
			@Context UriInfo uriInfo, @Context HttpHeaders headers, @Context HttpServletRequest request) throws URISyntaxException {
		
		logger.debug("PatientMygoalsResource.fetchLatestInventoryResponses() called");
		DateFilter dateFilter = DateFilterFactory.createFilterFromUri(uriInfo.getRequestUri());
		PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		final PdfGenerationContext context = buildContext(request, patientIdentifier, dateFilter);
		if (context.getInformation(Domain.mygoals + "," + patientId) == null) {
			MygoalsList mygoalsList = dataService.fetchLatestMygoals(patientIdentifier, scope);
			context.setInformation(Domain.mygoals + "," + patientId, mygoalsList);	
		}
		StreamingOutput stream = generatePdfToStreamingOutput(context);
	
		return Response.ok(stream).build();	
	}	
	
	/**
	 * Retrieves the report of {@link MygoalsList}
	 * 
	 * @param assigningAuthority
	 * @param patientId
	 */
	@GET
	@Path("/report")
	@Produces({ "application/pdf" })
	public Response getMygoalsReport(@PathParam("patient-id") String patientId, @PathParam("assigning-authority") String assigningAuthority,
			@Context UriInfo uriInfo, @Context HttpHeaders headers, @Context HttpServletRequest request) throws URISyntaxException {
		
		logger.debug("PatientMygoalsResource.getMygoalsReport() called");
		return fetchLatestMygoals(patientId, assigningAuthority, uriInfo, headers, request);
	}	
	
	/**
	 * Retrieves the {@link MygoalsURI}
	 * 
	 * @param assigningAuthority
	 * @param patientId
	 */
	@GET
	@Produces({ "application/xml", "application/json" })
	public MygoalsURI fetchMyGoalsInventory(@PathParam("patient-id") String patientId, @PathParam("assigning-authority") String assigningAuthority,
			@Context UriInfo uriInfo, @Context HttpHeaders headers) {
		
		logger.debug("PatientMygoalsResource.fetchInventoryResponses() called");
		PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		MygoalsURI mygoalsURI = new MygoalsURI();
		mygoalsURI.setPatientIdentifier(patientIdentifier);
		DataIdentifier invDataIdentifier = new DataIdentifier(patientIdentifier.getAssigningAuthority(), patientIdentifier.getUniqueId());
		mygoalsURI.setDataIdentifier(invDataIdentifier);
		MygoalsURILinkBuilder linkBuilder = new MygoalsURILinkBuilder(uriInfo.getBaseUri());
		linkBuilder.fillLinks(mygoalsURI, uriInfo.getRequestUri());

		return mygoalsURI;
	}

	private PdfGenerationContext buildContext(HttpServletRequest request,
			PatientIdentifier patientIdentifier, DateFilter dateFilter) {
		
		PdfDataService dataService = new PdfDataService();
		return dataService.buildContext(request, patientIdentifier, dateFilter, messageSource);
	}
	
	private StreamingOutput generatePdfToStreamingOutput(final PdfGenerationContext context) {
		StreamingOutput stream = new StreamingOutput() {
			public void write(OutputStream output) throws IOException, WebApplicationException {
				try {
					PdfDataService dataService = new PdfDataService();
					dataService.generateReport(output, context);
				} catch (Exception e) {
					logger.error("Error generating PDF Report", e);
					throw new WebApplicationException(Status.INTERNAL_SERVER_ERROR);
				}
			}
		};
		return stream;
	}
	
}
