package com.agilex.healthcare.mobilehealthplatform.restservice;

import javax.ws.rs.GET;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.UriInfo;

import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import com.agilex.healthcare.mobilehealthplatform.datalayer.medication.rxrefill.MedicationRefillDataService;
import com.agilex.healthcare.mobilehealthplatform.domain.Medication;
import com.agilex.healthcare.mobilehealthplatform.domain.Medications;
import com.agilex.healthcare.mobilehealthplatform.domain.Patient;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.utils.uriformaters.linkbuilder.MedicationRefillLinkBuilder;

@Path("/patient/{assigning-authority}/{patient-id}/medication-refills")
@Component
@Scope("request")
public class PatientMedicationRefillResource {

	/**
	 * A resource that returns all active {@link Medications} that include refill action information to indicate whether a
	 * {@link Medication} can be refilled or no action can be taken.
	 *
	 * Each {@link Medication} will contain self links
	 * 
	 * @param assigningAuthority
	 * 	The organization that the patientId is associated with
	 * @param patientId
	 * 	The patientId associated with the Assigning Authority
	 * @return
	 * 	Returns all active {@link Medications} associated with the specified {@link Patient}
	 */
	@GET
	@Produces({ "application/xml", "application/json" })
	public Medications fetchMedications(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @Context UriInfo uriInfo, @Context HttpHeaders headers) {

		PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		
		MedicationRefillDataService dataservice = new MedicationRefillDataService();
		Medications medications = dataservice.fetchMedications(patientIdentifier);
		
		if (uriInfo != null) {
			MedicationRefillLinkBuilder linkBuilder = new MedicationRefillLinkBuilder(uriInfo.getBaseUri());
			linkBuilder.fillLinks(medications, uriInfo.getRequestUri());
			// MedicationUriHelper.updateMedicationsLinks(medications, patientIdentifier, uriInfo);
		}
		
                return medications;

        }
	
	/**
	 * A resource that allows a client to request a refill by passing in the {@link Medication} associated with a {@link Patient}. Each {@link Medication}
	 * eligible for refill contain a self link to this resource method.
	 * 
	 * The {@link Medication} submitted for a refill will be verified before it will be processed. 
	 * 
	 * The returned {@link Medication} will contain self links
	 * 
	 * @param medication
	 * 	The medication to be refilled
	 * @param assigningAuthority
	 * 	The organization that the patientId is associated with
	 * @param patientId
	 * 	The patientId associated with the Assigning Authority
	 * @return
	 * 	A {@link Medication} that contains 
	 */
	@PUT
	@Path("refill")
	public Medication requestRefill(Medication medication, @PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		medication.setPatientIdentifier(patientIdentifier);
		
		MedicationRefillDataService dataservice = new MedicationRefillDataService();
		Medication modifiedMedication = dataservice.refillMedication(medication);
		
		if (uriInfo != null) {
			MedicationRefillLinkBuilder linkBuilder = new MedicationRefillLinkBuilder(uriInfo.getBaseUri());
			linkBuilder.fillLinks(medication, uriInfo.getRequestUri());
		}
		
		return modifiedMedication;
	}
}
