package com.agilex.healthcare.mobilehealthplatform.restservice;

import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.UriInfo;

import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import com.agilex.healthcare.mobilehealthplatform.datalayer.lab.ChemistryLabDataService;
import com.agilex.healthcare.mobilehealthplatform.domain.LabResults;
import com.agilex.healthcare.mobilehealthplatform.domain.LabTestGroups;
import com.agilex.healthcare.mobilehealthplatform.domain.LabTests;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilter;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilterFactory;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.ScopeFilter;
import com.agilex.healthcare.mobilehealthplatform.utils.uriformaters.linkbuilder.ChemistryLabResultLinkBuilder;
import com.agilex.healthcare.mobilehealthplatform.utils.uriformaters.linkbuilder.ChemistryLabTestLinkBuilder;
import com.agilex.healthcare.utility.NullChecker;

@Path("/patient/{assigning-authority}/{patient-id}/lab")
@Component
@Scope("request")
public class PatientLabsResource {
 
	@GET
	@Path("results/scope/{scope}")
	@Produces({ "application/xml", "application/json" })
	public LabResults getLabResultsByScope(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @QueryParam("loinc-code") String loinc, @PathParam("scope") String scope, @Context UriInfo uriInfo, @Context HttpHeaders headers) {

		PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		DateFilter dateFilter = DateFilterFactory.createFilterFromUri(uriInfo.getRequestUri());
		ScopeFilter scopeFilter = ScopeFilter.getInstanceForScope(scope);

		LabResults results = getLabResults(patientIdentifier, loinc, dateFilter, scopeFilter, uriInfo);
		return results;
	}

	@GET
	@Path("results/system/{system-id}")
	@Produces({ "application/xml", "application/json" })
	public LabResults getLabResultsBySystem(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @QueryParam("loinc-code") String loinc, @PathParam("system-id") String system, 
			@QueryParam("test") String test, @QueryParam("specimen") String specimen, @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		DateFilter dateFilter = DateFilterFactory.createFilterFromUri(uriInfo.getRequestUri());
		ScopeFilter scopeFilter = ScopeFilter.getInstanceForSystem(system);
		
		String groupingId = null;
		if(NullChecker.isNotNullish(test) && NullChecker.isNotNullish(specimen)) {
			groupingId = test + "-" + specimen;
		} else if(NullChecker.isNotNullish(loinc)){
			groupingId = loinc;
		}

		LabResults results = getLabResults(patientIdentifier, groupingId, dateFilter, scopeFilter, uriInfo);
		return results;
	}

	@GET
	@Path("resuts/scope/{scope}/loinc/{loinc-code}")
	@Produces({ "application/xml", "application/json" })
	public LabResults getLabResultsByScopeAndLoinc(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @PathParam("loinc-code") String loinc, @PathParam("scope") String scope, @Context UriInfo u, @Context HttpHeaders headers) {
		PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		DateFilter dateFilter = DateFilterFactory.createFilterFromUri(u.getRequestUri());
		ScopeFilter scopeFilter = ScopeFilter.getInstanceForScope(scope);

		LabResults results = getLabResults(patientIdentifier, loinc, dateFilter, scopeFilter, u);
		return results;
	}

	@GET
	@Path("results/system/{system-id}/loinc/{loinc-code}")
	@Produces({ "application/xml", "application/json" })
	public LabResults getLabResultsBySystemAndTest(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @PathParam("loinc-code") String loinc, @PathParam("system-id") String system, @Context UriInfo u, @Context HttpHeaders headers) {
		PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		DateFilter dateFilter = DateFilterFactory.createFilterFromUri(u.getRequestUri());
		ScopeFilter scopeFilter = ScopeFilter.getInstanceForSystem(system);

		LabResults results = getLabResults(patientIdentifier, loinc, dateFilter, scopeFilter, u);
		return results;
	}

	private LabResults getLabResults(PatientIdentifier patientIdentifier, String groupingId, DateFilter dateFilter, ScopeFilter scopeFilter, UriInfo u) {
		ChemistryLabDataService dataservice = new ChemistryLabDataService();
		LabResults results = dataservice.getLabResults(patientIdentifier, groupingId, dateFilter, scopeFilter);
		updateLabResultsLinks(patientIdentifier, results, u);
		return results;
	}

	@GET
	@Path("tests/scope/{scope}")
	@Produces({ "application/xml", "application/json" })
	public LabTests getLabTestsByScope(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @PathParam("scope") String scope, @Context UriInfo u, @Context HttpHeaders headers) {
		PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		DateFilter dateFilter = DateFilterFactory.createFilterFromUri(u.getRequestUri());
		ScopeFilter scopeFilter = ScopeFilter.getInstanceForScope(scope);

		LabTests results = getLabTests(patientIdentifier, dateFilter, scopeFilter, u);
		return results;
	}

	@GET
	@Path("tests/system/{system-id}")
	@Produces({ "application/xml", "application/json" })
	public LabTests getLabTestsBySystem(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @PathParam("system-id") String system, @Context UriInfo u, @Context HttpHeaders headers) {
		PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		DateFilter dateFilter = DateFilterFactory.createFilterFromUri(u.getRequestUri());
		ScopeFilter scopeFilter = ScopeFilter.getInstanceForSystem(system);

		LabTests results = getLabTests(patientIdentifier, dateFilter, scopeFilter, u);

		return results;
	}

	private LabTests getLabTests(PatientIdentifier patientIdentifier, DateFilter dateFilter, ScopeFilter scopeFilter, UriInfo u) {
		ChemistryLabDataService dataservice = new ChemistryLabDataService();
		LabTests labTests = dataservice.getLabTestsWithMostRecentResult(patientIdentifier, dateFilter, scopeFilter);
		
		ChemistryLabTestLinkBuilder linkBuilder = new ChemistryLabTestLinkBuilder(u.getBaseUri());
		linkBuilder.fillLinks(labTests, u.getRequestUri());

		return labTests;
	}

	@GET
	@Path("groups/scope/{scope}")
	@Produces({ "application/xml", "application/json" })
	public LabTestGroups getLabTestsInGroupsByScope(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @PathParam("scope") String scope, @Context UriInfo u, @Context HttpHeaders headers) {
		
		PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		DateFilter dateFilter = DateFilterFactory.createFilterFromUri(u.getRequestUri());
		ScopeFilter scopeFilter = ScopeFilter.getInstanceForScope(scope);

		LabTestGroups results = getLabTestsInGroups(patientIdentifier, dateFilter, scopeFilter, u);
		return results;
	}

	@GET
	@Path("groups/system/{system-id}")
	@Produces({ "application/xml", "application/json" })
	public LabTestGroups getLabTestsInGroupsBySystem(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @PathParam("system-id") String system, @Context UriInfo u, @Context HttpHeaders headers) {
		
		PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		DateFilter dateFilter = DateFilterFactory.createFilterFromUri(u.getRequestUri());
		ScopeFilter scopeFilter = ScopeFilter.getInstanceForSystem(system);

		LabTestGroups results = getLabTestsInGroups(patientIdentifier, dateFilter, scopeFilter, u);
		return results;
	}

	private LabTestGroups getLabTestsInGroups(PatientIdentifier patientIdentifier, DateFilter dateFilter, ScopeFilter scopeFilter, UriInfo uriInfo) {
		ChemistryLabDataService dataservice = new ChemistryLabDataService();
		LabTestGroups labTestGroups = dataservice.getLabTestsInGroups(patientIdentifier, dateFilter, scopeFilter);
		
		ChemistryLabTestLinkBuilder linkBuilder = new ChemistryLabTestLinkBuilder(uriInfo.getBaseUri());
		linkBuilder.fillLabTestGroupLinks(labTestGroups);
		
		return labTestGroups;
	}

	private void updateLabResultsLinks(PatientIdentifier patientIdentifier, LabResults results, UriInfo u) {
		ChemistryLabResultLinkBuilder linkbuilder = new ChemistryLabResultLinkBuilder(u.getBaseUri());
		linkbuilder.fillLinks(results, u.getRequestUri());
	}

}
