package com.agilex.healthcare.mobilehealthplatform.restservice;

import com.agilex.healthcare.mobilehealthplatform.datalayer.vital.PatientEnteredVitalDataService;
import com.agilex.healthcare.mobilehealthplatform.datalayer.vital.PatientEnteredVitalGraphGenerator;
import com.agilex.healthcare.mobilehealthplatform.datalayer.vital.VitalEntryFilter;
import com.agilex.healthcare.mobilehealthplatform.datalayer.vital.VitalsFilterFactory;
import com.agilex.healthcare.mobilehealthplatform.dataservice.PdfDataService;
import com.agilex.healthcare.mobilehealthplatform.domain.*;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilterFactory;
import com.agilex.healthcare.mobilehealthplatform.security.MhpUserFactory;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.ScopeFilter;
import com.agilex.healthcare.mobilehealthplatform.pdf.PatientEnteredVitalsMapper;
import com.agilex.healthcare.mobilehealthplatform.utils.QueryParameters;
import com.agilex.healthcare.mobilehealthplatform.utils.uriformaters.linkbuilder.PatientEnteredVitalsLinkBuilder;
import com.agilex.healthcare.mobilehealthplatform.utils.uriformaters.linkbuilder.VitalsLinkBuilder;
import com.agilex.healthcare.pdf.PdfGenerationContext;
import com.agilex.healthcare.utility.NullChecker;
import com.agilex.healthcare.utility.objectmapper.VitalTableDataGenerator;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.*;
import javax.ws.rs.core.*;
import java.io.IOException;
import java.io.OutputStream;
import java.util.List;

@Path("/patient/{assigning-authority}/{patient-id}/patient-entered-vitals")
@Component
@Scope("request")
public class PatientEnteredVitalsResource extends AbstractUserResource {
	
	private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(PatientEnteredVitalsResource.class);

	/**
	 * Retrieves Self/Patient Entered {@link com.agilex.healthcare.mobilehealthplatform.domain.VitalEntries}
	 *
	 * @param assigningAuthority
	 * @param patientId
	 * @return Patient Entered {@link com.agilex.healthcare.mobilehealthplatform.domain.VitalEntries}
	 */
	@GET
	@Produces({ "application/xml", "application/json" })
	public VitalEntries fetchVitalEntries(@PathParam("assigning-authority") String assigningAuthority,
                                          @PathParam("patient-id") String patientId,
                                          @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		logger.debug("Received GET request for a multiple Patient Entered VitalEntries");

		PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		VitalEntryFilter filter = VitalsFilterFactory.createFilter(uriInfo.getRequestUri());
		VitalEntries vitalEntries = fetchVitalEntries(patientIdentifier, filter);

		PatientEnteredVitalsLinkBuilder builder = new PatientEnteredVitalsLinkBuilder(uriInfo.getBaseUri());
		builder.fillLinks(vitalEntries, uriInfo.getRequestUri());

		return vitalEntries;
	}


    /**
     * Retrieves Self/Patient Entered {@link com.agilex.healthcare.mobilehealthplatform.domain.VitalEntries}
     *
     * @param assigningAuthority
     * @param patientId
     * @return a PDF document of {@link com.agilex.healthcare.mobilehealthplatform.domain.VitalEntries}
     */
    @GET
    @Path("pdf")
    @Produces("application/pdf")
    public Response fetchVitalEntriesAsPDF(@PathParam("assigning-authority") String assigningAuthority,
                              @PathParam("patient-id") String patientId,
                              @Context HttpServletRequest request,
                              @Context UriInfo uriInfo) {

        PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
        VitalEntryFilter filter = VitalsFilterFactory.createFilter(uriInfo.getRequestUri());
        VitalEntries vitalEntries = fetchVitalEntries(patientIdentifier, filter);

        PdfGenerationContext context = buildContext(uriInfo, vitalEntries);

        StreamingOutput stream = generatePdfToStreamingOutput(context);
        return Response.ok(stream).build();
    }

    private PdfGenerationContext buildContext(UriInfo uriInfo, VitalEntries vitalEntries) {
        PdfGenerationContext context = new PdfGenerationContext();
        context.setDateFilter(DateFilterFactory.createFilterFromUri(uriInfo.getRequestUri()));
        context.setPatient(MhpUserFactory.createFromSecurityContext().getPatient());
        String section = uriInfo.getQueryParameters().getFirst("section");
        context.setSection(section);

        List<List<String>> dataTable = null;
        if(section.equals("BP")) {
            dataTable = PatientEnteredVitalsMapper.getBloodPressureTable(vitalEntries);
        } else if(section.equals("weight")) {
            dataTable = PatientEnteredVitalsMapper.getWeightTable(vitalEntries);
        }
        context.setReportDataTable(dataTable);

        return context;
    }

    private StreamingOutput generatePdfToStreamingOutput(final PdfGenerationContext context) {
        StreamingOutput stream = new StreamingOutput() {
            public void write(OutputStream output) throws IOException, WebApplicationException {
                try {
                    PdfDataService pdfDataService = new PdfDataService();
                    pdfDataService.generateReport(output, context);
                } catch (Exception e) {
                    logger.error("Error generating PDF Report", e);
                    throw new WebApplicationException(Response.Status.INTERNAL_SERVER_ERROR);
                }
            }
        };
        return stream;
    }

	/**
	 * Provides graph data for Patient Entered {@link com.agilex.healthcare.mobilehealthplatform.domain.VitalEntries}
	 *
	 * @param assigningAuthority
	 * @param patientId
	 * @return
	 */
	@GET
	@Path("graphdata")
	@Produces({ "application/xml", "application/json" })
	public GraphData getVitalEntriesAsGraphData(@PathParam("assigning-authority") String assigningAuthority,
                                                @PathParam("patient-id") String patientId,
                                                @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		logger.debug("Received GET request for patient entered graph data");

		PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);

		VitalEntryFilter filter = VitalsFilterFactory.createFilter(uriInfo.getRequestUri());
		VitalEntries vitals = fetchVitalEntries(patientIdentifier, filter);

		PatientEnteredVitalGraphGenerator graphGenerator = new PatientEnteredVitalGraphGenerator();

		QueryParameters p = new QueryParameters(uriInfo.getRequestUri().getQuery());

		String graphType = p.getValue("graphType");
		if (NullChecker.isNullish(graphType))
			graphType = filter.getSection();

		GraphData graphData = graphGenerator.createGraphDataBySection(vitals, graphType);

		VitalsLinkBuilder linkbuilder = new VitalsLinkBuilder(uriInfo.getBaseUri());
		linkbuilder.fillLinks(graphData, uriInfo.getRequestUri());

		return graphData;
	}

	/**
	 * Provides tabular data for Patient Entered {@link com.agilex.healthcare.mobilehealthplatform.domain.VitalEntries}
	 *
	 * @param assigningAuthority
	 * @param patientId
	 * @return
	 */
	@GET
	@Path("tabledata")
	@Produces({ "application/xml", "application/json" })
	public VitalTableData getVitalEntriesAsTableData(@PathParam("assigning-authority") String assigningAuthority,
                                                     @PathParam("patient-id") String patientId,
                                                     @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		logger.debug("Received GET request for patient entered tablular data");

		PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);

		VitalEntryFilter filter = VitalsFilterFactory.createFilter(uriInfo.getRequestUri());
		VitalEntries vitals = fetchVitalEntries(patientIdentifier, filter);

		VitalTableDataGenerator tableGenerator = new VitalTableDataGenerator();
		VitalTableData vitalTableData = tableGenerator.generateVitalTableData(vitals);
		vitalTableData.sortDescending("entryTime");

		VitalsLinkBuilder linkbuilder = new VitalsLinkBuilder(uriInfo.getBaseUri());
		linkbuilder.fillLinks(vitalTableData, uriInfo.getRequestUri());

		return vitalTableData;
	}

	private VitalEntries fetchVitalEntries(PatientIdentifier patientIdentifier, VitalEntryFilter filter) {

		PatientEnteredVitalDataService dataService = new PatientEnteredVitalDataService();
		VitalEntries vitalEntries = dataService.fetchVitalEntries(patientIdentifier, filter, ScopeFilter.getInstanceForOperationalScope());

		return vitalEntries;
	}

	/**
	 * Retrieves a single {@link com.agilex.healthcare.mobilehealthplatform.domain.VitalEntry}.
	 *
	 * @param assigningAuthority
	 * @param patientId
	 * @param systemId
	 * @param vitalId
	 * @return
	 */
	@GET
	@Path("system/{system-id}/id/{vital-id}")
	@Produces({ "application/xml", "application/json" })
	public VitalEntry fetchVitalEntry(@PathParam("assigning-authority") String assigningAuthority,
                                      @PathParam("patient-id") String patientId,
                                      @PathParam("system-id") String systemId,
                                      @PathParam("vital-id") String vitalId,
                                      @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		logger.debug("Received GET request for a single Patient Entered VitalEntry");

		PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		DataIdentifier dataIdentifier = new DataIdentifier(systemId, vitalId);
		ScopeFilter scopeFilter = ScopeFilter.getInstanceForSystem(systemId);

		PatientEnteredVitalDataService dataService = new PatientEnteredVitalDataService();
		VitalEntry vitalEntry = dataService.fetchVitalEntry(patientIdentifier, dataIdentifier, scopeFilter);

		return vitalEntry;
	}

	/**
	 * Saves a new {@link com.agilex.healthcare.mobilehealthplatform.domain.VitalEntry}
	 *
	 * @param vitalEntry
	 * @param assigningAuthority
	 * @param patientId
	 * @return The saved {@link com.agilex.healthcare.mobilehealthplatform.domain.VitalEntry}
	 */
	@POST
	@Consumes({ "application/xml", "application/json" })
	@Produces({ "application/xml", "application/json" })
	public VitalEntry saveVital(VitalEntry vitalEntry, @PathParam("assigning-authority") String assigningAuthority,
                                @PathParam("patient-id") String patientId,
                                @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		logger.debug("Received POST request to save Patient Entered VitalEntry");

		return save(vitalEntry, assigningAuthority, patientId, uriInfo);
	}

	/**
	 * Updates the {@link com.agilex.healthcare.mobilehealthplatform.domain.VitalEntry} using it's self link
	 *
	 * @param vitalEntry
	 * @param assigningAuthority
	 * @param patientId
	 * @return
	 */
	@PUT
	@Path("system/{system-id}/id/{vital-id}")
	@Consumes({ "application/xml", "application/json" })
	@Produces({ "application/xml", "application/json" })
	public VitalEntry updateVital(VitalEntry vitalEntry, @PathParam("assigning-authority") String assigningAuthority,
                                  @PathParam("patient-id") String patientId,
                                  @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		logger.debug("Recieved PUT request to update a VitalEntry");

		return save(vitalEntry, assigningAuthority, patientId, uriInfo);
	}


	private VitalEntry save(VitalEntry vitalEntry, String assigningAuthority, String patientId, UriInfo uriInfo) {
		PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		vitalEntry.setPatientIdentifier(patientIdentifier);

		PatientEnteredVitalDataService dataService = new PatientEnteredVitalDataService();
		VitalEntry saved = dataService.save(vitalEntry);

		PatientEnteredVitalsLinkBuilder builder = new PatientEnteredVitalsLinkBuilder(uriInfo.getBaseUri());
		builder.fillLinks(saved, uriInfo.getRequestUri());

		return saved;
	}

	/**
	 * Deletes the {@link com.agilex.healthcare.mobilehealthplatform.domain.VitalEntry}
	 * 
	 * @param assigningAuthority
	 * @param patientId
	 * @param vitalId
	 */
	@DELETE
	@Path("system/{system-id}/id/{vital-id}")
	public void deleteVitalEntry(@PathParam("assigning-authority") String assigningAuthority,
                                 @PathParam("patient-id") String patientId, @PathParam("vital-id") String vitalId,
                                 @Context UriInfo uriInfo, @Context HttpHeaders headers) {
		logger.debug("Received DELETE request for VitalEntry");

		VitalEntry vitalEntry = new VitalEntry();

		PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
		vitalEntry.setPatientIdentifier(patientIdentifier);

		DataIdentifier vitalIdentifier = new DataIdentifier(null, vitalId);
		vitalEntry.setDataIdentifier(vitalIdentifier);

		PatientEnteredVitalDataService dataService = new PatientEnteredVitalDataService();
		dataService.delete(vitalEntry);
	}
}
